/**
 * 
 */
package alma.acsplugins.alarmsystem.gui;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.FlowLayout;

import javax.swing.BoxLayout;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;

import cern.laser.client.data.Alarm;
import cern.laser.client.services.selection.AlarmSelectionListener;

import alma.acs.container.ContainerServices;
import alma.acsplugins.alarmsystem.gui.detail.AlarmDetailTable;
import alma.acsplugins.alarmsystem.gui.sound.AlarmSound;
import alma.acsplugins.alarmsystem.gui.statusline.StatusLine;
import alma.acsplugins.alarmsystem.gui.table.AlarmTable;
import alma.acsplugins.alarmsystem.gui.table.AlarmTableModel;
import alma.acsplugins.alarmsystem.gui.toolbar.Toolbar;
import alma.alarmsystem.clients.CategoryClient;
import alma.maciErrType.wrappers.AcsJCannotGetComponentEx;

/**
 * The panel shown while the CERN AS is in use and the 
 * alarm client is connected.
 * <P>
 * @author acaproni
 *
 */
public class CernSysPanel extends JPanel {
	
	/**
     * The toolbar
     */
    private Toolbar toolbar;
    
    /**
     * The status line
     */
    private StatusLine statusLine;
	
	/**
	 * The startup option for reduction rules
	 */
	public final boolean ACTIVATE_RDUCTION_RULES=true;
	
	/**
	 * The model of the table of alarms
	 */
	private AlarmTableModel model;
	
	/**
	 * The table of alarms
	 */
	private AlarmTable alarmTable;
	
	/**
	 * The table with the details of an alarm
	 */
	private AlarmDetailTable detailTable;
	
	/**
	 * The object to play sounds for alarms
	 */
	private AlarmSound alarmSound;
	
	/**
	 * The scroll pane of the table
	 */
	private JScrollPane tableScrollPane = new JScrollPane(
			JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED,
			JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
	
	/**
	 * The scroll pane of the details table
	 */
	private JScrollPane detailsScrollPane = new JScrollPane(
			JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED,
			JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);

	/**
     * The split pane dividing the table of alarms and the detail view
     */
    private JSplitPane splitPane;
    
    /**
     * The panel showing this container
     */
    private final AlarmPanel alarmPanel;
    
    /**
     * Say if there is an attempt to connect
     */
    private volatile boolean connecting=false;
    
    /**
     * <code>true</code> if the panel has been closed.
     * It helps stopping the connection thread
     */
    private volatile boolean closed=false;
    
    /**
     * The listener of the connection
     */
    private ConnectionListener connectionListener;
    
    /**
     * The thread to connect/disconnect
     */
    private Thread connectThread;
    
    /**
     * Signal the thread to terminate
     */
    private Thread disconnectThread;
    
    /**
     *  The client to listen alarms from categories
     */
    private CategoryClient categoryClient=null;
    
    /**
	 * The container services
	 */
    private ContainerServices contSvc=null;
    
    /**
     * The panel to show messages while connecting
     */
    private final AlSysNotAvailPanel notAvaiPnl;
    
    /**
     * Constructor
     * 
     * @param owner The panel showing this container
     */
    public CernSysPanel(AlarmPanel owner, AlSysNotAvailPanel notAvaiPnl) {
    	if (notAvaiPnl==null) {
    		throw new IllegalArgumentException("AlSysNotAvailPanel can't be null");
    	}
    	alarmPanel=owner;
    	this.notAvaiPnl=notAvaiPnl;
    	initialize();
    }
    
	/**
	 * Init the GUI
	 *
	 */
	private void initialize() {
		setLayout(new BorderLayout());
		
		// Build GUI objects
		model = new AlarmTableModel(this,ACTIVATE_RDUCTION_RULES,false);
		alarmSound= new AlarmSound(model);
		alarmTable = new AlarmTable(model,this);
		statusLine = new StatusLine(model,this);
		connectionListener=statusLine;
		model.setConnectionListener(statusLine);
		detailTable = new AlarmDetailTable();
		
		// The table of alarms
		tableScrollPane.setViewportView(alarmTable);
		Dimension minimumSize = new Dimension(300, 150);
		tableScrollPane.setMinimumSize(minimumSize);
		tableScrollPane.setPreferredSize(minimumSize);
		
		// The details table
		detailsScrollPane.setViewportView(detailTable);
		
		// The panel with the details
		JPanel detailsPanel = new JPanel();
		BoxLayout layout = new BoxLayout(detailsPanel,BoxLayout.Y_AXIS);
		detailsPanel.setLayout(new BorderLayout());
		
		JPanel lblPnl = new JPanel(new FlowLayout(FlowLayout.CENTER));
		lblPnl.add(new JLabel("Alarm details"));
		detailsPanel.add(lblPnl,BorderLayout.PAGE_START);
		detailsPanel.add(detailsScrollPane,BorderLayout.CENTER);
		minimumSize = new Dimension(120, 150);
		detailsPanel.setMinimumSize(minimumSize);
		detailsPanel.setPreferredSize(minimumSize);
		
		splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT,tableScrollPane,detailsPanel);
		splitPane.setOneTouchExpandable(true);
		splitPane.setResizeWeight(1);
		//splitPane.setDividerLocation(tableScrollPane.getMinimumSize().width);
		add(splitPane,BorderLayout.CENTER);
		
		// Add the toolbar
		toolbar=new Toolbar(alarmTable,model,alarmSound,ACTIVATE_RDUCTION_RULES,this);
		add(toolbar,BorderLayout.NORTH);
		
		// Add the status line
		add(statusLine,BorderLayout.SOUTH);
	}
	
	/**
	 * Closes the panel
	 */
	public void close() {
		alarmSound.close();
		model.close();
		alarmTable.close();
	}
	
	/**
	 * @see IpauseResume
	 */
	public void pause() throws Exception {
		model.pause(true);
		statusLine.pause();
		toolbar.updatePauseBtn(true);
	}
	
	/**
	 * @see IPauseResume
	 */
	public void resume() throws Exception {
		model.pause(false);
		statusLine.resume();
		toolbar.updatePauseBtn(false);
	}
	
	/**
	 * Show a message in the status line
	 * 
	 * @param mesg
	 * @param red
	 * 
	 * @see StatusLine
	 */
	public void showMessage(String mesg, boolean red) {
		statusLine.showMessage(mesg, red);
	}
	
	/**
	 * Show the alarm in the details table
	 *  
	 * @param alarm The alarm to show in the details panel;
	 * 				if <code>null</code> the details table is cleared.
	 */
	public void showAlarmDetails(Alarm alarm) {
		detailTable.showAlarmDetails(alarm);
	}
	
	/**
	 * A method to send alarms to the GUI outside of the alarm service.
	 * <P>
	 * At the present it is used by the OMC GUI to send alarms before the alarm
	 * service is started.
	 * 
	 * @deprecated this method will be deleted when the alarm system will run as a daemon 
	 * 				or as an ACS service.
	 * 
	 * @param alarm The alarm to show in the table (can't be <code>null</code>)
	 * @throws Exception In case the alarm is not well formed
	 * 
	 * 
	 */
	public synchronized void addSpecialAlarm(Alarm alarm) throws Exception {
		if (alarm==null || alarm.getAlarmId()==null || alarm.getAlarmId().isEmpty()) {
			throw new Exception("The alarm cant'be null and must have a valid ID!");
		}
		model.onAlarm(alarm);
	}

	public void setModel(AlarmTableModel model) {
		this.model = model;
	}
	
	/**
	 * Connect
	 */
	public void connect() {
		if (connecting || closed) {
			return;
		}
		connecting=true;
		connectionListener.connecting();
		notAvaiPnl.addMessage("Connecting to the alarm service");
		notAvaiPnl.addMessage("Instantiating the category client");
		try {
			categoryClient = new CategoryClient(contSvc);
		} catch (Throwable t) {
			System.err.println("Error instantiating the CategoryClient: "+t.getMessage());
			notAvaiPnl.addMessage("Error instantiating the CategoryClient: "+t.getMessage());
			t.printStackTrace(System.err);
			connectionListener.disconnected();
			categoryClient=null;
			connecting=false;
			return;
		}
		/**
		 * Try to connect to the alarm service until it becomes available
		 */
		while (true && !closed) {
			notAvaiPnl.addMessage("Connecting to the categories");
			try {
				categoryClient.connect((AlarmSelectionListener)model);
				notAvaiPnl.addMessage("CategoryClient connected");
				// If the connection succeeded then exit the loop
				break;
			} catch (AcsJCannotGetComponentEx cgc) {
				// Wait 30 secs before retrying
				// but checks if it is closed every second.
				int t=0;
				while (t<30) {
					if (closed) {
						return;
					}
					try {
						Thread.sleep(1000);
					} catch (Exception e) {}
					t++;
				}
				cgc.printStackTrace();
				
				continue; // Try again
			} catch (Throwable t) {
				System.err.println("Error connecting CategoryClient: "+t.getMessage()+", "+t.getClass().getName());
				notAvaiPnl.addMessage("Error connecting CategoryClient: "+t.getMessage()+", "+t.getClass().getName());
				t.printStackTrace(System.err);
				connectionListener.disconnected();
				connecting=false;
				return;
			}
		}
		if (closed) {
			model.setCategoryClient(null);
			return;
		}
		notAvaiPnl.addMessage("Connected to the alarm service");
		connecting=false;
		connectionListener.connected();
		statusLine.start();
		model.setCategoryClient(categoryClient);
		alarmPanel.showPanel(AlarmPanel.cernSysName);
	}
	
	/**
	 * Disconnect
	 */
	public void disconnect() {
		statusLine.stop();
		model.setCategoryClient(null);
		// wait until the connect thread terminates (if it is running)
		while (connectThread!=null && connectThread.isAlive()) {
			try {
				Thread.sleep(1500);
			} catch (Exception e) {}
		}
		try {
			categoryClient.close();
		} catch (Throwable t) {
			System.err.println("Error closinging CategoryClient: "+t.getMessage());
			t.printStackTrace(System.err);
		} finally {
			categoryClient=null;
			connectionListener.disconnected();
		}
	}
	
	/**
	 * @return <code>true</code> if an attempt to connect is running
	 */
	public boolean isConnecting() {
		return connecting;
	}
	
	/**
	 * Set the ContainerServices
	 */
	public void setContainerServices(ContainerServices cs) {
		if (cs==null) {
			throw new IllegalArgumentException("Invalid null ContainerServices");
		}
		contSvc=cs;	
	}
	
	/**
	 * Connect the Client and listens to the categories.
	 * 
	 * The <code>CategoryClient</code> is built only if its reference is null.
	 * Otherwise it means that the client is still trying to connect and the user 
	 * restarted the plugin.
	 *  
	 * @see SubsystemPlugin
	 */
	public void start() throws Exception {
		// Check if the CS have been set
		if (contSvc==null) {
			throw new Exception("ContainerServices not set");
		}
		class StartAlarmPanel extends Thread {
			public void run() {
				CernSysPanel.this.connect();
			}
		}
		closed=false;
		// Connect the categoryClient only if it is null
		if (categoryClient==null) {
			connectThread = new StartAlarmPanel();
			connectThread.setName("StartAlarmPanel");
			connectThread.setDaemon(true);
			connectThread.start();
		}
	}
	
	/**
	 * @see SubsystemPlugin
	 */
	public void stop() throws Exception {
		class StopAlarmPanel extends Thread {
			public void run() {
				try {
					CernSysPanel.this.disconnect();
				} catch (Throwable t) {
					System.err.println("Ignored error while disconnecting category client: "+t.getMessage());
					t.printStackTrace(System.err);
				}
			}
		}
		closed=true;
		disconnectThread = new StopAlarmPanel();
		disconnectThread.setName("StopAlarmPanel");
		disconnectThread.start();
		close();
	}
}
