package CDT6WorkspaceCreator;

# ************************************************************
# Description   : Eclipse CDT 6 generator
# Author        : Chris Cleeland, Object Computing, Inc.
# Create Date   : 23-Apr-2010
# ************************************************************

# ************************************************************
# Pragmas
# ************************************************************

use strict;

use CDT6ProjectCreator;
use WorkspaceCreator;

use vars qw(@ISA);
@ISA = qw(WorkspaceCreator);

# ************************************************************
# Subroutine Section
# ************************************************************

sub requires_make_coexistence {
  #my $self = shift;
  return 1;
}

sub supports_make_coexistence {
  #my $self = shift;
  return 1;
}

sub workspace_file_name {
  my $self = shift;
  my $wsn = $self->get_workspace_name();
  return $self->get_modified_workspace_name("cdt_workspace_$wsn", '.txt');
}

sub pre_workspace {
  my($self, $fh) = @_;
  my $crlf = $self->crlf();

  ## Optionally print the workspace comment
  $self->print_workspace_comment($fh,
    '#----------------------------------------------------------------------------', $crlf,
    '#       Eclipse CDT 6 generator', $crlf,
    '#', $crlf,
    '# This file was generated by MPC.  Any changes made directly to', $crlf,
    '# this file will be lost the next time it is generated.', $crlf,
    '# Listed below are the Eclipse projects created for this MPC workspace.', $crlf,
    '# These should be imported into an existing Eclipse workspace.', $crlf,
    '# Each project is listed on a line consisting of the project name and the full', $crlf,
    '# path to its .project file (space-separated).', $crlf,
    '# The projects are listed in dependency order.', $crlf,
    '#', $crlf,
    '# MPC Command:', $crlf,
    "# $0 @ARGV", $crlf,
    '#----------------------------------------------------------------------------', $crlf);
}

sub write_comps {
  my($self, $fh, $creator) = @_;
  my $info = $self->get_project_info();
  my $crlf = $self->crlf();
  $self->{'seen_deps'} = {};

  my $prefix = ($self->{'into'} ne '') ? $self->{'into'} . '/' : '';
  foreach my $project ($self->sort_dependencies($self->get_projects(), 0)) {
    print $fh "$$info{$project}->[0] ",
      Cwd::abs_path($prefix . $self->mpc_dirname($project)), '/.project', $crlf;
    $self->add_dependencies($creator, $project);
  }
}

sub add_dependencies {
  my($self, $creator, $proj) = @_;
  my $outdir = $self->mpc_dirname($proj);

  my $pre     = '    <project>';
  my $post    = '</project>';
  my $outfile = $outdir . '/.project';

  my $fh = new FileHandle();
  if (open($fh, $outfile)) {
    ## Get the dependencies and store them based on the directory of
    ## the project file.  We will check them later.
    my $deps = $self->get_validated_ordering($proj);
    my $key = $self->mpc_basename($self->mpc_dirname($proj));
    $self->{'seen_deps'}->{$key} = {};
    foreach my $dep (@$deps) {
      $self->{'seen_deps'}->{$key}->{$dep} = 1;
    }

    my @read = ();
    my $cwd  = $self->getcwd();
    while(<$fh>) {
      ## This is a comment found in cdt6project.mpd.
      if (/MPC\s+ADD\s+DEPENDENCIES/) {
        my $crlf = $self->crlf();
        my %seen = ();
        my @lines;
        foreach my $dep (reverse @$deps) {
          ## If we've seen this dependency, we don't need to add it
          ## again.  The build tool will handle it correctly.
          if (!$seen{$dep}) {
            my $relative = $self->get_relative_dep_file($creator,
                                                        "$cwd/$proj", $dep);
            ## Since we're looking at the dependencies in reverse order
            ## now, we need to unshift them into another array to keep
            ## the correct order.
            unshift(@lines, "$pre$dep$post$crlf") if (defined $relative);

            ## We've now seen this dependency and all of the
            ## projects upon which this one depends.
            $seen{$dep} = 1;
            foreach my $key (keys %{$self->{'seen_deps'}->{$dep}}) {
              $seen{$key} = 1;
            }
          }
        }

        ## Add the dependency lines to the project file
        push(@read, @lines);
      }
      else {
        push(@read, $_);
      }
    }
    close($fh);

    ## We will always rewrite the project file (with or without dependencies)
    if (open($fh, ">$outfile")) {
      foreach my $line (@read) {
        print $fh $line;
      }
      close($fh);
    }
  }
}


1;
