
/* This file is part of xtmodcoll-3.0, (c) xtmod.com
 * 
 * xtmodcoll-3.0 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * xtmodcoll-3.0 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser Public License
 * along with xtmodcoll-3.0.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.xtmod.util.collections;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;

import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeNode;



/**
 * This is a utility class to compare two tree models (ignoring order of elements).
 * It recursively works through the tree models, running {@link ListDiffEx} on all
 * nodes. The result of the tree diff is a bunch of {@link ListDiffEx}s.
 * 
 */

@SuppressWarnings("unchecked")
public abstract class TreeDiff<C extends Comparable> {


	protected List<KidListDiff> listdiffs = new ArrayList<KidListDiff>();


	// Input
	// =====================================================================


	/**
	 * Sets both trees and diffs them.
	 * 
	 * @param exist - the first tree
	 * @param incom - the second tree
	 */
	public void diff (DefaultTreeModel exist, DefaultTreeModel incom) {
		listdiffs.clear();
		recursiveDiff((TreeNode) exist.getRoot(), (TreeNode) incom.getRoot());
	}



	// Computations
	// =====================================================================

	/**
	 * Convert given tree node into a Comparable of type {@code C}
	 */
	abstract protected C identifyExisting (TreeNode x);

	/**
	 * Convert given tree node into a Comparable of type {@code C}
	 */
	abstract protected C identifyIncoming (TreeNode x);

	/**
	 * Check an already existing tree node for internal change.
	 */
	abstract protected boolean isUpdate (TreeNode exist, TreeNode incom);



	/**
	 * Recursively works through the tree models, running a diff on the
	 * kids-list of each node (except added and removed ones).
	 */
	protected void recursiveDiff (TreeNode existParent, TreeNode incomParent) {

		KidListDiff listdiffer = new KidListDiff(existParent);

		List<TreeNode> existChildren = Collections.list( (Enumeration<TreeNode>) existParent.children() );
		List<TreeNode> incomChildren = Collections.list( (Enumeration<TreeNode>) incomParent.children() );
		listdiffer.diff(existChildren, incomChildren);

		// store results
		// -------------------------------------------------
		if (!listdiffer.areEqual()) {
			listdiffs.add(listdiffer);
		}

		// recurse on for all nodes except removed and added
		// -------------------------------------------------
		List<TreeNode> rec1 = new ArrayList<TreeNode>();
		rec1.addAll(listdiffer.getUnchanged());
		rec1.addAll(listdiffer.getUpdated());

		List<TreeNode> rec2 = new ArrayList<TreeNode>();
		rec2.addAll(listdiffer.getDuplicates());
		rec2.addAll(listdiffer.getUpdates());

		int n = rec1.size(); // both are same size
		for (int i = 0; i < n; i++) {
			recursiveDiff(rec1.get(i), rec2.get(i));
		}
	}


	/**
	 * This is a simple extension of {@link ListDiff}, storing the children diff of one
	 * particular node.
	 */
	public class KidListDiff extends ListDiff<TreeNode, C, TreeNode> {

		@Override
		protected C identifyExisting (TreeNode x) {
			return TreeDiff.this.identifyExisting(x);
		}

		@Override
		protected C identifyIncoming (TreeNode x) {
			return TreeDiff.this.identifyIncoming(x);
		}

		@Override
		protected boolean isUpdate (TreeNode e, TreeNode i) {
			return TreeDiff.this.isUpdate(e, i);
		}


		protected TreeNode parent;

		protected KidListDiff (TreeNode parent) {
			this.parent = parent;
		}

		@Override
		public String toString () {
			return "Differences found under " + parent + "\n" + super.toString();
		}

	}


	// Output
	// =====================================================================


	/**
	 * Returns the results of the tree diff. This is a bunch of list diffs.
	 */
	public List<? super KidListDiff> getDiffs () {
		return listdiffs;
	}

	/**
	 * Returns whether existing tree and incoming tree are equal.
	 */
	public boolean areEqual () {
		return listdiffs.isEmpty();
	}


	@Override
	public String toString () {
		StringBuilder s = new StringBuilder(getClass().getSimpleName());
		for (int i = 0; i < listdiffs.size(); i++)
			s.append(listdiffs.get(i));
		return s.toString();
	}

}
