/*
* Copyright (C) The Community OpenORB Project. All rights reserved.
*
* This software is published under the terms of The OpenORB Community Software
* License version 1.0, a copy of which has been included with this distribution
* in the LICENSE.txt file.
*/
package org.openorb.util;

/**
 * This class provides helper methods for repositiory id
 * handling.
 *
 * @author Chris Wood
 * @version $Revision: 1.8 $ $Date: 2002/07/14 19:12:40 $ 
 */
public final class RepoIDHelper
{
    /** A plain class type (ITF.java). */
    public static final int TYPE_PLAIN = 0;
    /** A stub class type (ITFStub.java). */
    public static final int TYPE_STUB = 1;
    /** A helper class type (ITFHelper.java). */
    public static final int TYPE_HELPER = 2;
    /** A holder class type (ITFHolder.java). */
    public static final int TYPE_HOLDER = 3;
    /** A default factory type (???). */
    public static final int TYPE_DEFAULT_FACTORY = 4;
    /** A operations class type (ITFOperations.java). */
    public static final int TYPE_OPERATIONS = 255;
    /** An implbase class type (ITFImplBase.java (BOA)). */
    public static final int TYPE_IMPLBASE = 256;
    /** A POA class type (ITFPOA.java). */
    public static final int TYPE_POA = 257;
    /** A POA tie class type (ITFPOATie.java). */
    public static final int TYPE_POATIE = 258;
    /** A tie class type (_ITFTie.java). */
    public static final int TYPE_TIE = 259;

    /**
     * This methods decorates a class name with the suffixes
     * depending on the type of the class about to be created.
     *
     * @param name The class name to be decorated.
     * @param type The type of the class about to be created.
     * @return The decorated name.
     */
    public static String decorate( String name, int type )
    {
        StringBuffer ret = new StringBuffer();

        switch ( type )
        {

        case TYPE_STUB:

        case TYPE_IMPLBASE:
            ret.append( '_' );
        }

        ret.append( name );

        switch ( type )
        {

        case TYPE_PLAIN:
            break;

        case TYPE_STUB:
            ret.append( "Stub" );
            break;

        case TYPE_HELPER:
            ret.append( "Helper" );
            break;

        case TYPE_HOLDER:
            ret.append( "Holder" );
            break;

        case TYPE_DEFAULT_FACTORY:
            ret.append( "DefaultFactory" );
            break;

        case TYPE_OPERATIONS:
            ret.append( "Operations" );
            break;

        case TYPE_IMPLBASE:
            ret.append( "ImplBase" );
            break;

        case TYPE_POA:
            ret.append( "POA" );
            break;

        case TYPE_POATIE:
            ret.append( "POATie" );
            break;

        case TYPE_TIE:
            ret.append( "Tie" );
            break;

        default:
            throw new IllegalArgumentException();
        }

        return ret.toString();
    }

    /**
     * This methods checks whether the passed repository identifier is valid
     * or not.
     *
     * @param id The repository id to check.
     * @return True if id is a valid repository identifier, false otherwise.
     */
    public static boolean checkID( String id )
    {
        if ( id.startsWith( "IDL:" ) )
        {
            int dot = id.indexOf( '.' );
            int lcl = id.lastIndexOf( ':' );
            int fsla = id.indexOf( '/' );
            int lsla = id.lastIndexOf( '/', lcl );
            int nsla;

            if ( dot < lcl )
            {
                int prev = fsla;

                while ( ( dot = id.lastIndexOf( '.', prev ) ) >= 0 )
                {
                    if ( !checkIdentifier( id.substring( dot + 1, prev ) ) )
                        return false;

                    prev = dot - 1;
                }

                if ( !checkIdentifier( id.substring( 4, prev + 1 ) ) )
                    return false;

                while ( fsla != lsla )
                {
                    nsla = id.indexOf( '/', fsla + 1 );

                    if ( !checkIdentifier( id.substring( fsla + 1, nsla ) ) )
                        return false;

                    fsla = nsla;
                }
            }
            else if ( lsla >= 0 )
            {
                fsla = 3;

                while ( fsla != lsla )
                {
                    nsla = id.indexOf( '/', fsla + 1 );

                    if ( !checkIdentifier( id.substring( fsla + 1, nsla ) ) )
                        return false;

                    fsla = nsla;
                }
            }
            else
                lsla = 3;

            if ( !checkIdentifier( id.substring( lsla + 1, lcl ) ) )
                return false;

            return true;
        }
        else if ( id.indexOf( ':' ) > 0 )
            return true;

        return false;
    }

    /**
     * This methods checks an indetifier for validity in the
     * target language. The identifier must match the following pattern
     * <code>([a-z][A-Z])+([a-z][A-Z_][0-9])*</code>.
     *
     * @param name The identifier to check.
     * @return True if ??? false otherwise.
     */
    public static boolean checkIdentifier( String name )
    {
        if ( name.length() == 0 )
            return false;

        char c = name.charAt( 0 );

        if ( !( ( c >= 'a' && c <= 'z' ) || ( c >= 'A' && c <= 'Z' ) ) )
            return false;

        for ( int i = 1; i < name.length(); ++i )
        {
            c = name.charAt( i );

            if ( ( c >= 'a' && c <= 'z' ) || ( c >= 'A' && c <= 'Z' )
                    || c == '0' || ( c >= '1' && c <= '9' ) || c == '_' )
                continue;

            return false;
        }

        return true;
    }


    /**
     * Extract the Java package name from the IDL repository identifier.
     *
     * @param name The repository id to extract the package name from.
     * @return The package information converted from the repository id.
     */
    public static String idToPackage( String name )
    {
        if ( name.startsWith( "IDL:" ) )
        {
            StringBuffer ret = new StringBuffer();
            int dot = name.indexOf( '.' );
            int lcl = name.lastIndexOf( ':' );
            int fsla = name.indexOf( '/' );
            int lsla = name.lastIndexOf( '/', lcl );

            if ( dot < lcl )
            {
                int prev = fsla;

                while ( ( dot = name.lastIndexOf( '.', prev ) ) >= 0 )
                {
                    ret.append( name.substring( dot + 1, prev ) ).append( '.' );
                    prev = dot - 1;
                }

                ret.append( name.substring( 4, prev + 1 ) );

                if ( fsla != lsla )
                    ret.append( '.' ).append(
                        name.substring( fsla + 1, lsla ).replace( '/', '.' ) );
            }
            else if ( lsla >= 0 )
                ret.append( name.substring( 4, lsla ).replace( '/', '.' ) );

            return ret.toString();
        }
        else if ( name.startsWith( "RMI:" ) )
        {
            // TODO: fillin stuff here to extract classname
            // from RMI repositoy ID
            throw new IllegalArgumentException();
        }

        throw new IllegalArgumentException();
    }

    /**
     * Convert the repository id to a class name.
     *
     * @param name The repository id to extract the class name from.
     * @return The name of the class extracted from the repository id.
     */
    public static String idToClassname( String name )
    {
        if ( name.startsWith( "IDL:" ) )
        {
            int end = name.lastIndexOf( ':' );
            int beg;

            if ( ( beg = name.lastIndexOf( '/', end ) ) == -1 )
                beg = 3;

            return name.substring( beg + 1, end );
        }
        else if ( name.startsWith( "RMI:" ) )
        {
            // TODO: fillin stuff here to extract classname
            // from RMI repositoy ID
            throw new IllegalArgumentException();
        }

        throw new IllegalArgumentException();
    }

    /**
     * This method tries to extract the class name from the repository id.
     *
     * @param name The repository id name.
     * @return The Java class name for the repository id.
     */
    public static String idToClass( String name )
    {
        return idToClass( name, 0 );
    }

    /**
     * This method tries to extract the class name from the repository id.
     *
     * @param name The repository id name.
     * @param type The type of the class to create.
     * @return The Java class name for the repository id.
     */
    public static String idToClass( String name, int type )
    {
        if ( name.startsWith( "IDL:" ) )
        {
            StringBuffer ret = new StringBuffer();
            int dot = name.indexOf( '.' );
            int lcl = name.lastIndexOf( ':' );
            int fsla = name.indexOf( '/' );
            int lsla = name.lastIndexOf( '/', lcl );

            if ( dot < lcl )
            {
                int prev = fsla - 1;

                while ( ( dot = name.lastIndexOf( '.', prev ) ) >= 0 )
                {
                    ret.append( name.substring( 
                            dot + 1, prev + 1 ) ).append( '.' );
                    prev = dot - 1;
                }

                ret.append( name.substring( 4, prev + 1 ) ).append( '.' );

                if ( fsla != lsla )
                    ret.append( name.substring( fsla + 1, lsla ).replace( 
                        '/', '.' ) ).append( '.' );
            }
            else if ( lsla >= 0 )
                ret.append( name.substring( 4, lsla ).replace( 
                    '/', '.' ) ).append( '.' );
            else
                lsla = 3;

            ret.append( decorate( name.substring( lsla + 1, lcl ), type ) );

            return ret.toString();
        }
        else if ( name.startsWith( "RMI:" ) )
        {
            // TODO: fillin stuff here to extract classname from RMI
            // repositoy ID return empty string for types not supported
            //  (i.e. operations interface)
            throw new IllegalArgumentException();
        }

        throw new IllegalArgumentException();
    }

    /**
     * This returns an object which can be used to test if some other 
     * repository ID is an instance of this one. <p>Examples:<p/>
     * <pre>
     * RepoIDHelper.createIsATest( "IDL:MyObj:1.0" ).equals( 
     *         "IDL:MyObj:1.0" ) == true
     * RepoIDHelper.createIsATest( "IDL:MyObj:1.1" ).equals(
     *         "IDL:MyObj:1.0" ) == true
     * RepoIDHelper.createIsATest( "IDL:MyObj:1.0" ).equals(
     *         "IDL:MyObj:1.1" ) == false
     * RepoIDHelper.createIsATest( "IDL:MyObj:2.0" ).equals(
     *         "IDL:MyObj:1.1" ) == false
     * </pre>
     *
     * @param id The repository id to test.
     * @return The test object or the id if the parsing
     * failed.
     */
    public static Object createIsATest( String id )
    {
        try
        {
            if ( id.startsWith( "IDL:" ) )
                return new IsA( id );
        }
        catch ( NumberFormatException ex )
        {
            // we return the id that was passed
        }

        return id;
    }

    private static class IsA
    {
        private String m_me;
        private String m_nomi;
        private int m_minor;

        public IsA( String me )
             throws NumberFormatException
        {
            m_me = me;
            int idx = m_me.lastIndexOf( '.' );
            m_nomi = m_me.substring( 0, idx + 1 );
            m_minor = Integer.parseInt( m_me.substring( idx + 1 ) );
        }

        public String toString()
        {
            return m_me;
        }

        public boolean equals( Object obj )
        {
            if ( obj == null )
                return false;

            if ( this == obj )
                return true;

            String str = obj.toString();

            if ( str.equals( m_me ) )
                return true;

            if ( !str.startsWith( m_nomi ) )
                return false;

            int minor;

            try
            {
                minor = Integer.parseInt( str.substring( m_nomi.length() ) );
            }
            catch ( NumberFormatException ex )
            {
                return false;
            }

            return m_minor <= minor;
        }
    }
}

