/*
* Copyright (C) The Community OpenORB Project. All rights reserved.
*
* This software is published under the terms of The OpenORB Community Software
* License version 1.0, a copy of which has been included with this distribution
* in the LICENSE.txt file.
*/
package org.openorb.util;

/**
 * A utility class for obtaining <code>Number</code> instances.
 * This class caches the most commonly used values.
 *
 * @author Richard G Clark
 */
public final class NumberCache 
{
    private static final Byte[] BYTE_CACHE;
    
    private static final Short MIN_SHORT = new Short(Short.MIN_VALUE);
    private static final Short MAX_SHORT = new Short(Short.MAX_VALUE);
    private static final Short[] SHORT_BLOCK;
    private static final short MIN_SHORT_BLOCK = -2;
    private static final short MAX_SHORT_BLOCK = 100;
    
    private static final Integer MIN_INTEGER = new Integer(Integer.MIN_VALUE);
    private static final Integer MAX_INTEGER = new Integer(Integer.MAX_VALUE);
    private static final Integer[] INTEGER_BLOCK;
    private static final int MIN_INTEGER_BLOCK = -2;
    private static final int MAX_INTEGER_BLOCK = 100;

    private static final Long MIN_LONG = new Long(Long.MIN_VALUE);
    private static final Long MAX_LONG = new Long(Long.MAX_VALUE);
    private static final Long[] LONG_BLOCK;
    private static final long MIN_LONG_BLOCK = -2;
    private static final long MAX_LONG_BLOCK = 100;

    private static final Float ZERO_FLOAT = new Float(0.0f);
    private static final Float ONE_FLOAT = new Float(1.0f);
    private static final Float NEG_ONE_FLOAT = new Float(-1.0f);

    private static final Double ZERO_DOUBLE = new Double(0.0);
    private static final Double ONE_DOUBLE = new Double(1.0);
    private static final Double NEG_ONE_DOUBLE = new Double(-1.0);

    static 
    {
        BYTE_CACHE = new Byte[Byte.MAX_VALUE - Byte.MIN_VALUE + 1];
        for (byte i = Byte.MIN_VALUE; i < Byte.MAX_VALUE; i++) 
        {
            BYTE_CACHE[i - Byte.MIN_VALUE] = new Byte(i);
        }
        BYTE_CACHE[Byte.MAX_VALUE - Byte.MIN_VALUE] = new Byte(Byte.MAX_VALUE);
        
        SHORT_BLOCK = new Short[MAX_SHORT_BLOCK - MIN_SHORT_BLOCK + 1];
        for (short i = MIN_SHORT_BLOCK; i <= MAX_SHORT_BLOCK; i++)
        {
            SHORT_BLOCK[i - MIN_SHORT_BLOCK] = new Short(i);
        }

        INTEGER_BLOCK = new Integer[MAX_INTEGER_BLOCK - MIN_INTEGER_BLOCK + 1];
        for (int i = MIN_INTEGER_BLOCK; i <= MAX_INTEGER_BLOCK; i++)
        {
            INTEGER_BLOCK[i - MIN_INTEGER_BLOCK] = new Integer(i);
        }

        LONG_BLOCK = new Long[(int) (MAX_LONG_BLOCK - MIN_LONG_BLOCK) + 1];
        for (long i = MIN_LONG_BLOCK; i <= MAX_LONG_BLOCK; i++)
        {
            LONG_BLOCK[(int) (i - MIN_LONG_BLOCK)] = new Long(i);
        }
    }

    private NumberCache()
    {
    }

    /**
     * Gets the <code>Byte</code> instance for the specifed value.
     *
     * @param value the value for the object
     *
     * @return the <code>Byte</code> instance
     */
    public static Byte getByte(final byte value)
    {
        return BYTE_CACHE[value - Byte.MIN_VALUE];
    }

    /**
     * Gets the <code>Short</code> instance for the specifed value.
     *
     * @param value the value for the object
     *
     * @return the <code>Short</code> instance
     */
    public static Short getShort(final short value)
    {
        if ((MIN_SHORT_BLOCK <= value) && (value <= MAX_SHORT_BLOCK)) 
        {
            return SHORT_BLOCK[value - MIN_SHORT_BLOCK];
        }
        return (Short.MIN_VALUE == value) ? MIN_SHORT
              : ((Short.MAX_VALUE == value) ? MAX_SHORT : new Short(value));
    }

    /**
     * Gets the <code>Integer</code> instance for the specifed value.
     *
     * @param value the value for the object
     *
     * @return the <code>Integer</code> instance
     */
    public static Integer getInteger(final int value)
    {
        if ((MIN_INTEGER_BLOCK <= value) && (value <= MAX_INTEGER_BLOCK)) 
        {
            return INTEGER_BLOCK[value - MIN_INTEGER_BLOCK];
        }
        return (Integer.MIN_VALUE == value) ? MIN_INTEGER
              : ((Integer.MAX_VALUE == value) ? MAX_INTEGER : new Integer(value));
    }

    /**
     * Gets the <code>Long</code> instance for the specifed value.
     *
     * @param value the value for the object
     *
     * @return the <code>Long</code> instance
     */
    public static Long getLong(final long value)
    {
        if ((MIN_LONG_BLOCK <= value) && (value <= MAX_LONG_BLOCK)) 
        {
            return LONG_BLOCK[(int) (value - MIN_LONG_BLOCK)];
        }
        return (Long.MIN_VALUE == value) ? MIN_LONG
              : ((Long.MAX_VALUE == value) ? MAX_LONG : new Long(value));
    }

    /**
     * Gets the <code>Float</code> instance for the specifed value.
     *
     * @param value the value for the object
     *
     * @return the <code>Float</code> instance
     */
    public static Float getFloat(final float value)
    {
        if (0.0f == value) 
        {
            return ZERO_FLOAT;
        }
        else if (1.0f == value)
        {
            return ONE_FLOAT;
        }
        else if (-1.0f == value)
        {
            return NEG_ONE_FLOAT;
        }
        
        return new Float(value);
    }

    /**
     * Gets the <code>Double</code> instance for the specifed value.
     *
     * @param value the value for the object
     *
     * @return the <code>Double</code> instance
     */
    public static Double getDouble(final double value)
    {
        if (0.0 == value) 
        {
            return ZERO_DOUBLE;
        }
        else if (1.0 == value)
        {
            return ONE_DOUBLE;
        }
        else if (-1.0 == value)
        {
            return NEG_ONE_DOUBLE;
        }

        return new Double(value);
    }
}

