/*
* Copyright (C) The Community OpenORB Project. All rights reserved.
*
* This software is published under the terms of The OpenORB Community Software
* License version 1.0, a copy of which has been included with this distribution
* in the LICENSE.txt file.
*/
package org.openorb.util;

import java.util.Set;
import java.util.Map;
import java.util.TreeMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.SortedMap;
import java.util.Iterator;
import java.util.Arrays;
import java.util.ConcurrentModificationException;

import org.omg.CosNaming.NamingContextExt;
import org.omg.CosNaming.NamingContextExtHelper;
import org.omg.CosNaming.NamingContextExtPOA;
import org.omg.CosNaming.NameComponent;
import org.omg.CosNaming.NamingContext;
import org.omg.CosNaming.Binding;
import org.omg.CosNaming.BindingType;
import org.omg.CosNaming.BindingListHolder;
import org.omg.CosNaming.BindingIterator;
import org.omg.CosNaming.BindingIteratorHolder;
import org.omg.CosNaming.BindingIteratorHelper;
import org.omg.CosNaming.BindingIteratorPOA;
import org.omg.CosNaming.NamingContextHolder;

import org.omg.CosNaming.NamingContextPackage.NotFound;
import org.omg.CosNaming.NamingContextPackage.NotFoundReason;
import org.omg.CosNaming.NamingContextPackage.NotEmpty;
import org.omg.CosNaming.NamingContextPackage.CannotProceed;
import org.omg.CosNaming.NamingContextPackage.InvalidName;
import org.omg.CosNaming.NamingContextPackage.AlreadyBound;

import org.omg.CosNaming.NamingContextExtPackage.InvalidAddress;

import org.apache.avalon.framework.logger.Logger;

/**
 * <p>This class can be used for an easy to use transient naming context. The
 * context implements the SortedMap interface, and stores all it's bindings
 * as name/value pairs. Using the put operation with a key containing a 
 * non-existent context parent(s) will result in the parent contexts being
 * created.</p>
 *
 * <p>Special notes:</p>
 * <p>Deleting an internal context with the remove operation on 
 * any of the associated iterators or collections will cause undefined
 * behaviour. This may change in the future.</p>
 *
 * @author Chris Wood
 * @version $Revision: 1.17 $ $Date: 2002/06/28 08:59:06 $ 
 * @deprecated This class has been deprecated since OpenORB 1.2.2. A new 
 * version has been made available in the NamingService package
 * under org.openorb.tns. See the INS documentation for details.
 * This class will be removed very soon now, i.e. after the 1.3.0 release.
 */
public class MapNamingContext
    extends TreeMap
{
    private static int c_nameservs = 0;

    private org.omg.CORBA.ORB m_orb;

    private Set m_subContexts = new HashSet();

    private static final int SHUTDOWN_DISALLOW = 0;
    private static final int SHUTDOWN_DESTROY_POA = 1;
    private static final int SHUTDOWN_DESTROY_ORB = 2;

    private int m_shutdownMode = SHUTDOWN_DISALLOW;

    private NamingContextExt m_rootCtxt = null;

    private org.omg.PortableServer.POA m_firstPOA;
    private org.omg.PortableServer.POA m_ctxtPOA;
    private org.omg.PortableServer.POA m_iterPOA;

    private NamingContextExtImpl m_ctxtImpl = new NamingContextExtImpl();
    private BindingIteratorImpl m_iterImpl = new BindingIteratorImpl();

    private String m_endpoint = null;

    private boolean m_noDefaultAdapter = false;

    private Logger m_logger = null;

    /**
     * Create a new MapNamingContext.
     * The context's poa name will be somthing like NameServ_10. 
     *
     * @param orb orb under which the context is to be activated.
     * @param rootPOA parent POA under which to create the context's poa. 
     *    If null the orb's root POA is used.
     */
    public MapNamingContext( org.omg.CORBA.ORB orb, org.omg.PortableServer.POA rootPOA )
    {
        try
        {
            init( orb, rootPOA, null );
        }
        catch ( org.omg.PortableServer.POAPackage.AdapterAlreadyExists ex )
        {
            // this exception should be impossible
            if ( getLogger().isErrorEnabled() )
                getLogger().error( "The RootPOA already exists.", ex );

            throw new org.omg.CORBA.INTERNAL( "The RootPOA already exists (" + ex + ")" );
        }
    }

    /**
     * Create a new MapNamingContext.
     *
     * @param orb orb under which the context is to be activated.
     * @param rootPOA parent POA under which to create the context's poa. 
     *    If null the orb's root POA is used.
     * @param poaName name of poa created under rootPOA. If null the name will be
     *    somthing like NameServ_10.
     * @throws org.omg.PortableServer.POAPackage.AdapterAlreadyExists An adapter
     *    with the specified name already exists. This is never thrown if poaName
     *    is null.
     */
    public MapNamingContext( org.omg.CORBA.ORB orb, 
      org.omg.PortableServer.POA rootPOA, String poaName )
      throws org.omg.PortableServer.POAPackage.AdapterAlreadyExists
    {
        init( orb, rootPOA, poaName );
    }

    private void init( org.omg.CORBA.ORB orb, 
      org.omg.PortableServer.POA rootPOA, String poaName )
      throws org.omg.PortableServer.POAPackage.AdapterAlreadyExists
    {
        if ( rootPOA == null )
        {
            try
            {
                rootPOA = ( org.omg.PortableServer.POA ) 
                  orb.resolve_initial_references( "RootPOA" );
            }
            catch ( org.omg.CORBA.ORBPackage.InvalidName ex )
            {
                if ( getLogger().isErrorEnabled() )
                    getLogger().error( "Unable to resolve RootPOA.", ex );

                throw new org.omg.CORBA.INTERNAL( 
                  "Unable to resolve RootPOA (" + ex + ")" );
            }
        }

        m_orb = orb;

        org.omg.CORBA.Policy [] policies = new org.omg.CORBA.Policy[ 6 ];
        policies[ 0 ] = rootPOA.create_servant_retention_policy( 
          org.omg.PortableServer.ServantRetentionPolicyValue.NON_RETAIN );
        policies[ 1 ] = rootPOA.create_id_assignment_policy( 
          org.omg.PortableServer.IdAssignmentPolicyValue.USER_ID );
        policies[ 2 ] = rootPOA.create_id_uniqueness_policy( 
          org.omg.PortableServer.IdUniquenessPolicyValue.MULTIPLE_ID );
        policies[ 3 ] = rootPOA.create_implicit_activation_policy( 
          org.omg.PortableServer.ImplicitActivationPolicyValue.NO_IMPLICIT_ACTIVATION );
        policies[ 4 ] = rootPOA.create_lifespan_policy( 
          org.omg.PortableServer.LifespanPolicyValue.TRANSIENT );
        policies[ 5 ] = rootPOA.create_request_processing_policy( 
          org.omg.PortableServer.RequestProcessingPolicyValue.USE_DEFAULT_SERVANT );

        do
        {
            try
            {
                m_firstPOA = m_ctxtPOA = rootPOA.create_POA( 
                  ( ( poaName == null ) ? ( "NameServ_" 
                    + ( c_nameservs++ ) ) : poaName ), null, policies );

                policies[ 1 ] = rootPOA.create_id_assignment_policy( 
                  org.omg.PortableServer.IdAssignmentPolicyValue.SYSTEM_ID );
                m_iterPOA = m_firstPOA.create_POA( 
                  "_IterPOA", m_firstPOA.the_POAManager(), policies );
            }
            catch ( org.omg.PortableServer.POAPackage.AdapterAlreadyExists ex )
            {
                if ( poaName == null )
                    throw ex;
                continue;
            }
            catch ( org.omg.PortableServer.POAPackage.InvalidPolicy ex )
            {
                if ( getLogger().isErrorEnabled() )
                  getLogger().error( 
                   "Invalid policy passed to POA creation.", ex );

                throw new org.omg.CORBA.INTERNAL( 
                  "Invalid policy passed to POA creation (" + ex + ")" );
            }
        }
        while ( false );

        try
        {
            m_ctxtPOA.set_servant( m_ctxtImpl );
            m_iterPOA.set_servant( m_iterImpl );
        }
        catch ( org.omg.PortableServer.POAPackage.WrongPolicy ex )
        {
            if ( getLogger().isErrorEnabled() )
              getLogger().error( "POA has wrong policies.", ex );

            throw new org.omg.CORBA.INTERNAL( 
              "POA has wrong policies (" + ex + ")" );
        }

        m_rootCtxt = NamingContextExtHelper.narrow(
          m_ctxtPOA.create_reference_with_id( new byte[ 0 ], 
          NamingContextExtHelper.id() ) );

        try
        {
            m_ctxtPOA.the_POAManager().activate();
        }
        catch ( org.omg.PortableServer.POAManagerPackage.AdapterInactive ex )
        {
            if ( getLogger().isErrorEnabled() )
                getLogger().error( "Adapter is inactive.", ex );

            throw new org.omg.CORBA.INTERNAL( 
             "Adapter is inactive (" + ex + ")" );
        }
    }

    /**
     * Get a reference to the root naming context.
     * @return NamingContextExt the root context
     */
    public NamingContextExt getRootCtxt()
    {
        return m_rootCtxt;
    }

    /**
     * Bind the nameservice in the forward adapter, if there is a forward adapter.
     * @return A corbaloc style reference.
     * @throws IllegalStateException if no default adapter exists.
     */
    public String bindCorbaloc() throws IllegalStateException
    {
        if ( m_noDefaultAdapter )
        {
            throw new IllegalStateException( 
              "No default adapter exists. " 
              + "Default adapter must be initialized with the orb" );
        }

        if ( m_endpoint == null )
        {
            org.openorb.corbaloc.CorbalocService corbalocService;
            try
            {
                corbalocService = org.openorb.corbaloc.CorbalocServiceHelper.narrow(
                  m_orb.resolve_initial_references( "CorbalocService" ) );
            }
            catch ( org.omg.CORBA.ORBPackage.InvalidName ex )
            {
                m_noDefaultAdapter = true;

                if ( getLogger().isErrorEnabled() )
                    getLogger().error( 
                      "No default adapter exists. "
                      + " Default adapter must be initialized with the orb.", ex );

                throw new IllegalStateException( 
                  "No default adapter exists."
                  + " Default adapter must be initialized with the orb (" 
                  + ex + ")" );
            }

            corbalocService.put( "NameService", m_rootCtxt );

            org.openorb.CORBA.Delegate deleg = ( org.openorb.CORBA.Delegate ) 
             ( ( org.omg.CORBA.portable.ObjectImpl ) m_rootCtxt )._get_delegate();

            org.openorb.net.Address [] addrs = deleg.getAddresses( m_rootCtxt );

            for ( int i = 0; i < addrs.length; ++i )
                if ( addrs[ i ].getProtocol().equals( "iiop" ) )
                {
                    m_endpoint = addrs[ i ].getEndpointString();
                    break;
                }

            if ( m_endpoint == null )
                m_endpoint = addrs[ 0 ].getEndpointString();
        }

        return "corbaloc:" + m_endpoint + "/NameService";
    }

    /**
     * Return a corbaname style address for the string name passed.
     * @param str stringified corbaname of target.
     * @return String the corbaname
     * @throws IllegalArgumentException if str is not a valid stringified name
     * @throws IllegalStateException nameservice has not been bound as a corbaloc.
     */
    public String getCorbaname( String str )
      throws IllegalArgumentException, IllegalStateException  
    {
        if ( m_endpoint == null )
        {
          throw new IllegalStateException( 
            "Nameservice has not been bound as a corbaloc" );
        }

        try
        {
            return NamingUtils.to_url( m_endpoint, str );
        }
        catch ( InvalidName ex )
        {
            if ( getLogger().isErrorEnabled() )
                getLogger().error( "Invalid name " + str + ".", ex );
            throw new IllegalArgumentException( 
              "Invalid name " + str + " (" + ex + ")" );
        }
        catch ( InvalidAddress ex )
        {
            // this should be impossible
            if ( getLogger().isErrorEnabled() )
                getLogger().error( "Address is invalid.", ex );

            throw new IllegalArgumentException( 
              "Address is invalid (" + ex + ")" );
        }
    }

    /**
     * Return a corbaname style address for the name passed.
     * @param name the name to use.
     * @return String the corbaname
     * @throws IllegalArgumentException if name is invalid for some reason.
     * @throws IllegalStateException nameservice has not been bound as a corbaloc.
     */
    public String getCorbaname( final NameComponent[] name ) 
      throws IllegalArgumentException, IllegalStateException
    {
        if ( m_endpoint == null )
          throw new IllegalStateException( 
           "Nameservice has not been bound as a corbaloc" );

        String str;

        try
        {
            str = m_ctxtImpl.to_string( name );
        }
        catch ( InvalidName ex )
        {
            if ( getLogger().isErrorEnabled() )
                getLogger().error( "Invalid name " + name + ".", ex );

            throw new IllegalArgumentException( 
               "Invalid name " + name + " (" + ex + ")" );
        }

        return getCorbaname( str );
    }

    /**
     * Will the destroy_context operation work on the root context?
     *
     * @return Value of property allowSelfDestruct.
     */
    public boolean getAllowSelfDestruct()
    {
        return ( m_shutdownMode != SHUTDOWN_DISALLOW );
    }

    /**
     * When this is set to true the root context may be destroyed with the 
     * destroy operation. 
     *
     * @param allowSelfDestruct New value of property allowSelfDestruct.
     */
    public void setAllowSelfDestruct( boolean allowSelfDestruct )
    {
        m_shutdownMode = allowSelfDestruct ?
          SHUTDOWN_DESTROY_POA : SHUTDOWN_DISALLOW;
    }

    /**
     * Deactivate the server reference, and all other naming contexts created 
     * with the create_context operation.
     *
     * @param waitForComplete wait for completion before returning. If this
     *    parameter is true and this operation is called from a server thread
     *    an exception will be thrown.
     */
    public void deactivate( boolean waitForComplete )
    {
        m_ctxtPOA.destroy( true, waitForComplete );
    }

    /**
     * Add a binding to the map.<p>
     *
     * To insert a new empty context into the map use a key string ending in /
     * The object reference in this case will be ignored.
     * 
     * @param key String or NameComponent[] composing the binding's name.
     * @param value org.omg.CORBA.Object to be bound to the name.
     * @return Object 
     * @throws NullPointerException if key is null.
     * @throws ClassCastException if key or value is the wrong type.
     * @throws IllegalArgumentException the key is not a valid name.
     * @throws IllegalStateException the name is bound to a context, or one of
     *     it's ancestors are bound to an object.
     */
    public java.lang.Object put( final java.lang.Object key, final java.lang.Object value )
      throws NullPointerException, ClassCastException, IllegalArgumentException,
      IllegalStateException
    {
        if ( key == null )
            throw new NullPointerException( "Null keys cannot be used." );

        if ( value != null && !( value instanceof org.omg.CORBA.Object ) )
            throw new ClassCastException( 
              "Only org.omg.CORBA.Object may be used as values in this map." );

        if ( key instanceof String )
            return putStr( ( String ) key, ( org.omg.CORBA.Object ) value );

        if ( key instanceof NameComponent[] )
            return putName( ( NameComponent[] ) key, ( org.omg.CORBA.Object ) value );

        throw new ClassCastException( "Key type not compatible." );
    }

    /**
     * Add a binding to the map, creating ancestor contexts as neccicary.
     *
     * @param name The binding's name.
     * @param obj org.omg.CORBA.Object to be bound to the name.
     * @return Object
     * @throws NullPointerException if key is null.
     * @throws IllegalArgumentException the key is not a valid name.
     * @throws IllegalStateException the name is bound to a context, or one of
     *     it's ancestors are bound to an object.
     */
    public org.omg.CORBA.Object putName( final NameComponent[] name, 
      final org.omg.CORBA.Object obj )
      throws NullPointerException, IllegalStateException, IllegalArgumentException
    {
        if ( name.length == 0 )
            throw new IllegalArgumentException( "Name is invalid: Zero length" );

        String str;

        try
        {
            str = m_ctxtImpl.to_string( name );
        }
        catch ( InvalidName ex )
        {
            if ( getLogger().isErrorEnabled() )
                getLogger().error( "Invalid name " + name + ".", ex );
            throw new IllegalArgumentException( 
               "Invalid name " + name + " (" + ex + ")" );
        }

        synchronized ( this )
        {
            if ( m_subContexts.contains( str ) )
                throw new IllegalStateException( "Name is bound to a context" );

            addContext( NamingUtils.parent( name ) );

            return ( org.omg.CORBA.Object ) super.put( str, obj );
        }
    }

    /**
     * Add a binding to the map, creating ancestor contexts as neccicary.<p>
     *
     * To insert a new empty context into the map use a key string ending in /
     * The object reference in this case will be ignored.
     *
     * @param str The binding's name.
     * @param obj org.omg.CORBA.Object to be bound to the name.
     * @return org.omg.CORBA.Object object
     * @throws NullPointerException if key is null.
     * @throws IllegalArgumentException the key is not a valid name.
     * @throws IllegalStateException the name is bound to a context, or one of
     *     it's ancestors are bound to an object.
     */
    public org.omg.CORBA.Object putStr( String str, final org.omg.CORBA.Object obj )
      throws NullPointerException, IllegalArgumentException, IllegalStateException
    {
        if ( str.length() == 0 )
            throw new IllegalArgumentException( "Name is invalid: Zero length string" );

        boolean addBinding = true;

        if ( str.endsWith( "/" ) )
        {
            int pp = str.length() - 2;

            while ( pp >= 0 && str.charAt( pp ) == '\\' )
                --pp;

            if ( pp >= 0 && ( str.length() - pp ) % 2 == 0 )
            {
                str = str.substring( 0, str.length() - 2 );
                addBinding = false;
            }
        }

        NameComponent[] name;

        try
        {
            name = m_ctxtImpl.to_name( str );
        }
        catch ( InvalidName ex )
        {
            if ( getLogger().isErrorEnabled() )
                getLogger().error( "Invalid name " + str + ".", ex );

            throw new IllegalArgumentException( "Invalid name " + str + " (" + ex + ")" );
        }

        synchronized ( this )
        {
            if ( addBinding )
            {
                if ( m_subContexts.contains( str ) )
                    throw new IllegalStateException( "Name is bound to a context" );

                addContext( NamingUtils.parent( name ) );

                return ( org.omg.CORBA.Object ) super.put( str, obj );
            }
            else
            {
                addContext( name );
                return null;
            }
        }
    }

    /**
     * Add a context to the map, adding ancestor contexts as neccicary.
     * 
     * @param context Name of the context.
     * @return NamingContextExt naming context
     * @throws NullPointerException if context is null.
     * @throws IllegalArgumentException the context is not a valid name.
     * @throws IllegalStateException the context or one of
     *     it's ancestors are bound to an object.
     */
    public NamingContextExt addContext( final String context )
      throws NullPointerException, IllegalArgumentException, IllegalStateException
    {
        NameComponent[] name;

        try
        {
            name = m_ctxtImpl.to_name( context );
        }
        catch ( InvalidName ex )
        {
            if ( getLogger().isErrorEnabled() )
                getLogger().error( "Invalid name " + context + ".", ex );

            throw new IllegalArgumentException( "Invalid name " + context + " (" + ex + ")" );
        }

        return addContext( name );
    }

    /**
     * Add a context to the map, adding ancestor contexts as neccicary.
     * 
     * @param name Name of the context.
     * @return NamingContextExt naming context
     * @throws NullPointerException if name is null.
     * @throws IllegalArgumentException the name is not a valid name.
     * @throws IllegalStateException the name or one of
     *     it's ancestors are bound to an object.
     */
    public NamingContextExt addContext( final NameComponent[] name )
      throws NullPointerException, IllegalArgumentException, IllegalStateException
    {
        // generate binding names
        if ( name.length == 0 )
            return m_rootCtxt;

        String [] ctxt = new String[ name.length ];

        try
        {
            ctxt[ name.length - 1 ] = m_ctxtImpl.to_string( name );
        }
        catch ( InvalidName ex )
        {
            if ( getLogger().isErrorEnabled() )
                getLogger().error( "Invalid name " + name + ".", ex );

            throw new IllegalArgumentException( 
              "Invalid name " + name + " (" + ex + ")" );
        }

        for ( int i = name.length - 2; i >= 0; --i )
            ctxt[ i ] = NamingUtils.parent( ctxt[ i + 1 ] );

        NamingContextExt last = null;

        synchronized ( this )
        {
            // check for preexisting objects
            int i = 0;

            for ( ; i < ctxt.length; ++i )
            {
                Object find = get( ctxt[ i ] );

                if ( find == null )
                    break;
                else if ( !( find instanceof NamingContext ) )
                {
                    throw new java.lang.IllegalStateException( 
                      "Could not create context \"" + ctxt[ i ] 
                      + "\", already bound to object" );
                }
                else if ( !m_subContexts.contains( ctxt[ i ] ) )
                {
                    throw new java.lang.IllegalStateException( 
                      "Could not create context \"" + ctxt[ i ] 
                      + "\", already bound to external context" );
                }
                last = ( NamingContextExt ) find;
            }

            // create any needed contexts.
            for ( ; i < ctxt.length; ++i )
            {
                last = m_ctxtImpl.createRef( ctxt[ i ] );
            }
        }

        return last;
    }

    /**
     * Remove a context. This will remove the context and all it's decendants.
     * @param context the name of the context.
     * @return boolean true if the context was destroyed.
     */
    public synchronized boolean removeContext( String context )
    {
        if ( m_subContexts.remove( context ) )
        {
            super.remove( context );
            super.subMap( context + '/', context + ( '/' + 1 ) ).clear();
            return true;
        }
        return false;
    }

    /**
     * Determine if the specified name is a subcontext and must be removed with
     * the removeContext operation.
     * @param context the name of the context.
     * @return true if the name is bound to a context.
     */
    public synchronized boolean isContext( String context )
    {
        return m_subContexts.contains( context );
    }

    /**
     * Put all entries in the map.
     * @param p1 the map
     */
    public void putAll( final Map p1 )
    {
        if ( p1 instanceof MapNamingContext )
        {
            // use the sorted map's putAll, it's faster.
            super.putAll( p1 );
        }
        else
        {
            // gotta check every entry.
            Iterator i = p1.entrySet().iterator();

            while ( i.hasNext() )
            {
                Map.Entry e = ( Map.Entry ) i.next();
                put( e.getKey(), e.getValue() );
            }
        }
    }

    /**
     * This function finds the first parent which is bound context, and which
     * is not a context created by this server. 
     * If an object is found a NotFound exception is thrown with not_context
     * status, if an internal context is found a NotFound exception is thrown
     * with missing_node status. Lock on this must be owned.
     * @param name a name
     * @param parent parent context
     * @exception NotFound 
     */
    private NameComponent [] tailOffExternal( final String name, NamingContextHolder parent )
        throws NotFound
    {
        String par = name;

        while ( par.length() > 0 )
        {
            Object obj = get( par );

            if ( obj == null )
            {
                par = NamingUtils.parent( par );
                continue;
            }

            NameComponent [] rest;

            try
            {
                rest = m_ctxtImpl.to_name( name.substring( par.length() + 1 ) );
            }
            catch ( InvalidName ex )
            {
                if ( getLogger().isErrorEnabled() )
                    getLogger().error( "Invalid name " + name + ".", ex );

                throw new org.omg.CORBA.INTERNAL( 
                    "Invalid name " + name + " (" + ex + ")" );
            }

            if ( obj instanceof NamingContext )
            {
                if ( m_subContexts.contains( par ) )
                    throw new NotFound( NotFoundReason.missing_node, rest );

                parent.value = ( NamingContext ) obj;

                return rest;
            }

            throw new NotFound( NotFoundReason.not_context, rest );
        }

        NameComponent [] rest;

        try
        {
            rest = m_ctxtImpl.to_name( name );
        }
        catch ( InvalidName ex )
        {
            if ( getLogger().isErrorEnabled() )
                getLogger().error( "Invalid name " + name + ".", ex );

            throw new org.omg.CORBA.INTERNAL( 
              "Invalid name " + name + " (" + ex + ")" );
        }

        throw new NotFound( NotFoundReason.missing_node, rest );
    }

    private void putToMap( String key, Object value )
    {
        super.put( key, value );
    }

    private static class ByteSeqKey extends java.lang.Object
    {
        public ByteSeqKey( byte [] key )
        {
            _key = key;
            _hashCode = 0;

            for ( int i = 0; i < _key.length; ++i )
                _hashCode = 31 * _hashCode + _key[ i ];
        }

        private byte [] _key;
        private int _hashCode;

        public int hashCode()
        {
            return _hashCode;
        }

        public boolean equals( final java.lang.Object obj )
        {
            if ( obj == null || !( obj instanceof ByteSeqKey ) )
                return false;

            if ( obj == this )
                return true;

            ByteSeqKey b2 = ( ByteSeqKey ) obj;

            if ( b2._hashCode != _hashCode )
                return false;

            return Arrays.equals( _key, b2._key );
        }
    }

    private class NamingContextExtImpl extends NamingContextExtPOA
    {
        /**
         * Names can have multiple components; therefore, name resolution
         * can traverse multiple contexts.
         *
         * @param  n The compound name for the object to resolve.
         * @return  the resolved object.
         *
         * @exception  NotFound  Indicates the name does not identify a binding.
         * @exception  CannotProceed Indicates that the implementation has
         *         given up for some reason. The client, however,
         *         may be able to continue the operation at the
         *         returned naming context.
         * @exception  InvalidName Indicates the name is invalid. (A name
         *         of length 0 is invalid; implementations may
         *         place other restrictions on names.)
         */
        public org.omg.CORBA.Object resolve( NameComponent[] n )
        throws NotFound, CannotProceed, InvalidName
        {
            // check for invalid name
            if ( n.length == 0 )
                throw new InvalidName();

            String name = getPrefix() + to_string( n );

            NamingContextHolder parent;

            NameComponent [] parent_name;

            synchronized ( MapNamingContext.this )
            {
                if ( m_rootCtxt == null )
                {
                    if ( m_shutdownMode == SHUTDOWN_DESTROY_POA )
                    {
                        throw new org.omg.CORBA.OBJECT_NOT_EXIST( 
                          0, org.omg.CORBA.CompletionStatus.COMPLETED_YES );
                    }
                    else
                    {
                        throw new org.omg.CORBA.COMM_FAILURE( 
                          0, org.omg.CORBA.CompletionStatus.COMPLETED_NO );
                    }
                }

                if ( containsKey( name ) )
                    return ( org.omg.CORBA.Object ) get( name );

                parent = new NamingContextHolder();

                parent_name = tailOffExternal( name, parent );
            }

            return parent.value.resolve( parent_name );
        }

        /**
         * This is a convenience operation that performs a resolve in the
         * same manner as NamingContext::resolve.  It accepts a
         * stringified name as an argument instead of a Name.
         *
         * @param  n the stringified name of the object (or naming context)
         * to resolve
         * @return  the resolved object.
         *
         * @exception  NotFound  Indicates the name does not identify a binding.
         * @exception  CannotProceed Indicates that the implementation has
         *         given up for some reason. The client, however,
         *         may be able to continue the operation at the
         *         returned naming context.
         * @exception  InvalidName Indicates the name is invalid. (A name
         *         of length 0 is invalid; implementations may
         *         place other restrictions on names.)
         */
        public org.omg.CORBA.Object resolve_str( String name )
        throws NotFound, CannotProceed, InvalidName
        {
            // check for invalid name
            to_name( name );

            name = getPrefix() + name;

            NamingContextHolder parent;
            NameComponent [] parent_name;

            synchronized ( MapNamingContext.this )
            {
                if ( m_rootCtxt == null )
                {
                    if ( m_shutdownMode == SHUTDOWN_DESTROY_POA )
                    {
                        throw new org.omg.CORBA.OBJECT_NOT_EXIST(
                          0, org.omg.CORBA.CompletionStatus.COMPLETED_YES );
                    }
                    else
                    {
                        throw new org.omg.CORBA.COMM_FAILURE( 
                          0, org.omg.CORBA.CompletionStatus.COMPLETED_NO );
                    }
                }

                if ( containsKey( name ) )
                    return ( org.omg.CORBA.Object ) get( name );

                parent = new NamingContextHolder();

                parent_name = tailOffExternal( name, parent );
            }

            return parent.value.resolve( parent_name );
        }

        /**
         * Creates a binding of a name and an object in the naming
         * context. Naming contexts that are bound using bind do not
         * participate in name resolution when compound names are passed
         * to be resolved.
         *
         * @param  n The compound name for the object to bind
         * @param  obj The object to bind
         *
         * @exception  NotFound  Indicates the name does not identify a binding.
         *
         * @exception  CannotProceed Indicates that the implementation has
         *         given up for some reason. The client, however,
         *         may be able to continue the operation at the
         *         returned naming context.
         * @exception  Indicates the name is invalid. (A name of length 0
         *         is invalid; implementations may place other
         *         restrictions on names.)
         * @exception  AlreadyBound Indicates an object is already bound to
         *         the specified name. Only one object can be bound
         *         to a particular name in a context. The bind and
         *         the bind_context operations raise the
         *         AlreadyBound exception if the name is bound in
         *         the context; the rebind and rebind_context
         *         operations unbind the name and rebind the name
         *         to the object passed as an argument.
         */
        public void bind( NameComponent[] n, org.omg.CORBA.Object obj )
        throws NotFound, CannotProceed, InvalidName, AlreadyBound
        {
            if ( n.length == 0 )
                throw new InvalidName();

            String name = getPrefix() + to_string( n );

            String cname = NamingUtils.parent( name );

            NamingContextHolder parent;

            NameComponent [] parent_name;

            synchronized ( MapNamingContext.this )
            {
                if ( m_rootCtxt == null )
                {
                    if ( m_shutdownMode == SHUTDOWN_DESTROY_POA )
                    {
                        throw new org.omg.CORBA.OBJECT_NOT_EXIST( 
                          0, org.omg.CORBA.CompletionStatus.COMPLETED_YES );
                    }
                    else
                    {
                        throw new org.omg.CORBA.COMM_FAILURE( 
                          0, org.omg.CORBA.CompletionStatus.COMPLETED_NO );
                    }
                }

                if ( containsKey( name ) )
                    throw new AlreadyBound();

                if ( cname.length() == 0 || m_subContexts.contains( cname ) )
                {
                    putToMap( name, obj );
                    return ;
                }

                parent = new NamingContextHolder();
                parent_name = tailOffExternal( name, parent );
            }

            parent.value.bind( parent_name, obj );
        }

        /**
         * Creates a binding of a name and an object in the naming context
         * even if the name is already bound in the context. Naming
         * contexts that are bound using rebind do not participate in name
         * resolution when compound names are passed to be resolved.
         *
         * @param  n The compound name for the object to rebind
         * @param  obj The object to rebind
         *
         * @exception  NotFound  Indicates the name does not identify a binding.
         * @exception  CannotProceed In this implmementation this exception will
         *                be thrown if the the target cannot be deleted as it is
         *                not a standalone context, it was created with the 
         *                bind_new_context operation or is an entry in the map view.
         * @exception  InvalidName Indicates the name is invalid. (A name
         *         of length 0 is invalid; implementations may
         *         place other restrictions on names.)
         */
        public void rebind( NameComponent[] n, org.omg.CORBA.Object obj )
        throws NotFound, CannotProceed, InvalidName
        {
            if ( n.length == 0 )
                throw new InvalidName();

            String name = getPrefix() + to_string( n );

            String cname = NamingUtils.parent( name );

            NamingContextHolder parent;

            NameComponent [] parent_name;

            synchronized ( MapNamingContext.this )
            {
                if ( m_rootCtxt == null )
                {
                    if ( m_shutdownMode == SHUTDOWN_DESTROY_POA )
                    {
                        throw new org.omg.CORBA.OBJECT_NOT_EXIST( 
                          0, org.omg.CORBA.CompletionStatus.COMPLETED_YES );
                    }
                    else
                    {
                        throw new org.omg.CORBA.COMM_FAILURE( 
                          0, org.omg.CORBA.CompletionStatus.COMPLETED_NO );
                    }
                }

                if ( m_subContexts.contains( name ) )
                {
                    if ( ( ( org.omg.CORBA.Object ) get( name ) )._is_equivalent( obj ) )
                        return ;

                    NamingContextExt ref;

                    if ( cname.length() != 0 )
                    {
                        ref = ( NamingContextExt ) get( cname );

                        if ( ref == null )
                            ref = createRef( cname );
                    }
                    else
                        ref = m_rootCtxt;

                    throw new CannotProceed( ref, new NameComponent[]
                                             {
                                                 n[ n.length - 1 ]
                                             }

                                           );
                }

                if ( cname.length() == 0 || m_subContexts.contains( cname ) )
                {
                    putToMap( name, obj );
                    return ;
                }

                parent = new NamingContextHolder();
                parent_name = tailOffExternal( name, parent );
            }

            parent.value.rebind( parent_name, obj );
        }

        /**
         * The unbind operation removes a name binding from a context.
         *
         * @param  n The compound name for the node to unbind ( an object
         * or a naming context )
         *
         * @exception  NotFound  Indicates the name does not identify a binding.
         * @exception  CannotProceed In this implmementation this exception will
         *                be thrown if the the target cannot be deleted as it is
         *                not a standalone context, it was created with the 
         *                bind_new_context operation or is an entry in the map view.
         * @exception  InvalidName Indicates the name is invalid. (A name
         *         of length 0 is invalid; implementations may
         *         place other restrictions on names.)
         */
        public void unbind( NameComponent[] n )
        throws NotFound, CannotProceed, InvalidName
        {
            if ( n.length == 0 )
                throw new InvalidName();

            String name = getPrefix() + to_string( n );

            NamingContextHolder parent;

            NameComponent [] parent_name;

            synchronized ( MapNamingContext.this )
            {
                if ( m_rootCtxt == null )
                {
                    if ( m_shutdownMode == SHUTDOWN_DESTROY_POA )
                    {
                        throw new org.omg.CORBA.OBJECT_NOT_EXIST( 
                          0, org.omg.CORBA.CompletionStatus.COMPLETED_YES );
                    }
                    else
                    {
                        throw new org.omg.CORBA.COMM_FAILURE( 
                          0, org.omg.CORBA.CompletionStatus.COMPLETED_NO );
                    }
                }

                if ( m_subContexts.contains( name ) )
                {
                    String cname = NamingUtils.parent( name );
                    NamingContextExt ref;

                    if ( cname.length() != 0 )
                    {
                        ref = ( NamingContextExt ) get( cname );

                        if ( ref == null )
                            ref = createRef( cname );
                    }
                    else
                        ref = m_rootCtxt;

                    throw new CannotProceed( ref, new NameComponent[]
                                             {
                                                 n[ n.length - 1 ]
                                             }

                                           );
                }

                if ( remove( name ) != null )
                    return ;

                parent = new NamingContextHolder();

                parent_name = tailOffExternal( name, parent );
            }

            parent.value.unbind( parent_name );
        }

        /**
         * This operation creates a new context and binds it to the name
         * supplied as an argument. The newly-created context is
         * implemented by the same naming server as the context in which
         * it was bound (that is, the naming server that implements the
         * context denoted by the name argument excluding the last
         * component).
         *
         * Note that in this server this operation produces different results 
         * to a new_context followed by a bind_context operation, the context
         * created by this operation cannot be unbound by the unbind operation
         * but will create entries in the map view of the context.
         *
         * @param  n The compound name for the naming context to create and
         * to bind.
         *
         * @exception  NotFound Indicates the name does not identify a
         * binding.
         * @exception  CannotProceed Indicates that the implementation has
         * given up for some reason. The client, however,
         * may be able to continue the operation at the
         * returned naming context.
         * @exception  InvalidName Indicates the name is invalid. (A name
         *         of length 0 is invalid; implementations may
         *         place other restrictions on names.)
         * @exception  AlreadyBound Indicates an object is already bound to
         *         the specified name. Only one object can be bound
         *         to a particular name in a context.
         */
        public NamingContext bind_new_context( NameComponent[] n )
        throws NotFound, AlreadyBound, CannotProceed, InvalidName
        {
            if ( n.length == 0 )
                throw new InvalidName();

            String name = getPrefix() + to_string( n );

            String cname = NamingUtils.parent( name );

            NamingContextHolder parent;

            NameComponent [] parent_name;

            synchronized ( MapNamingContext.this )
            {
                if ( m_rootCtxt == null )
                {
                    if ( m_shutdownMode == SHUTDOWN_DESTROY_POA )
                    {
                        throw new org.omg.CORBA.OBJECT_NOT_EXIST(
                          0, org.omg.CORBA.CompletionStatus.COMPLETED_YES );
                    }
                    else
                    {
                        throw new org.omg.CORBA.COMM_FAILURE( 
                          0, org.omg.CORBA.CompletionStatus.COMPLETED_NO );
                    }
                }

                if ( containsKey( name ) )
                    throw new AlreadyBound();

                if ( cname.length() == 0 || m_subContexts.contains( cname ) )
                    return createRef( name );

                parent = new NamingContextHolder();

                parent_name = tailOffExternal( name, parent );
            }

            return parent.value.bind_new_context( parent_name );
        }

        /**
         * Names an object that is a naming context. Naming contexts that
         * are bound using bind_context() participate in name resolution
         * when compound names are passed to be resolved.
         *
         * @param  n The compound name for the naming context to bind
         * @param  obj The naming context to bind  
         *
         * @exception  NotFound  Indicates the name does not identify a binding.
         * @exception  CannotProceed Indicates that the implementation has
         *         given up for some reason. The client, however,
         *         may be able to continue the operation at the
         *         returned naming context.
         * @exception  InvalidName Indicates the name is invalid. (A name
         *         of length 0 is invalid; implementations may
         *         place other restrictions on names.)
         * @exception  AlreadyBound Indicates an object is already bound to
         *         the specified name. Only one object can be bound
         *         to a particular name in a context. The bind and
         *         the bind_context operations raise the
         *         AlreadyBound exception if the name is bound in
         *         the context; the rebind and rebind_context
         *         operations unbind the name and rebind the name
         *         to the object passed as an argument.
         */
        public void bind_context( NameComponent[] n, NamingContext nc )
        throws NotFound, CannotProceed, InvalidName, AlreadyBound
        {
            if ( n.length == 0 )
                throw new InvalidName();

            String name = getPrefix() + to_string( n );

            String cname = NamingUtils.parent( name );

            NamingContextHolder parent;

            NameComponent [] parent_name;

            synchronized ( MapNamingContext.this )
            {
                if ( m_rootCtxt == null )
                {
                    if ( m_shutdownMode == SHUTDOWN_DESTROY_POA )
                    {
                        throw new org.omg.CORBA.OBJECT_NOT_EXIST( 
                         0, org.omg.CORBA.CompletionStatus.COMPLETED_YES );
                    }
                    else
                    {
                        throw new org.omg.CORBA.COMM_FAILURE( 
                          0, org.omg.CORBA.CompletionStatus.COMPLETED_NO );
                    }
                }

                if ( containsKey( name ) )
                    throw new AlreadyBound();

                if ( cname.length() == 0 || m_subContexts.contains( cname ) )
                {
                    putToMap( name, nc );
                    return ;
                }

                parent = new NamingContextHolder();
                parent_name = tailOffExternal( name, parent );
            }

            parent.value.bind_context( parent_name, nc );
        }

        /**
         * Creates a binding of a name and a naming context in the naming
         * context even if the name is already bound in the
         * context. Naming contexts that are bound using rebind_context()
         * participate in name resolution when compound names are passed
         * to be resolved.
         *
         * @param  n The compound name for the naming context to rebind
         * @param  obj The naming context to rebind    
         *
         * @exception  NotFound Indicates the name does not identify a
         * binding.
         * @exception  CannotProceed In this implmementation this exception will
         *                be thrown if the the target cannot be deleted as it is
         *                not a standalone context, it was created with the 
         *                bind_new_context operation or is an entry in the map view.
         * @exception  InvalidName Indicates the name is invalid. (A name
         *         of length 0 is invalid; implementations may
         *         place other restrictions on names.)
         * @exception  AlreadyBound Indicates an object is already bound to
         *         the specified name. Only one object can be bound
         *         to a particular name in a context. The bind and
         *         the bind_context operations raise the
         *         AlreadyBound exception if the name is bound in
         *         the context; the rebind and rebind_context
         *         operations unbind the name and rebind the name
         *         to the object passed as an argument.
         */
        public void rebind_context( NameComponent[] n, NamingContext nc )
        throws NotFound, CannotProceed, InvalidName
        {
            if ( n.length == 0 )
                throw new InvalidName();

            String name = getPrefix() + to_string( n );

            String cname = NamingUtils.parent( name );

            NamingContextHolder parent;

            NameComponent [] parent_name;

            synchronized ( MapNamingContext.this )
            {
                if ( m_rootCtxt == null )
                {
                    if ( m_shutdownMode == SHUTDOWN_DESTROY_POA )
                    {
                        throw new org.omg.CORBA.OBJECT_NOT_EXIST( 
                          0, org.omg.CORBA.CompletionStatus.COMPLETED_YES );
                    }
                    else
                    {
                        throw new org.omg.CORBA.COMM_FAILURE( 
                          0, org.omg.CORBA.CompletionStatus.COMPLETED_NO );
                    }
                }

                if ( m_subContexts.contains( name ) )
                {
                    if ( ( ( org.omg.CORBA.Object ) get( name ) )._is_equivalent( nc ) )
                        return ;

                    NamingContextExt ref;

                    if ( cname.length() != 0 )
                    {
                        ref = ( NamingContextExt ) get( cname );

                        if ( ref == null )
                            ref = createRef( cname );
                    }
                    else
                        ref = m_rootCtxt;

                    throw new CannotProceed( ref, new NameComponent[]
                                             {
                                                 n[ n.length - 1 ]
                                             }

                                           );
                }

                if ( cname.length() == 0 || m_subContexts.contains( cname ) )
                {
                    putToMap( name, nc );
                    return ;
                }

                parent = new NamingContextHolder();
                parent_name = tailOffExternal( name, parent );
            }

            parent.value.rebind_context( parent_name, nc );
        }

        /**
         * This operation returns a naming context implemented by the same
         * naming server as the context on which the operation was
         * invoked. The new context is not bound to any name.<p/>
         *
         * Note that in this implementation this operation followed by a 
         * bind_context operation produces different results to a bind_new_context,
         * the context created with this operation can be unbound since it exists
         * as a separate entity from the context which created it, but the bindings
         * within it will not appear in the Map view of the context which created it.<p/>
         *
         * The only association between this context and the context which created
         * it is that this context is lifetime constrained by it's creator, that is
         * if the root of this context is destroyed then the context returned from
         * this operation will also be destoryed.
         *
         * @return  A new naming context.
         */
        public NamingContext new_context()
        {
            MapNamingContext newImpl = new MapNamingContext( _orb(), m_firstPOA );
            newImpl.m_firstPOA = m_firstPOA;
            newImpl.setAllowSelfDestruct( true );
            return newImpl.getRootCtxt();
        }


        /**
         * The destroy operation deletes a naming context.
         *
         * @exception  NotEmpty If the naming context contains bindings,
         * the NotEmpty exception is raised.
         */
        public void destroy()
        throws NotEmpty
        {
            String name = getPrefix();

            // self destruct sequence activated.
            if ( name.length() == 0 )
            {
                synchronized ( MapNamingContext.this )
                {
                    if ( !isEmpty() )
                        throw new NotEmpty();

                    if ( m_shutdownMode == SHUTDOWN_DISALLOW )
                        throw new org.omg.CORBA.NO_PERMISSION();

                    m_rootCtxt = null;
                }

                if ( m_shutdownMode == SHUTDOWN_DESTROY_POA )
                    m_ctxtPOA.destroy( true, false );
                else
                    _orb().shutdown( false );

                return ;
            }

            name = name.substring( 0, name.length() - 1 );

            synchronized ( MapNamingContext.this )
            {
                if ( m_rootCtxt == null )
                {
                    if ( m_shutdownMode == SHUTDOWN_DESTROY_POA )
                    {
                        throw new org.omg.CORBA.OBJECT_NOT_EXIST( 
                          0, org.omg.CORBA.CompletionStatus.COMPLETED_YES );
                    }
                    else
                    {
                        throw new org.omg.CORBA.COMM_FAILURE( 
                          0, org.omg.CORBA.CompletionStatus.COMPLETED_NO );
                    }
                }

                if ( !subMap( name + '/', name + ( '/' + 1 ) ).isEmpty() )
                    throw new NotEmpty();

                remove( name );

                m_subContexts.remove( name );
            }
        }

        /**
         * This operation accepts a stringified name and returns a Name.       
         *
         * @param  sn the stringified name to transform to a name.
         * @exception  InvalidName This exception is raised if the
         *         stringified name is syntactically malformed or
         *         violates an implementation limit.
         */
        public NameComponent[] to_name( java.lang.String sn )
        throws InvalidName
        {
            return NamingUtils.to_name( sn );
        }

        /**
         * This operation accepts Name and returns a stringified name. 
         *
         * @param  n the name to stringified.
         * @exception  InvalidName This exception is raised if the name is
         * invalid.
         */
        public java.lang.String to_string( NameComponent[] n )
        throws InvalidName
        {
            return NamingUtils.to_string( n );
        }

        /**
         * This operation takes an URL address and performs any escapes
         * necessary on the stringified name and returns a fully formed
         * URL string.
         *
         * @param  addr the address ( for example myhost.xyz.com )
         * @param  sn  the stringified name to add to the URL
         * @return  the URL string format.
         *
         * @exception  InvalidAddress This exception is raises if a address
         *         is invalid ( it means that the address does not
         *         respect the address format ).
         * @exception  InvalidName This exception is raised if the
         *         stringified name is syntactically malformed or
         *         violates an implementation limit.
         */
        public String to_url( String addr, String sn )
        throws InvalidAddress, InvalidName
        {
            return NamingUtils.to_url( addr, sn );
        }

        /**
         * The list operation allows a client to iterate through a set of
         * bindings in a naming context.
         *
         * @param  how_many Maximum number of elements into the binding list.
         * @param  bl This parameter returns a list that contains all node
         * of the naming context
         * @param  bi This parameter returns a binding iterator to iterate
         * in the list.
         *
         * @return  The list operation returns at most the requested number
         * of bindings in BindingList bl.
         *     - If the naming context contains additional bindings,
         *     the list operation returns a BindingIterator with the
         *     additional bindings.
         *     - If the naming context does not contain additional
         *     bindings, the binding iterator is a nil object
         *     reference.
         */
        public void list( int how_many, BindingListHolder bl, BindingIteratorHolder bi )
        {
            String parent = getPrefix();
            String last = null;
            java.util.Iterator itt;
            SortedMap subMap;

            if ( parent.length() > 0 )
                parent = parent.substring( 0, parent.length() - 1 );

            int count = 0;

            Map.Entry [] ents = new Map.Entry[ how_many ];

            synchronized ( MapNamingContext.this )
            {
                // setup the iterator
                if ( parent.length() == 0 )
                    subMap = MapNamingContext.this;
                else
                    subMap = subMap( parent + '/', parent + ( '/' + 1 ) );

                itt = subMap.entrySet().iterator();

                // read off entries.
                String tpar;

                while ( count < how_many && itt.hasNext() )
                {
                    ents[ count ] = ( Map.Entry ) itt.next();
                    last = ( String ) ents[ count ].getKey();
                    tpar = NamingUtils.parent( last );

                    if ( tpar.equals( parent ) )
                        ++count;
                    else
                    {
                        // skip over child contexts in one hop
                        subMap = subMap.tailMap( tpar + ( '/' + 1 ) );
                        itt = subMap.entrySet().iterator();
                    }
                }
            }

            if ( count == 0 )
            {
                bl.value = new Binding[ 0 ];
            }
            else
            {
                String name;
                Binding [] ret = new Binding[ count ];

                for ( int i = 0; i < count; ++i )
                {
                    if ( ents[ i ].getValue() instanceof NamingContext )
                        ret[ i ] = new Binding( null, BindingType.ncontext );
                    else
                        ret[ i ] = new Binding( null, BindingType.nobject );

                    name = ( String ) ents[ i ].getKey();

                    if ( parent.length() != 0 )
                        name = name.substring( NamingUtils.parent( name ).length() + 1 );

                    try
                    {
                        ret[ i ].binding_name = m_ctxtImpl.to_name( name );
                    }
                    catch ( InvalidName ex )
                    {
                        if ( getLogger().isErrorEnabled() )
                            getLogger().error( "Invalid name " + name + ".", ex );

                        throw new org.omg.CORBA.INTERNAL( 
                          "Invalid name " + name + " (" + ex + ")" );
                    }
                }

                bl.value = ret;
            }

            if ( itt.hasNext() )
            {
                if ( last == null )
                    last = "";

                bi.value = m_iterImpl.createIterator( parent, last, itt, subMap );
            }
        }

        private String getPrefix()
        {
            String prefix = new String( _object_id() );

            if ( prefix.length() == 0 )
                return "";

            if ( !m_subContexts.contains( prefix ) )
            {
                throw new org.omg.CORBA.OBJECT_NOT_EXIST( 
                  0, org.omg.CORBA.CompletionStatus.COMPLETED_YES );
            }
            return prefix + "/";
        }

        NamingContextExt createRef( String name )
        {
            NamingContextExt ret = NamingContextExtHelper.narrow(
               m_ctxtPOA.create_reference_with_id( name.getBytes(), 
               NamingContextExtHelper.id() ) );

            m_subContexts.add( name );
            putToMap( name, ret );
            return ret;
        }
    }

    private static class IteratorData
    {
        String parent;
        String last;
        java.util.Iterator itt;
        SortedMap subMap;
    }

    private class BindingIteratorImpl 
        extends BindingIteratorPOA
    {
        private Map m_iteratorData = new HashMap();

        public BindingIterator createIterator( String parent, String last, 
          Iterator itt, SortedMap subMap )
        {
            IteratorData id = new IteratorData();
            id.parent = parent;
            id.last = last;
            id.itt = itt;
            id.subMap = subMap;

            try
            {
                BindingIterator bi = BindingIteratorHelper.narrow( 
                  m_iterPOA.create_reference( BindingIteratorHelper.id() ) );
                m_iteratorData.put( new ByteSeqKey( m_iterPOA.reference_to_id( bi ) ), id );
                return bi;
            }
            catch ( org.omg.PortableServer.POAPackage.WrongPolicy ex )
            {
                if ( getLogger().isErrorEnabled() )
                    getLogger().error( "POA has wrong policies.", ex );

                throw new org.omg.CORBA.INTERNAL( "POA has wrong policies (" + ex + ")" );
            }
            catch ( org.omg.PortableServer.POAPackage.WrongAdapter ex )
            {
                if ( getLogger().isErrorEnabled() )
                    getLogger().error( "Wrong adapter used.", ex );

                throw new org.omg.CORBA.INTERNAL( 
                  "Wrong adapter used (" + ex + ")" );
            }
        }

        /**
         * This operation destroys the iterator.
         */
        public void destroy()
        {
            synchronized ( MapNamingContext.this )
            {
                if ( m_iteratorData.remove( getKey() ) == null )
                {
                    throw new org.omg.CORBA.OBJECT_NOT_EXIST( 
                      0, org.omg.CORBA.CompletionStatus.COMPLETED_YES );
                }
            }
        }

        /**
         * This operation returns the next binding.
         *
         * @return  If there are no more bindings, false is returned.
         */
        public boolean next_one( org.omg.CosNaming.BindingHolder b )
        {
            Map.Entry next = null;
            String tpar;
            boolean rootParent;

            synchronized ( MapNamingContext.this )
            {
                IteratorData id = ( IteratorData ) m_iteratorData.get( getKey() );

                if ( id == null )
                {
                    throw new org.omg.CORBA.OBJECT_NOT_EXIST( 
                     0, org.omg.CORBA.CompletionStatus.COMPLETED_YES );
                }

                rootParent = id.parent.length() == 0;

                while ( next == null && id.itt.hasNext() )
                {
                    try
                    {
                        do
                        {
                            next = ( Map.Entry ) id.itt.next();
                            id.last = ( String ) next.getKey();
                            tpar = NamingUtils.parent( id.last );

                            if ( !tpar.equals( id.parent ) )
                            {
                                // skip over child contexts in one hop
                                id.subMap = id.subMap.tailMap( tpar + ( '/' + 1 ) );
                                id.itt = id.subMap.entrySet().iterator();
                                next = null;
                            }
                        }
                        while ( next == null && id.itt.hasNext() );
                    }
                    catch ( ConcurrentModificationException ex )
                    {
                        id.subMap = id.subMap.tailMap( id.last + '\0' );
                        id.itt = id.subMap.entrySet().iterator();
                    }
                }
            }

            if ( next == null )
            {
                b.value = new Binding( new NameComponent[ 0 ], BindingType.nobject );
                return false;
            }

            if ( next.getValue() instanceof NamingContext )
                b.value = new Binding( null, BindingType.ncontext );
            else
                b.value = new Binding( null, BindingType.nobject );

            String name = ( String ) next.getKey();

            if ( !rootParent )
                name = name.substring( NamingUtils.parent( name ).length() + 1 );

            try
            {
                b.value.binding_name = m_ctxtImpl.to_name( name );
            }
            catch ( InvalidName ex )
            {
                if ( getLogger().isErrorEnabled() )
                    getLogger().error( "Invalid name " + name + ".", ex );

                throw new org.omg.CORBA.INTERNAL( 
                  "Invalid name " + name + " (" + ex + ")" );
            }

            return true;
        }

        /**
         * This operation returns at most the requested number of bindings.
         */
        public boolean next_n( int how_many, org.omg.CosNaming.BindingListHolder bl )
        {
            int count = 0;
            Map.Entry [] ents = new Map.Entry[ how_many ];
            String tpar;
            boolean rootParent;

            synchronized ( MapNamingContext.this )
            {
                IteratorData id = ( IteratorData ) m_iteratorData.get( getKey() );

                if ( id == null )
                    throw new org.omg.CORBA.OBJECT_NOT_EXIST( 
                      0, org.omg.CORBA.CompletionStatus.COMPLETED_YES );

                rootParent = id.parent.length() == 0;

                while ( count < how_many && id.itt.hasNext() )
                {
                    try
                    {
                        do
                        {
                            ents[ count ] = ( Map.Entry ) id.itt.next();
                            id.last = ( String ) ents[ count ].getKey();
                            tpar = NamingUtils.parent( id.last );

                            if ( tpar.equals( id.parent ) )
                            {
                                ++count;
                            }
                            else
                            {
                                // skip over child contexts in one hop
                                id.subMap = id.subMap.tailMap( tpar + ( '/' + 1 ) );
                                id.itt = id.subMap.entrySet().iterator();
                            }
                        }
                        while ( count < how_many && id.itt.hasNext() );
                    }
                    catch ( ConcurrentModificationException ex )
                    {
                        id.subMap = id.subMap.tailMap( id.last + '\0' );
                        id.itt = id.subMap.entrySet().iterator();
                    }
                }
            }

            if ( count == 0 )
            {
                bl.value = new Binding[ 0 ];
                return false;
            }

            String name;
            Binding [] ret = new Binding[ count ];

            try
            {
                for ( int i = 0; i < count; ++i )
                {
                    if ( ents[ i ].getValue() instanceof NamingContext )
                        ret[ i ] = new Binding( null, BindingType.ncontext );
                    else
                        ret[ i ] = new Binding( null, BindingType.nobject );

                    name = ( String ) ents[ i ].getKey();

                    if ( !rootParent )
                        name = name.substring( NamingUtils.parent( name ).length() + 1 );

                    ret[ i ].binding_name = m_ctxtImpl.to_name( name );
                }
            }
            catch ( InvalidName ex )
            {
                if ( getLogger().isErrorEnabled() )
                    getLogger().error( ex.toString(), ex );

                throw new org.omg.CORBA.INTERNAL( ex.toString() );
            }

            bl.value = ret;
            return true;
        }

        private Object getKey()
        {
            return new ByteSeqKey( _object_id() );
        }
    }

    /**
     * Main method.
     * @param args command line args
     */
    public static void main( String [] args )
    {
        boolean printIOR = false;
        boolean printLoc = false;
        boolean asDefault = false;
        boolean shutdownAll = false;

        for ( int i = 0; i < args.length; ++i )
        {
            if ( args[ i ].equals( "-default" ) || args[ i ].equals( "--default" ) )
            {
                asDefault = true;
            }
            else if ( args[ i ].equals( "-printIOR" ) || args[ i ].equals( "--printIOR" ) )
            {
                printIOR = true;
            }
            else if ( args[ i ].equals( "-print" ) || args[ i ].equals( "--print" ) )
            {
                printLoc = true;
            }
            else if ( args[ i ].equals( "-shutdown" ) || args[ i ].equals( "--shutdown" ) )
            {
                shutdownAll = true;
            }
            else if ( args[ i ].startsWith( "-ORB" ) )
            {
                if ( args.length < i + 1 && !args[ i + 1 ].startsWith( "-" ) )
                {
                    ++i;
                }
            }
            else
            {
                System.err.println( "usage: org.openorb.util.MapNamingContext " 
                  + "[-default] [-printref] [-shutdown]" );
                System.err.println( "       print      Print corbaloc format "
                  + "reference to stderr." );
                System.err.println( "       printIOR   Print IOR format reference "
                  + "to stderr." );
                System.err.println( "       default    Run on default port." );
                System.err.println( "       shutdown   Allow the destroy operation "
                  + "on the rootNC to shutdown the server." );

                if ( args[ i ].equals( "-help" ) || args[ i ].equals( "--help" ) )
                {
                    return ;
                }
                System.exit( 1 );
            }
        }

        java.util.Properties props = new java.util.Properties();

        if ( asDefault )
        {
            props.put( "ImportModule.DefaultCorbalocService", 
             "${openorb.home}config/default.xml#DefaultCorbalocService" );
        }
        else
        {
            props.put( "ImportModule.CorbalocService", 
              "${openorb.home}config/default.xml#CorbalocService" );
        }
        org.omg.CORBA.ORB orb = org.omg.CORBA.ORB.init( args, props );
        initialize(args, orb);
        orb.run();
    }

    /**
     * This method provides an alternative way of starting the name service. Other projects, 
     * e.g. the OpenEJB container, use only one ORB for all the services to make use the 
     * optimized CORBA local invocations.
     * @param args the command line flags in the same format as in the main method
     * @param orb the ORB instance the name service should be registered with
     * @return the root name context
     */
    public static NamingContextExt initialize( String [] args, org.omg.CORBA.ORB orb )
    {
        boolean printIOR = false;
        boolean printLoc = false;
        boolean asDefault = false;
        boolean shutdownAll = false;

        for ( int i = 0; i < args.length; ++i )
        {
            if ( args[ i ].equals( "-default" ) || args[ i ].equals( "--default" ) )
                asDefault = true;
            else if ( args[ i ].equals( "-printIOR" ) || args[ i ].equals( "--printIOR" ) )
                printIOR = true;
            else if ( args[ i ].equals( "-print" ) || args[ i ].equals( "--print" ) )
                printLoc = true;
            else if ( args[ i ].equals( "-shutdown" ) || args[ i ].equals( "--shutdown" ) )
                shutdownAll = true;
            else if ( args[ i ].startsWith( "-ORB" ) )
            {
                if ( args.length < i + 1 && !args[ i + 1 ].startsWith( "-" ) )
                    ++i;
            }
            else
            {
                System.err.println( 
                  "usage: org.openorb.util.MapNamingContext [-default] [-printref] [-shutdown]" );
                System.err.println( "       print      Print corbaloc format "
                  + "reference to stderr." );
                System.err.println( "       printIOR   Print IOR format reference to stderr." );
                System.err.println( "       default    Run on default port." );
                System.err.println( "       shutdown   Allow the destroy operation on the rootNC " 
                  + " to shutdown the server." );
                if ( args[ i ].equals( "-help" ) || args[ i ].equals( "--help" ) )
                    return null;
                System.exit( 1 );
            }
        }

        MapNamingContext map = new MapNamingContext( orb, null );

        if ( shutdownAll )
            map.m_shutdownMode = SHUTDOWN_DESTROY_ORB;

        NamingContextExt root_ns = map.getRootCtxt();

        String corbaloc = map.bindCorbaloc();

        System.out.println( "Transient naming service activated." );

        if ( printIOR )
        {
            System.out.println( "IOR Reference:" );
            System.err.println( orb.object_to_string( root_ns ) );
        }

        if ( printLoc )
        {
            System.out.println( "Corbaloc Reference:" );
            System.err.println( corbaloc );
        }

        try
        {
            org.omg.PortableServer.POA rootPOA = ( org.omg.PortableServer.POA ) 
              orb.resolve_initial_references( "RootPOA" );
            rootPOA.the_POAManager().activate();
        }
        catch ( org.omg.CORBA.ORBPackage.InvalidName ex )
        {
            ex.printStackTrace();
            throw new org.omg.CORBA.INTERNAL( "Unable to resolve RootPOA (" + ex + ")" );
        }
        catch ( org.omg.PortableServer.POAManagerPackage.AdapterInactive ex )
        {
            ex.printStackTrace();
            throw new org.omg.CORBA.INTERNAL( "Adapter is in the inactive state (" + ex + ")" );
        }
        return root_ns;
    }

    private Logger getLogger()
    {
        if ( null == m_logger )
        {
            m_logger = ((org.openorb.CORBA.ORBSingleton) m_orb).getLogger();
        }
        return m_logger;
    };

}
