/*
* Copyright (C) The Community OpenORB Project. All rights reserved.
*
* This software is published under the terms of The OpenORB Community Software
* License version 1.0, a copy of which has been included with this distribution
* in the LICENSE.txt file.
*/
package org.openorb.util;

/**
 * A utility class for obtaining <code>Character</code> instances.
 * This class caches the most commonly used values.
 *
 * @author Richard G Clark
 */
public final class CharacterCache 
{
    private static final Character MIN_VALUE = new Character( Character.MIN_VALUE );
    private static final Character MAX_VALUE = new Character( Character.MAX_VALUE );
    private static final Character[] CACHE_BLOCK;
    private static final char BLOCK_MIN = 0x20;
    private static final char BLOCK_MAX = 0x7E;
        
    static 
    {
        CACHE_BLOCK = new Character[BLOCK_MAX - BLOCK_MIN + 1];
        for (char value = BLOCK_MIN; value <= BLOCK_MAX; value++)
        {
            CACHE_BLOCK[value - BLOCK_MIN] = new Character(value);
        }
    }

    private CharacterCache()
    {
    }
    
    /**
     * Gets the <code>Character</code> instance for the specifed value.
     *
     * @param value the value for the object
     *
     * @return the <code>Character</code> instance
     */
    public static Character getCharacter( final char value )
    {   
        if ( ( BLOCK_MIN <= value ) && ( value <= BLOCK_MAX ) )
        {
            return CACHE_BLOCK[value - BLOCK_MIN];
        }
        return ((Character.MIN_VALUE == value) ? MIN_VALUE
              : ( ( Character.MAX_VALUE == value) ? MAX_VALUE : new Character(value)));
    }
}

