/*
* Copyright (C) The Community OpenORB Project. All rights reserved.
*
* This software is published under the terms of The OpenORB Community Software
* License version 1.0, a copy of which has been included with this distribution
* in the LICENSE.txt file.
*/

package org.openorb.compiler;

import org.openorb.compiler.object.*;
import org.openorb.compiler.parser.*;
import org.openorb.compiler.generator.*;
import java.util.*;
import java.net.URL;
import java.net.MalformedURLException;

import org.openorb.ReleaseInfo;

/**
 * This class is the IDL compiler implementation.
 *
 * @author Jerome Daniel
 * @version $Revision: 1.11 $ $Date: 2002/08/21 08:19:18 $ 
 */

public class IdlCompiler
{

    public static final String tab = "    ";

    /**
     * Include directories
     */
    public static Vector includeList = new java.util.Vector();

    /**
     * Maps #defined macros to their expansions
     */
    public static Hashtable definedMacros = new Hashtable();

    /**
     * Pseudo Import list
     */
    public static Vector importLink = new java.util.Vector();

    /**
     * Native descriptions
     */
    public static Vector nativeDefinition = new java.util.Vector();

    /**
     * IDL file name to compile
     */
    public static String [] idl_file_name;

    /**
     * IDL files list
     */
    public static java.util.Vector idl_file_name_list = new java.util.Vector();

    /**
     * Indicates if the stub must be generated
     */
    public static boolean map_stub = true;

    /**
     * Indicates using local stubs
     */
    public static boolean local_stub = true;

    /**
     * Indicates if the skeleton must be generated
     */
    public static boolean map_skeleton = true;

    /**
     * Indicates if the included elements must be mapped
     */
    public static boolean map_all = false;

    /**
     * Indicates if TIE approach must be generated
     */
    public static boolean map_tie = true;

    /**
     * Indicates if the user code must be generated
     */
    public static boolean map_user = false;

    /**
     * Indicates if the compilation uses PIDL
     */
    public static boolean pidl = false;

    /**
     * Indicates if the POA adapter is used
     */
    public static boolean map_poa = true;

    /**
     * Indicates if verbose mode is activated
     */
    public static boolean verbose = false;

    /**
     * Package name for the generated code
     */
    public static String packageName = null;

    /**
     * Indicates if a package must be used
     */
    public static boolean use_package = false;

    /**
     * Use portable helper. This is only needed if the class must be compiled
      * without openorb.
      */
    public static boolean portableHelper = false;

    /**
     * Indicates the output directory
     */
    public static String outdir = null;

    /**
     * Indicates if the the stub and skeleton uses DII and DSI
     */
    public static boolean dynamic = false;

    /**
     * Indicates if the prefix is used to name the packages
     *
     */
    public static boolean usePrefix = true;

    /**
     * Active silent mode
     */
    public static boolean silentMode = false;

    /**
     * Display bad flag
     */
    public static boolean displayBadFlag = true;

    /**
     * Reverse the prefix
     */
    public static boolean reversePrefix = true;
    
    public static boolean useReflection;
    public static boolean useSwitch;
    public static boolean useClasses = true;

    public static boolean jdk1_4;
    
    public static int minTableSize = 9;
    
    public static String generateValueFactory;
    public static String generateValueImpl;
    /**
     * Display a help message
     */
    public static void display_help()
    {
        System.out.println( "" );
        System.out.println( "" );
        System.out.println( "---------------------------------------------------------------" );
        System.out.println( "OpenORB                                                        " );
        System.out.println( "(c) 2002 The Community OpenORB" );
        System.out.println( "IDL To Java Compiler                                           " );
        System.out.println( "---------------------------------------------------------------" );
        System.out.println( "" );
        System.out.println( "" );
        System.out.println( "Options" );
        System.out.println( "\t-all" );
        System.out.println( "\t\tGenerate mapping for included files." );

        System.out.println( "\t-boa" );
        System.out.println( "\t\tGenerate skeleton for BOA." );

        System.out.println( "\t-d directory_name" );
        System.out.println( "\t\tProvide a way to specify the ouput dir. This option" );
        System.out.println( "\t\twill not use the 'generated' directory." );
        System.out.println( "\t\tFor example :" );
        System.out.println( "\t\t org.openorb.compiler.IdlCompiler demo.idl -d /home/me/" );

        System.out.println( "\t-dynamic" );
        System.out.println( "\t\tGenerate stub with DII and skeleton with DSI" );
        System.out.println( "\t\t( portable way before CORBA 2.3 )." );

        System.out.println( "\t-D" );
        System.out.println( "\t\tDefine a symbol. It is equivalent to #define." );

        System.out.println( "\t-importLink link" );

        System.out.println( "\t-I" );
        System.out.println( "\t\tAllow specification of include directory." );
        System.out.println( "\t\tExample:" );
        System.out.println( "\t\t  org.openorb.compiler.IdlCompiler demo.idl -I /home/me/idl -I ../other" );

        System.out.println( "\t-native native_name native_mapping" );
        System.out.println( "\t\tDefine native type mapping." );
        System.out.println( "\t\tFor example : " );
        System.out.println( "\t\t org.openorb.compiler.IdlCompiler demo.idl -native cookie java.lang.Object" );
        System.out.println( "\t\t this command implies the mapping of" );
        System.out.println( "\t\t cookie in java.lang.Object." );

        System.out.println( "\t-nolocalstub" );
        System.out.println( "\t\tGenerate stubs without local invocation path." );

        System.out.println( "\t-noprefix" );
        System.out.println( "\t\tDon't use prefixes as packages names." );

        System.out.println( "\t-noreverseprefix" );
        System.out.println( "\t\tThe prefixes are used as package name but they are not reversed." );

        System.out.println( "\t-noskeleton" );
        System.out.println( "\t\tDon't generate skeleton." );

        System.out.println( "\t-nostub" );
        System.out.println( "\t\tDon't generate stub." );

        System.out.println( "\t-notie" );
        System.out.println( "\t\tDon't generate TIE classes for delegation mode." );

        System.out.println( "\t-package package_name" );
        System.out.println( "\t\tGenerate files in package_name." );
        System.out.println( "\t\tExample:" );
        System.out.println( "\t\t org.openorb.compiler.IdlCompiler demo.idl -package exemple" );

        System.out.println( "\t-pidl" );

        System.out.println( "\t-portablehelper" );
        System.out.println( "\t\tGenerate portable helper classes. Nonportable helpers must be" );
        System.out.println( "\t\tcompiled using OpenORB, but the generated class files will work" );
        System.out.println( "\t\tanywhere. Portable helpers can be compiled anywhere and will" );
        System.out.println( "\t\twork identically on all orbs with Any.extract_Streamable." );

        System.out.println( "\t-quiet" );
        System.out.println( "\t\tSuppress any output. Same as -silence." );

        System.out.println( "\t-release" );
        System.out.println( "\t\tShow version number." );

        System.out.println( "\t-silence" );
        System.out.println( "\t\tSuppress any output. Same as -quiet." );

        System.out.println( "\t-verbose" );
        System.out.println( "\t\tShow debug output." );
        
        System.out.println( "\t-jdk1.4" );
        System.out.println( "\t\tGenerate classes that use JDK1.4 features." );
        System.out.println( "\t\tThe generate classes will not compile on previous versions." );

        System.out.println( "\t-invokeMethod <Classes|Reflection|Switch>" );
        System.out.println( "\t\tThe method used to implement the xxxPOA invoke method." );
        System.out.println( "\t\tIf not specified Classes is used." );

        System.out.println( "\t-minTableSize <size>" );
        System.out.println( "\t\tThe minimum size of method table." );

        System.out.println( "\t-XgenerateValueFactory <impl postfix>" );
        System.out.println( "\t\tGenerate default value factories for valuetypes." );
        System.out.println( "\t\tCreates instance of <valuetype name><impl postfix>." );

        System.out.println( "\t-XgenerateValueImpl <postfix>" );
        System.out.println( "\t\tGeneratae default implementation of valuetypes." );
        System.out.println( "\t\tCreates classes with name <valuetype name><postfix>." );

        System.out.println( "" );
        System.out.println( "" );
        System.exit( 1 );
    }

    /**
     * Scan command line arguments
     */
    public static void scan_arguments( String[] args )
    {
        for ( int i = 0; i < args.length; i++ )
        {
            if ( args[ i ].charAt( 0 ) != '-' )
                idl_file_name_list.addElement( args[ i ] );
            else
                if ( args[ i ].equals( "-release" ) )
                {
                    System.out.println( ReleaseInfo.RELEASE );
                    System.exit( 1 );
                }
                else if ( args[ i ].equals( "-h" ) || args[ i ].equals( "-help" ) )
                    display_help();
                else if ( args[ i ].equals( "-silence" ) || args[ i ].equals( "-quiet" ) )
                    silentMode = true;
                else if ( args[ i ].equals( "-verbose" ) )
                    verbose = true;
                else if ( args[ i ].equals( "-noprefix" ) )
                    usePrefix = false;
                else if ( args[ i ].equals( "-noreverseprefix" ) )
                    reversePrefix = false;
                else if ( args[ i ].equals( "-nostub" ) )
                    map_stub = false;
                else if ( args[ i ].equals( "-nolocalstub" ) )
                    local_stub = false;
                else if ( args[ i ].equals( "-noskeleton" ) )
                    map_skeleton = false;
                else if ( args[ i ].equals( "-notie" ) )
                    map_tie = false;
                else if ( args[ i ].equals( "-portablehelper" ) )
                    portableHelper = true;
                else if ( args[ i ].equals( "-boa" ) )
                    map_poa = false;
                else if ( args[ i ].equals( "-dynamic" ) )
                    dynamic = true;
                else if ( args[ i ].equals( "-all" ) )
                    map_all = true;
                else if ( args[ i ].equals( "-pidl" ) )
                {
                    pidl = true;
                    map_stub = false;
                    map_skeleton = false;
                    map_tie = false;
                }
                else if ( "-jdk1.4".equals( args[i] ) )
                {
                	jdk1_4 = true;
                }
                else if ( "-XgenerateValueFactory".equals( args[i] ) )
                {
                    if ( ( i + 1 ) == args.length )
                    {
                        System.out.println( "Argument expected after '-XgenerateValueFactory'" );
                        System.exit( 2 );
                    }
                    i++;
                    generateValueFactory = args[i];
                }
                else if ( "-XgenerateValueImpl".equals( args[i] ) )
                {
                    if ( ( i + 1 ) == args.length )
                    {
                        System.out.println( "Argument expected after '-XgenerateValueImpl'" );
                        System.exit( 2 );
                    }
                    i++;
                    generateValueImpl = args[i];
                }
                else if ( "-invokeMethod".equals( args[i] ) )
                {
                    if ( ( i + 1 ) == args.length )
                    {
                        System.out.println( "Argument expected after '-invokeMethod'" );
                        System.exit( 2 );
                    }
                    i++;
                    useReflection = false;
                    useSwitch = false;
                    useClasses = false;
                    
                    if ( "Classes".equals( args[i] ) )
                    {
                        useClasses = true;
                    }
                    else if ( "Reflection".equals( args[i] ) )
                    {
                        useReflection = true;
                    }
                    else if ( "Switch".equals( args[i] ) )
                    {
                        useSwitch = true;
                    }
                    else
                    {
                        System.out.println( "'-invokeMethod' support arguments: " + 
                                "'Classes', 'Switch' and 'Reflection'" );
                        System.exit( 2 );
                    }
                }
                else if ( "-minTableSize".equals( args[i] ) )
                {
                    if ( i + 1 == args.length )
                    {
                        System.out.println( "Argument expected after '-minTableSize'" );
                        System.exit( 2 );
                    }

                    minTableSize = Integer.parseInt(args[ ++i ]);
                }
                else if ( args[ i ].equals( "-package" ) )
                {
                    if ( i + 1 == args.length )
                    {
                        System.out.println( "Argument expected after '-package'" );
                        System.exit( 2 );
                    }

                    packageName = args[ ++i ];
                    use_package = false;
                }
                else if ( args[ i ].equals( "-importlink" ) )
                {
                    if ( i + 1 == args.length )
                    {
                        System.out.println( "Argument expected after '-importlink'" );
                        System.exit( 2 );
                    }

                    importLink.addElement( args[ ++i ] );
                }
                else if ( args[ i ].equals( "-d" ) )
                {
                    if ( i + 1 == args.length )
                    {
                        System.out.println( "Argument expected after '-d'" );
                        System.exit( 2 );
                    }

                    outdir = args[ ++i ];

                    if ( packageName == null )
                    {
                        packageName = "";
                        use_package = false;
                    }
                }
                else if ( args[ i ].equals( "-I" ) )
                {
                    if ( ++i == args.length )
                    {
                        System.out.println( "Argument expected after '-I'" );
                        System.exit( 2 );
                    }

                    URL url = null;

                    try
                    {
                        url = new URL( args[ i ] );
                    }
                    catch ( MalformedURLException ex )
                    {
                        try
                        {
                            url = new java.io.File( args[ i ] ).toURL();
                        }
                        catch ( MalformedURLException ex1 )
                        {}

                    }

                    if ( url != null )
                        includeList.addElement( url );
                }
                else if ( args[ i ].startsWith( "-I" ) )
                {
                    String path = args[ i ].substring( 2 );

                    URL url = null;

                    try
                    {
                        url = new URL( path );
                    }
                    catch ( MalformedURLException ex )
                    {
                        try
                        {
                            url = new java.io.File( path ).toURL();
                        }
                        catch ( MalformedURLException ex1 )
                        {}

                    }

                    if ( url != null )
                        includeList.addElement( url );
                }
                else if ( args[ i ].startsWith( "-D" ) )
                {
                    try
                    {
                        int idx = args[ i ].indexOf( '=' );

                        if ( idx < 0 )
                            definedMacros.put( args[ i ].substring( 2, args[ i ].length() ) , "" );
                        else
                            definedMacros.put( args[ i ].substring( 2, idx ) , args[ i ].substring( idx + 1 ) );
                    }
                    catch ( StringIndexOutOfBoundsException ex )
                    { }

                }
                else if ( args[ i ].equals( "-native" ) )
                {
                    if ( i + 2 == args.length )
                    {
                        System.out.println( "Argument expected after '-native'" );
                        System.exit( 2 );
                    }

                    nativeDefinition.addElement( args[ ++i ] + ":" + args[ ++i ] );
                }
                else if ( displayBadFlag )
                {
                    System.out.println( "Bad parameter : " + args[ i ] );
                    System.out.println( "Please, use no flag to display all compiler option flags" );
                    System.exit( 2 );
                }
        }

        if ( outdir == null || outdir.equals( "" ) )
        {
            outdir = new String( "generated" );

            if ( packageName == null )
            {
                packageName = "";
                use_package = false;
            }
        }

        idl_file_name = new String[ idl_file_name_list.size() ];

        for ( int i = 0; i < idl_file_name_list.size(); i++ )
            idl_file_name[ i ] = ( String ) idl_file_name_list.elementAt( i );
    }

    /**
     * Get argument from configuration file
     */
    public static void configFile()
    {
        Configurator config = null;

        // -- try to load the configurator --

        try
        {
            java.lang.Object obj = Thread.currentThread().getContextClassLoader().loadClass( "org.openorb.compiler.orb.DefaultConfigurator" ).newInstance();

            config = ( org.openorb.compiler.Configurator ) obj;
        }
        catch ( java.lang.Exception ex )
        {
            return ;
        }

        // -- Invoke the configurator --

        config.updateInfo( includeList, importLink );
    }

    /**
     * Prepare the compilation process
     */
    public static void prepare()
    {
        // define some standard macros
        definedMacros.put( "__IDL_TO_JAVA__", 
            "" );
        definedMacros.put( "__OPENORB__", 
            ReleaseInfo.RELEASE );
        definedMacros.put( "__OPENORB_MAJOR__", 
            "" + ReleaseInfo.VERSION_MAJOR );
        definedMacros.put( "__OPENORB_MINOR__", 
            "" + ReleaseInfo.VERSION_MINOR );
        definedMacros.put( "__OPENORB_MINOR_CHANGE__", 
            "" + ReleaseInfo.VERSION_MINOR_CHANGE );
        definedMacros.put( "__CORBA_IDL__", 
            ReleaseInfo.SPEC_VERSION_MAJOR + "." + ReleaseInfo.SPEC_VERSION_MINOR );
        definedMacros.put( "__CORBA_MAJOR__", 
            "" + ReleaseInfo.SPEC_VERSION_MAJOR );
        definedMacros.put( "__CORBA_MINOR__", 
            "" + ReleaseInfo.SPEC_VERSION_MINOR );
        definedMacros.put( "__CORBA_MINOR_CHANGE__", 
            "" + ReleaseInfo.SPEC_VERSION );
    }

    /**
     * This operation is used to compile an IDL file
     */
    public static void compile_file( String file_name, IdlParser Parser )
    throws org.openorb.compiler.parser.CompilationException
    {
        IdlObject CompilationGraph = Parser.compile_idl( file_name );

        if ( Parser.totalError != 0 )
        {
            System.out.println( "there are errors..." );
            System.out.println( "compilation process stopped !" );
            System.exit( 2 );
        }

        // -- Start to generate Java code --

        if ( verbose )
            System.out.println( "Now translate to Java..." );

        IdlToJava toJava = new IdlToJava();

        toJava.translateData( CompilationGraph, packageName );

        if ( map_stub == true )
            toJava.translateStub( CompilationGraph, packageName );

        if ( map_skeleton == true )
            toJava.translateSkeleton( CompilationGraph, packageName );

        if ( map_tie == true )
            toJava.translateTIE( CompilationGraph, packageName );
    }

    public static IdlParser createIDLParser( String [] args )
    {
        prepare();

        scan_arguments( args );

        configFile();

        return new IdlParser();
    }

    // ===========================
    // Class entry point
    // ===========================
    /**
     * The IDL compiler starts here
     *
     * @param args  command line arguments list
     */
    public static void main( String[] args )
    {
        SymboleDef s;

        if ( args.length == 0 )
            display_help();

        IdlParser Parser = createIDLParser( args );

        if ( idl_file_name.length == 0 )
            display_help();

        // -- Welcome message --
        if ( !silentMode )
            System.out.println( "OpenORB  IDL To Java Compiler / (c) 2002 The Community OpenORB" );

        // -- Starts the compilation --

        try
        {
            java.util.Hashtable definedCopy = ( java.util.Hashtable ) definedMacros.clone();

            for ( int i = 0; i < idl_file_name.length; i++ )
            {
                definedMacros = definedCopy;

                if ( !silentMode )
                    System.out.println( "compile : " + idl_file_name[ i ] );

                compile_file( idl_file_name[ i ], Parser );
            }
        }
        catch ( org.openorb.compiler.parser.CompilationException ex )
        {
            System.out.println( "there are errors..." );
            System.out.println( "compilation process stopped !" );
            System.exit( 2 );
        }
    }
}
