
/* This file is part of xtmodcoll-3.0, (c) xtmod.com
 * 
 * xtmodcoll-3.0 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * xtmodcoll-3.0 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser Public License
 * along with xtmodcoll-3.0.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.xtmod.util.algo;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import com.xtmod.util.algo.Feature.SortableFeature;




/**
 * This class takes three type parameters:
 * A is the element type of the first list,
 * B is the element type of the second list, and 
 * C is the type they have in common (the type to
 * use when comparing elements from both lists, often String).
 * 
 */
public class ListDiff<A, B, R extends Comparable> {
	
	// initial values
	protected List<A> aa;
	protected List<B> bb; 
	protected SortableFeature<A, R> fa;
	protected SortableFeature<B, R> fb;
	
	// computed values
	protected List<A> inAOnly;
	protected List<A> inBothA;
	protected List<B> inBothB;
	protected List<B> inBOnly;

	
	/**
	 * This constructor will assume the lists are unsorted.
	 * 
	 * @param aa
	 * @param bb
	 * @param fa
	 * @param fb
	 */
	public ListDiff (List<A> aa, List<B> bb, SortableFeature<A, R> fa, SortableFeature<B, R> fb) {
		this(aa, bb, fa, fb, false);
	}

	/**
	 *  
	 * @param aa
	 * @param bb
	 * @param fa
	 * @param fb
	 * @param listsAreSorted  to indicate that the lists are already sorted
	 */
	public ListDiff (List<A> aa, List<B> bb, SortableFeature<A, R> fa, SortableFeature<B, R> fb, boolean listsAreSorted) {
		this.aa = aa;
		this.bb = bb;
		this.fa = fa;
		this.fb = fb;

		if (!listsAreSorted) {
			sortLists();
		}
		
		diff();
	}

	protected void sortLists() {

		// sort first list
		Comparator<A> compA = new Comparator<A>() {
			@SuppressWarnings("unchecked")
			public int compare (A a1, A a2) {
				return fa.feature(a1).compareTo(fa.feature(a2));
			}
		};
		
		List<A> aa_sorted = new ArrayList<A>(this.aa);
		Collections.sort(aa_sorted, compA);

		
		// sort second list
		Comparator<B> compB = new Comparator<B>() {
			@SuppressWarnings("unchecked")
			public int compare (B b1, B b2) {
				return fb.feature(b1).compareTo(fb.feature(b2));
			}
		};
		
		List<B> bb_sorted = new ArrayList<B>(this.bb);
		Collections.sort(bb_sorted, compB);

		// re-assign lists
		this.aa = aa_sorted;
		this.bb = bb_sorted;
	}
	
	
	protected void diff () {

		inAOnly = new ArrayList<A>();
		inBothA = new ArrayList<A>();
		inBothB = new ArrayList<B>();
		inBOnly = new ArrayList<B>();
	
		int ib = 0;
		int ia = 0;

		for (;;) {

			if (ia == aa.size()) {
				inBOnly.addAll(bb.subList(ib, bb.size()));
				break;
			}
			if (ib == bb.size()) {
				inAOnly.addAll(aa.subList(ia, aa.size()));
				break;
			}

			A a = aa.get(ia);
			B b = bb.get(ib);
			R a_ = fa.feature(a);
			R b_ = fb.feature(b);
			
			int eq = a_.compareTo(b_);
			
			if (eq < 0) {
				/* a<b : no such b */
				inAOnly.add(a);
				ia += 1;
			} else if (eq == 0) {
				/* a=b : */
				inBothA.add(a);
				inBothB.add(b);
				ia += 1;
				ib += 1;
			} else if (eq > 0) {
				/* a>b : */
				inBOnly.add(b);
				ib += 1;
			}
		}

	}
	

	
	/** 
	 * A retaining only B's
	 */
	public List<A> inBothA() {
		return inBothA;
	}

	/** 
	 * B retaining only A's
	 */
	public List<B> inBothB() {
		return inBothB;
	}
	
	/** 
	 * A after removing all B's
    */
	public List<A> inAOnly() {
		return inAOnly;
	}
	
	/** 
	 * B after removing all A's
	 */
	public List<B> inBOnly() {
		return inBOnly;
	}
	
	/**
	 * Do both lists contain the same?
	 * @return equivalence 
	 */
	public boolean areEquivalent() {
		return (inAOnly().size() == 0 && inBOnly().size() == 0);
	}
	
}

