/* Copyright (C) 2003-2005 Peter J. Verveer
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met: 
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.      
 */

#define ND_IMPORT_ARRAY
#include "nd_image.h"
#undef ND_IMPORT_ARRAY
#include "ni_support.h"
#include "ni_filters.h"
#include "ni_fourier.h"
#include "ni_morphology.h"
#include "ni_interpolation.h"
#include "ni_measure.h"

typedef struct {
  PyObject *function;
  PyObject *extra_arguments;
  PyObject *extra_keywords;
} NI_PythonCallbackData;

/* return number of elements */
maybelong NI_GetArraySize(PyArrayObject *array)
{
  return NA_elements(array);
}

/* Convert an input array of any type, not necessarily contiguous */
static int 
NI_ObjectToInputArray(PyObject *object, PyArrayObject **array)
{
  *array = NA_InputArray(object, tAny, NUM_ALIGNED|NUM_NOTSWAPPED);
  return *array ? 1 : 0;
}

/* Convert an output array of any type, not necessarily contiguous */
static int 
NI_ObjectToOutputArray(PyObject *object, PyArrayObject **array)
{
  *array = NA_OutputArray(object, tAny, NUM_ALIGNED|NUM_NOTSWAPPED);
  return *array ? 1 : 0;
}

/*********************************************************************/
/* old stuff: */
/*********************************************************************/

/* return the rank of an array */
int NI_GetArrayRank(PyArrayObject *array)
{
  assert(array != NULL);
  return array->nd;
}

/* return the type of an array */
NumarrayType NI_GetArrayType(PyArrayObject *array)
{
  assert(array != NULL);
  return array->descr->type_num;
}

/* return the dimensions of an array */
void NI_GetArrayDimensions(PyArrayObject *array, int* dimensions)
{
  int ii;
  assert(array != NULL);
  for(ii = 0; ii < array->nd; ii++)
    dimensions[ii] = array->dimensions[ii];
}

/* return the strides of an array */
void NI_GetArrayStrides(PyArrayObject *array, int* strides)
{
  int ii;
  assert(array != NULL);
  for(ii = 0; ii < array->nd; ii++)
    strides[ii] = array->strides[ii];
}

/* return the data of an array */
char* NI_GetArrayData(PyArrayObject *array)
{
  assert(array != NULL);
  return (char*)NA_OFFSETDATA(array);
}

/* test for equal shape */
int NI_ShapeEqual(PyArrayObject *array1, PyArrayObject *array2)
{
  return NA_ShapeEqual(array1, array2);
}

/* Check the properties of an array */
int NI_CheckArray(PyArrayObject *output, NumarrayType type,
                  int rank, int *dimensions)
{
  int ii;
  assert (output != NULL);
  assert(dimensions != NULL);

  if (type != tAny && output->descr->type_num != type) {
    PyErr_SetString(PyExc_RuntimeError, "output type incorrect");
    return 0;
  }
  if (output->nd != rank) {
    PyErr_SetString(PyExc_RuntimeError, "output rank incorrect");
    return 0;
  }
  for(ii = 0; ii < rank; ii++) {
    if (output->dimensions[ii] != dimensions[ii]) {
      PyErr_SetString(PyExc_RuntimeError, "output dimensions incorrect");
      return 0;
    }
  }
  return 1;
}

/* Convert an input array of any type, not necessarily contiguous */
static int 
NI_ObjectToOptionalInputArray(PyObject *object, PyArrayObject **array)
{
  if (object == Py_None) {
    *array = NULL;
    return 1;
  } else {
    *array = NA_InputArray(object, tAny, NUM_ALIGNED|NUM_NOTSWAPPED);
    return *array ? 1 : 0;
  }
}

/* Convert an input/output array of any type, not necessarily contiguous */
static int 
NI_ObjectToIoArray(PyObject *object, PyArrayObject **array)
{
  *array = NA_IoArray(object, tAny, NUM_ALIGNED|NUM_NOTSWAPPED);
  return *array ? 1 : 0;
}

/* Convert an output array of any type, not necessarily contiguous */
static int 
NI_ObjectToOptionalOutputArray(PyObject *object, PyArrayObject **array)
{
  if (object == Py_None) {
    *array = NULL;
    return 1;
  } else {
    *array = NA_OutputArray(object, tAny, NUM_ALIGNED|NUM_NOTSWAPPED);
    return *array ? 1 : 0;
  }
}

/* Convert a contiguous Bool array */
static int 
NI_ObjectToContiguousBool(PyObject *object, PyArrayObject **array)
{
  *array = NA_InputArray(object, tBool, NUM_C_ARRAY);
  return *array ? 1 : 0;
}

/* Convert a contiguous double array */
static int 
NI_ObjectToContiguousDoubleArray(PyObject *object, PyArrayObject **array)
{
  *array = NA_InputArray(object, PyArray_DOUBLE, NUM_C_ARRAY);
  return *array ? 1 : 0;
}

/* Convert a contiguous int array */
static int 
NI_ObjectToContiguousIntArray(PyObject *object, PyArrayObject **array)
{
  *array = NA_InputArray(object, PyArray_INT, NUM_C_ARRAY);
  return *array ? 1 : 0;
}

static PyObject *Py_MinimumMaximumFilter1D(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL;
  int axis, filter_size, mode, shift, minimum;
  double cvalue;
  
  if (!PyArg_ParseTuple(args, "O&iiO&idii", NI_ObjectToInputArray, &input, 
                  &filter_size, &axis, NI_ObjectToOutputArray, &output,
                  &mode, &cvalue, &shift, &minimum))
    goto exit;
  if (!NI_MinimumMaximumFilter1D(input, filter_size, axis, output,
                              (NI_ExtendMode)mode, cvalue, shift, minimum))
    goto exit;
exit:
  Py_XDECREF(input);
  Py_XDECREF(output);
  return PyErr_Occurred() ? NULL : Py_BuildValue("");
}


static PyObject *Py_MinimumMaximumFilter(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL, *footprint = NULL;
  PyArrayObject *structure_values = NULL, *shifts = NULL;
  PyObject *structure_values_object = NULL;
  int mode, minimum;
  double cvalue;

  if (!PyArg_ParseTuple(args, "O&O&OO&idO&i", NI_ObjectToInputArray,
        &input, NI_ObjectToContiguousBool, &footprint,
        &structure_values_object, NI_ObjectToOutputArray, &output, &mode,
        &cvalue, NI_ObjectToContiguousIntArray, &shifts, &minimum))
    goto exit;

  if (structure_values_object != Py_None) {
    structure_values = NA_InputArray(structure_values_object,
                                     tFloat64, NUM_C_ARRAY);
    if (!structure_values) {
      PyErr_SetString(PyExc_RuntimeError, 
                      "cannot convert structure values");
      goto exit;
    }
  }

  if (!NI_MinimumMaximumFilter(input, footprint, structure_values, output,
     (NI_ExtendMode)mode, cvalue, (int*)NI_GetArrayData(shifts), minimum))
    goto exit;

 exit:
  Py_XDECREF(input);
  Py_XDECREF(shifts);
  Py_XDECREF(footprint);
  Py_XDECREF(output);
  return PyErr_Occurred() ? NULL : Py_BuildValue("");
}

static void _FreeCoordinateList(void* ptr) 
{
  NI_FreeCoordinateList((NI_CoordinateList*)ptr);
}

static PyObject *Py_BinaryErosion(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL, *strct = NULL;
  PyArrayObject *mask = NULL, *shifts = NULL;
  PyObject *cobj = NULL;
  int border_value, invert, center_is_true;
  int changed = 0, return_coordinates;
  NI_CoordinateList *coordinate_list = NULL;

  if (!PyArg_ParseTuple(args, "O&O&O&O&iO&iii", NI_ObjectToInputArray,
          &input, NI_ObjectToContiguousBool, &strct,
          NI_ObjectToOptionalInputArray, &mask,
          NI_ObjectToOutputArray, &output, &border_value, 
          NI_ObjectToContiguousIntArray, &shifts,  &invert,
          &center_is_true, &return_coordinates))
    goto exit;
    
  if (!NI_BinaryErosion(input, strct, mask, output, border_value,
          (int*)NI_GetArrayData(shifts), invert, center_is_true,
          &changed, return_coordinates ? &coordinate_list : NULL))
    goto exit;

  if (return_coordinates) {
    cobj = PyCObject_FromVoidPtr(coordinate_list, _FreeCoordinateList);
  }

exit:
  Py_XDECREF(input);
  Py_XDECREF(strct);
  Py_XDECREF(mask);
  Py_XDECREF(shifts);
  Py_XDECREF(output);
  if (PyErr_Occurred()) {
    Py_XDECREF(cobj);
    return NULL;
  } else {
    if (return_coordinates) {
      return Py_BuildValue("iN", changed, cobj);
    } else {
      return Py_BuildValue("i", changed);
    }
  }
}

static PyObject *Py_BinaryErosion2(PyObject *obj, PyObject *args)
{
  PyArrayObject *array = NULL, *strct = NULL;
  PyArrayObject *shifts = NULL, *mask = NULL;
  PyObject *cobj = NULL;
  int invert, niter;

  if (!PyArg_ParseTuple(args, "O&O&O&iO&iO", NI_ObjectToIoArray, &array, 
          NI_ObjectToContiguousBool, &strct, NI_ObjectToOptionalInputArray, 
          &mask, &niter, NI_ObjectToContiguousIntArray, &shifts, &invert,
          &cobj))
    goto exit;
    
  if (PyCObject_Check(cobj)) {
    NI_CoordinateList *cobj_data = PyCObject_AsVoidPtr(cobj);
    if (!NI_BinaryErosion2(array, strct, mask, niter, 
                        (int*)NI_GetArrayData(shifts), invert, &cobj_data))
      goto exit;
  } else {
    PyErr_SetString(PyExc_RuntimeError, "cannot convert CObject");
    goto exit;
  }

exit:
  Py_XDECREF(array);
  Py_XDECREF(strct);
  Py_XDECREF(mask);
  Py_XDECREF(shifts);
  return PyErr_Occurred() ? NULL : Py_BuildValue("");
}


static PyObject *Py_DistanceTransformOnePass(PyObject *obj, PyObject *args)
{
  PyArrayObject *strct = NULL, *distances = NULL, *features = NULL;

  if (!PyArg_ParseTuple(args, "O&O&O&", NI_ObjectToInputArray, &strct,
                        NI_ObjectToIoArray, &distances,
                        NI_ObjectToOptionalOutputArray, &features))
    goto exit;
    
  if (!NI_DistanceTransformOnePass(strct, distances, features))
    goto exit;

exit:
  Py_XDECREF(strct); 
  Py_XDECREF(distances); 
  Py_XDECREF(features); 
  return PyErr_Occurred() ? NULL : Py_BuildValue("");
}

static PyObject *Py_DistanceTransformBruteForce(PyObject *obj,
                                                PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL, *features = NULL;
  PyArrayObject *sampling = NULL;
  int metric, len = 0;
  PyObject *sampling_object = NULL;

  if (!PyArg_ParseTuple(args, "O&iOO&O&", NI_ObjectToInputArray, &input,
          &metric, &sampling_object, NI_ObjectToOptionalOutputArray,
          &output, NI_ObjectToOptionalOutputArray, &features))
    goto exit;
  if (sampling_object != Py_None) {
    if (!NI_ObjectToContiguousDoubleArray(sampling_object, &sampling))
      goto exit;
    len = NI_GetArraySize(sampling);
    if (len != input->nd) {
      PyErr_SetString(PyExc_RuntimeError, "sampling parameter incorrect");
      goto exit;
    }
  }  
  if (!NI_DistanceTransformBruteForce(input, metric,
          sampling ? (double*)NI_GetArrayData(sampling) : NULL, 
          output, features))
    goto exit;
exit:
  Py_XDECREF(input);
  Py_XDECREF(sampling);
  Py_XDECREF(output);
  Py_XDECREF(features);
  return PyErr_Occurred() ? NULL : Py_BuildValue("");
}

static PyObject *Py_EuclideanFeatureTransform(PyObject *obj,
                                              PyObject *args)
{
  PyArrayObject *input = NULL, *features = NULL, *sampling = NULL;
  PyObject *sampling_object = NULL;
  int len;

  if (!PyArg_ParseTuple(args, "O&OO&", NI_ObjectToInputArray, &input, 
                      &sampling_object, NI_ObjectToOutputArray, &features))
    goto exit;
  if (sampling_object != Py_None) {
    if (!NI_ObjectToContiguousDoubleArray(sampling_object, &sampling))
      goto exit;
    len = NI_GetArraySize(sampling);
    if (len != input->nd) {
      PyErr_SetString(PyExc_RuntimeError, "sampling parameter incorrect");
      goto exit;
    }
  }  
  if (!NI_EuclideanFeatureTransform(input, 
      sampling ? (double*)NI_GetArrayData(sampling) : NULL, features))
    goto exit;
exit:
  Py_XDECREF(input);
  Py_XDECREF(sampling);
  Py_XDECREF(features);
  return PyErr_Occurred() ? NULL : Py_BuildValue("");
}

static PyObject *Py_WatershedIFT(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL, *markers = NULL;
  PyArrayObject *strct = NULL;

  if (!PyArg_ParseTuple(args, "O&O&O&O&", NI_ObjectToInputArray, &input, 
          NI_ObjectToInputArray, &markers, NI_ObjectToContiguousBool,
          &strct, NI_ObjectToOutputArray, &output))
    goto exit;
    
  if (!NI_WatershedIFT(input, markers, strct, output))
    goto exit;

exit:
  Py_XDECREF(input);
  Py_XDECREF(markers);
  Py_XDECREF(strct);
  Py_XDECREF(output);
  return PyErr_Occurred() ? NULL : Py_BuildValue("");
}

static PyObject *Py_Label(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL, *strct = NULL;
  int max_label;

  if (!PyArg_ParseTuple(args, "O&O&O&", NI_ObjectToInputArray, &input, 
          NI_ObjectToContiguousBool, &strct, NI_ObjectToOutputArray,
          &output))
    goto exit;
    
  if (!NI_Label(input, strct, &max_label, output))
    goto exit;

exit:
  Py_XDECREF(input);
  Py_XDECREF(strct);
  Py_XDECREF(output);
  return PyErr_Occurred() ? NULL : Py_BuildValue("i", max_label);
}

static PyObject *Py_FindObjects(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL;
  PyObject *result = NULL, *tuple = NULL, *start = NULL, *end = NULL;
  PyObject *slc = NULL;
  int ii, jj, max_label, *regions = NULL;

  if (!PyArg_ParseTuple(args, "O&i", NI_ObjectToInputArray, &input,
                        &max_label))
    goto exit;
    
  if (max_label < 0)
    max_label = 0;
  if (max_label > 0) {
    if (input->nd > 0) {
      regions = (int*)malloc(2 * max_label * input->nd * sizeof(int));
    } else {
      regions = (int*)malloc(max_label * sizeof(int));
    }
    if (!regions) {
      PyErr_NoMemory();
      goto exit;
    }
  }
   
  if (!NI_FindObjects(input, max_label, regions))
    goto exit;

  result = PyList_New(max_label);
  if (!result) {
    PyErr_NoMemory();
    goto exit;
  }

  for(ii = 0; ii < max_label; ii++) {
    int idx = input->nd > 0 ? 2 * input->nd * ii : ii;
    if (regions[idx] >= 0) {
      PyObject *tuple = PyTuple_New(input->nd);
      if (!tuple) {
        PyErr_NoMemory();
        goto exit;
      }
      for(jj = 0; jj < input->nd; jj++) {
        start = PyInt_FromLong(regions[idx + jj]);
        end = PyInt_FromLong(regions[idx + jj + input->nd]);
        if (!start || !end) {
          PyErr_NoMemory();
          goto exit;
        }
        slc = PySlice_New(start, end, NULL);
        if (!slc) {
          PyErr_NoMemory();
          goto exit;
        }
        Py_XDECREF(start);
        Py_XDECREF(end);
        start = end = NULL;
        PyTuple_SetItem(tuple, jj, slc);
        slc = NULL;
      }
      PyList_SetItem(result, ii, tuple);
      tuple = NULL;
    } else {
      Py_INCREF(Py_None);
      PyList_SetItem(result, ii, Py_None);
    }
  }

  Py_INCREF(result);

 exit:
  Py_XDECREF(input);
  Py_XDECREF(result);
  Py_XDECREF(tuple);
  Py_XDECREF(start);
  Py_XDECREF(end);
  Py_XDECREF(slc);
  if (regions)
    free(regions);
  if (PyErr_Occurred()) {
    Py_XDECREF(result);
    return NULL;
  } else {
    return result;
  }
}

static PyObject *Py_SplineFilter1D(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL;
  int axis, order;
  
  if (!PyArg_ParseTuple(args, "O&iiO&", NI_ObjectToInputArray, &input,
          &order, &axis, NI_ObjectToOutputArray, &output))
    goto exit;
    
  if (!NI_SplineFilter1D(input, order, axis, output))
    goto exit;

exit:
  Py_XDECREF(input);
  Py_XDECREF(output);
  return PyErr_Occurred() ? NULL : Py_BuildValue("");
}


static PyObject *Py_AffineTransform(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL, *shift = NULL;
  PyArrayObject *matrix = NULL;
  int mode, order, ii;
  double cval, *data = NULL, *p;
  
  if (!PyArg_ParseTuple(args, "O&O&O&O&iid", NI_ObjectToInputArray, 
          &input, NI_ObjectToContiguousDoubleArray, &matrix,
          NI_ObjectToContiguousDoubleArray, &shift, NI_ObjectToOutputArray,
          &output, &order, &mode, &cval))
    goto exit;

  if (NI_GetArraySize(shift) != input->nd) {
    PyErr_SetString(PyExc_RuntimeError, "shift dimensions not correct");
    goto exit;
  }

  if (matrix->nd == 1) {
    if (matrix->dimensions[0] != input->nd) {
      PyErr_SetString(PyExc_RuntimeError, "matrix dimensions not correct");
      goto exit;
    }
    data = (double*)malloc(input->nd * sizeof(double));
    if (!data) {
      PyErr_NoMemory();
      goto exit;
    }
    p = (double*)NI_GetArrayData(matrix);
    for(ii = 0; ii < input->nd; ii++)
      data[ii] = (double)p[ii];
    if (!NI_ZoomShift(input, data, (double*)NI_GetArrayData(shift), 
                      output, order, (NI_ExtendMode)mode, cval))
      goto exit;
  } else {
    if (matrix->dimensions[0] != input->nd ||
        matrix->dimensions[1] != output->nd) {
      PyErr_SetString(PyExc_RuntimeError, "matrix dimensions not correct");
      goto exit;
    }
    data = (double*)malloc(input->nd * output->nd * sizeof(double));
    if (!data) {
      PyErr_NoMemory();
      goto exit;
    }
    p = (double*)NI_GetArrayData(matrix);
    for(ii = 0; ii < input->nd * output->nd; ii++)
      data[ii] = (double)p[ii];
    if (!NI_AffineTransform(input, data, (double*)NI_GetArrayData(shift),
                            output, order, (NI_ExtendMode)mode, cval))
      goto exit;
  }

 exit:
  Py_XDECREF(input);
  Py_XDECREF(shift);
  Py_XDECREF(matrix);
  if (data)
    free(data);
  Py_XDECREF(output);
  return PyErr_Occurred() ? NULL : Py_BuildValue("");
}

static int Py_Map(int *ocoor, double* icoor, int orank, int irank, 
                  void *data)
{
  PyObject *coors = NULL, *rets = NULL, *args = NULL;
  int ii, len = 1;
  NI_PythonCallbackData *cbdata = (NI_PythonCallbackData*)data;

  coors = PyTuple_New(orank);
  if (!coors)
    goto exit;
  for(ii = 0; ii < orank; ii++) {
    PyTuple_SetItem(coors, ii, PyInt_FromLong(ocoor[ii]));
    if (PyErr_Occurred())
      goto exit;
  }
  if (cbdata->extra_arguments && PyTuple_Check(cbdata->extra_arguments))
    len += PyTuple_Size(cbdata->extra_arguments);
  args = PyTuple_New(len);
  if (!args)
    goto exit;
  Py_INCREF(coors);
  PyTuple_SET_ITEM(args, 0, (PyObject*)coors);
  if (cbdata->extra_arguments && PyTuple_Check(cbdata->extra_arguments)) {
    for(ii = 1; ii < len; ii++) {
      PyObject *tmp = PyTuple_GET_ITEM(cbdata->extra_arguments, ii - 1);
      Py_INCREF(tmp);
      PyTuple_SET_ITEM(args, ii, tmp);
    }
  }
  rets = PyObject_Call(cbdata->function, args, cbdata->extra_keywords);
  if (!rets)
    goto exit;
  for(ii = 0; ii < irank; ii++) {
    icoor[ii] = PyFloat_AsDouble(PyTuple_GetItem(rets, ii));
    if (PyErr_Occurred())
      goto exit;
  }
 exit:
  Py_XDECREF(coors);
  Py_XDECREF(rets);
  Py_XDECREF(args);
  return PyErr_Occurred() ? 0 : 1;
}


static PyObject *Py_GeometricTransform(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL;
  PyObject *mapping = NULL;
  PyObject *extra_arguments = NULL, *extra_keywords = NULL;
  int mode, order;
  double cval;
  void *func = Py_Map, *data = NULL;
  NI_PythonCallbackData cbdata;
  
  if (!PyArg_ParseTuple(args, "O&OO&iidOO", NI_ObjectToInputArray,
          &input, &mapping, NI_ObjectToOutputArray, &output, &order, &mode,
          &cval, &extra_arguments, &extra_keywords))
    goto exit;

  if (PyCObject_Check(mapping)) {
    func = PyCObject_AsVoidPtr(mapping);
    data = PyCObject_GetDesc(mapping);
  } else if (PyCallable_Check(mapping)) {
    if (extra_arguments && !PyTuple_Check(extra_arguments)) {
      PyErr_SetString(PyExc_RuntimeError,
                      "extra_arguments must be a tuple");
      goto exit;
    }
    if (extra_keywords && !PyDict_Check(extra_keywords)) {
      PyErr_SetString(PyExc_RuntimeError,
                      "extra_keywords must be a dictionary");
      goto exit;
    }
    cbdata.function = mapping;
    cbdata.extra_arguments = extra_arguments;
    cbdata.extra_keywords = extra_keywords;
    data = (void*)&cbdata;
  } else {
    PyErr_SetString(PyExc_RuntimeError,
                    "mapping parameter is not callable");
    goto exit;
  }
    
  if (!NI_GeometricTransform(input, func, data, output, order,
                             (NI_ExtendMode)mode, cval))
    goto exit;

 exit:
  Py_XDECREF(input);
  Py_XDECREF(output);
  return PyErr_Occurred() ? NULL : Py_BuildValue("");
}


static PyObject *Py_MapCoordinates(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL, *coordinates = NULL;
  int mode, order;
  double cval;
  
  if (!PyArg_ParseTuple(args, "O&O&O&iid", NI_ObjectToInputArray, 
          &input, NI_ObjectToInputArray, &coordinates, 
          NI_ObjectToOutputArray, &output, &order, &mode, &cval))
    goto exit;
  
  if (!NI_MapCoordinates(input, coordinates, output, order,
                         (NI_ExtendMode)mode, cval))
    goto exit;

exit:
  Py_XDECREF(input);
  Py_XDECREF(coordinates);
  Py_XDECREF(output);
  return PyErr_Occurred() ? NULL : Py_BuildValue("");
}


static PyObject *Py_Zoom(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL, *zoom = NULL;
  int mode, order;
  double cval;
  
  if (!PyArg_ParseTuple(args, "O&O&O&iid", NI_ObjectToInputArray,
          &input, NI_ObjectToContiguousDoubleArray, &zoom, 
          NI_ObjectToInputArray, &output, &order, &mode, &cval))
    goto exit;
    
  if (NI_GetArraySize(zoom) != input->nd) {
    PyErr_SetString(PyExc_RuntimeError, "number of zooms not correct");
    goto exit;
  }
    
  if (!NI_ZoomShift(input, (double*)NI_GetArrayData(zoom), NULL,
                    output, order, (NI_ExtendMode)mode, cval))
    goto exit;

 exit:
  Py_XDECREF(input);
  Py_XDECREF(zoom);
  Py_XDECREF(output);
  return PyErr_Occurred() ? NULL : Py_BuildValue("");
}


static PyObject *Py_Shift(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL, *shift = NULL;
  int mode, order;
  double cval;
  
  if (!PyArg_ParseTuple(args, "O&O&O&iid", NI_ObjectToInputArray, &input, 
          NI_ObjectToContiguousDoubleArray, &shift, NI_ObjectToOutputArray,
          &output, &order, &mode, &cval))
    goto exit;

  if (NI_GetArraySize(shift) != input->nd) {
    PyErr_SetString(PyExc_RuntimeError, "number of shifts not correct");
    goto exit;
  }

  if (!NI_ZoomShift(input, NULL, (double*)NI_GetArrayData(shift), 
                    output, order, (NI_ExtendMode)mode, cval))
    goto exit;

 exit:
  Py_XDECREF(input);
  Py_XDECREF(shift);
  Py_XDECREF(output);
  return PyErr_Occurred() ? NULL : Py_BuildValue("");
}

static int _NI_GetIndices(PyObject* indices_object, int** result_indices,
                          int* min_label, int* max_label, int* n_results)
{
  PyArrayObject *indices_array = NULL;
  int *indices = NULL, n_indices, ii;

  if (indices_object == Py_None) {
    *min_label = -1;
    *n_results = 1;
  } else {
    if (!NI_ObjectToContiguousIntArray(indices_object, &indices_array))
      goto exit;
    indices = (int*)NI_GetArrayData(indices_array);
    n_indices = NI_GetArraySize(indices_array);
    if (n_indices < 1) {
      PyErr_SetString(PyExc_RuntimeError, "no correct indices provided");
      goto exit;
    } else {
      *min_label = *max_label = indices[0];
      if (*min_label < 0) {
        PyErr_SetString(PyExc_RuntimeError,
                        "negative indices not allowed");
        goto exit;
      }
      for(ii = 1; ii < n_indices; ii++) {
        if (indices[ii] < 0) {
          PyErr_SetString(PyExc_RuntimeError,
                          "negative indices not allowed");
          goto exit;
        }
        if (indices[ii] < *min_label)
          *min_label = indices[ii];
        if (indices[ii] > *max_label)
          *max_label = indices[ii];
      }
      *result_indices = (int*)malloc((*max_label - *min_label + 1) * 
                                     sizeof(int));
      if (!*result_indices) {
        PyErr_NoMemory();
        goto exit;
      }
      for(ii = 0; ii < *max_label - *min_label + 1; ii++)
        (*result_indices)[ii] = -1;
      *n_results = 0;
      for(ii = 0; ii < n_indices; ii++) {
        if ((*result_indices)[indices[ii] - *min_label] >= 0) {
          PyErr_SetString(PyExc_RuntimeError, "duplicate index");
          goto exit;
        }
        (*result_indices)[indices[ii] - *min_label] = ii;
        ++(*n_results);
      }
    }
  }
  
 exit:
  Py_XDECREF(indices_array);
  return PyErr_Occurred() == NULL;
}


static int _NI_GetLabels(PyObject* labels_object, PyArrayObject** labels)
{
  if (labels_object != Py_None) {
    *labels = NA_InputArray(labels_object, tAny, 
                            NUM_ALIGNED|NUM_NOTSWAPPED);
    if (!*labels) {
      PyErr_SetString(PyExc_RuntimeError, "cannot convert labels");
      return 0;
    }
  } else {
    *labels = NULL;
  }
  return 1;
}

PyObject* _NI_BuildMeasurementResultArrayObject(int n_results,
                                                PyArrayObject** values)
{
  PyObject *result = NULL;
  if (n_results > 1) {
    result = PyList_New(n_results);
    if (result) {
      int ii;
      for(ii = 0; ii < n_results; ii++) {
        PyList_SET_ITEM(result, ii, (PyObject*)values[ii]);
        Py_XINCREF(values[ii]);
      }
    }
  } else {
    result = (PyObject*)values[0];
    Py_XINCREF(values[0]);
  }
  return result;
}


PyObject* _NI_BuildMeasurementResultDouble(int n_results, double* values)
{
  PyObject *result = NULL;
  if (n_results > 1) {
    result = PyList_New(n_results);
    if (result) {
      int ii;
      for(ii = 0; ii < n_results; ii++) {
        PyObject* val = PyFloat_FromDouble(values[ii]);
        if (!val) {
          Py_XDECREF(result);
          return NULL;
        }
        PyList_SET_ITEM(result, ii, val);
      }
    }
  } else {
    result = Py_BuildValue("d", values[0]);
  }
  return result;
}


PyObject* _NI_BuildMeasurementResultDoubleTuple(int n_results,
                                            int tuple_size, double* values)
{
  PyObject *result = NULL;
  int ii, jj;

  if (n_results > 1) {
    result = PyList_New(n_results);
    if (result) {
      for(ii = 0; ii < n_results; ii++) {
        PyObject* val = PyTuple_New(tuple_size);
        if (!val) {
          Py_XDECREF(result);
          return NULL;
        }
        for(jj = 0; jj < tuple_size; jj++) {
          int idx = jj + ii * tuple_size;
          PyTuple_SetItem(val, jj, PyFloat_FromDouble(values[idx]));
          if (PyErr_Occurred()) {
            Py_XDECREF(result);
            return NULL;
          }
        }
        PyList_SET_ITEM(result, ii, val);
      }
    }
  } else {
    result = PyTuple_New(tuple_size);
    if (result) {
      for(ii = 0; ii < tuple_size; ii++) {
        PyTuple_SetItem(result, ii, PyFloat_FromDouble(values[ii]));
        if (PyErr_Occurred()) {
          Py_XDECREF(result);
          return NULL;
        }
      }
    }
  }
  return result;
}


PyObject* _NI_BuildMeasurementResultInt(int n_results, int* values)
{
  PyObject *result = NULL;
  if (n_results > 1) {
    result = PyList_New(n_results);
    if (result) {
      int ii;
      for(ii = 0; ii < n_results; ii++) {
        PyObject* val = PyInt_FromLong(values[ii]);
        if (!val) {
          Py_XDECREF(result);
          return NULL;
        }
        PyList_SET_ITEM(result, ii, val);
      }
    }
  } else {
    result = Py_BuildValue("i", values[0]);
  }
  return result;
}


static PyObject *Py_Sum(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *labels = NULL;
  PyObject *labels_object = NULL, *indices_object, *result = NULL;
  double *sum = NULL;
  int min_label, max_label, *result_indices = NULL, n_results;
  
  if (!PyArg_ParseTuple(args, "O&OO", NI_ObjectToInputArray, &input, 
                        &labels_object, &indices_object))
    goto exit;

  if (!_NI_GetIndices(indices_object, &result_indices, &min_label,
                      &max_label, &n_results))
    goto exit;

  if (!_NI_GetLabels(labels_object, &labels))
    goto exit;

  sum = (double*)malloc(n_results * sizeof(double));
  if (!sum) {
    PyErr_NoMemory();
    goto exit;
  }
  
  if (!NI_Statistics(input, labels, min_label, max_label, result_indices, 
                     n_results, sum, NULL, NULL, NULL, NULL, NULL, NULL))
    goto exit;

  result = _NI_BuildMeasurementResultDouble(n_results, sum);

 exit:
  Py_XDECREF(input);
  Py_XDECREF(labels);
  if (result_indices)
    free(result_indices);
  if (sum)
    free(sum);
  return result;
}


static PyObject *Py_Mean(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *labels = NULL;
  PyObject *labels_object = NULL, *indices_object, *result = NULL;
  double *sum = NULL;
  int min_label, max_label, *result_indices = NULL, n_results;
  int ii, *total = NULL;
  
  if (!PyArg_ParseTuple(args, "O&OO", NI_ObjectToInputArray, &input, 
                        &labels_object, &indices_object))
    goto exit;

  if (!_NI_GetIndices(indices_object, &result_indices, &min_label,
                      &max_label, &n_results))
    goto exit;

  if (!_NI_GetLabels(labels_object, &labels))
    goto exit;

  sum = (double*)malloc(n_results * sizeof(double));
  total = (int*)malloc(n_results * sizeof(int));
  if (!sum || !total) {
    PyErr_NoMemory();
    goto exit;
  }
  
  if (!NI_Statistics(input, labels, min_label, max_label, result_indices, 
                     n_results, sum, total, NULL, NULL, NULL, NULL, NULL))
    goto exit;
  for(ii = 0; ii < n_results; ii++)
    sum[ii] = total[ii] > 0 ? sum[ii] / total[ii] : 0.0;

  result = _NI_BuildMeasurementResultDouble(n_results, sum);

 exit:
  Py_XDECREF(input);
  Py_XDECREF(labels);
  if (result_indices)
    free(result_indices);
  if (sum)
    free(sum);
  if (total)
    free(total);
  return result;
}


static PyObject *Py_Variance(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *labels = NULL;
  PyObject *labels_object = NULL, *indices_object, *result = NULL;
  double *sum = NULL, *variance = NULL;
  int min_label, max_label, *result_indices = NULL, n_results;
  int *total = NULL;
  
  if (!PyArg_ParseTuple(args, "O&OO", NI_ObjectToInputArray, &input, 
                        &labels_object, &indices_object))
    goto exit;

  if (!_NI_GetIndices(indices_object, &result_indices, &min_label,
                      &max_label, &n_results))
    goto exit;

  if (!_NI_GetLabels(labels_object, &labels))
    goto exit;

  sum = (double*)malloc(n_results * sizeof(double));
  total = (int*)malloc(n_results * sizeof(int));
  variance = (double*)malloc(n_results * sizeof(double));
  if (!sum || !total || !variance) {
    PyErr_NoMemory();
    goto exit;
  }
  
  if (!NI_Statistics(input, labels, min_label, max_label, result_indices, 
                     n_results, sum, total, variance, NULL, NULL, NULL,
                    NULL))
    goto exit;

  result = _NI_BuildMeasurementResultDouble(n_results, variance);

 exit:
  Py_XDECREF(input);
  Py_XDECREF(labels);
  if (result_indices)
    free(result_indices);
  if (sum)
    free(sum);
  if (total)
    free(total);
  if (variance)
    free(variance);
  return result;
}

static PyObject *Py_Minimum(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *labels = NULL;
  PyObject *labels_object = NULL, *indices_object, *result = NULL;
  double *minimum = NULL;
  int min_label, max_label, *result_indices = NULL, n_results;
  
  if (!PyArg_ParseTuple(args, "O&OO", NI_ObjectToInputArray, &input, 
                        &labels_object, &indices_object))
    goto exit;

  if (!_NI_GetIndices(indices_object, &result_indices, &min_label,
                      &max_label, &n_results))
    goto exit;

  if (!_NI_GetLabels(labels_object, &labels))
    goto exit;

  minimum = (double*)malloc(n_results * sizeof(double));
  if (!minimum) {
    PyErr_NoMemory();
    goto exit;
  }
  
  if (!NI_Statistics(input, labels, min_label, max_label, result_indices, 
                     n_results, NULL, NULL, NULL, minimum, NULL, NULL,
                     NULL))
    goto exit;

  result = _NI_BuildMeasurementResultDouble(n_results, minimum);

 exit:
  Py_XDECREF(input);
  Py_XDECREF(labels);
  if (result_indices)
    free(result_indices);
  if (minimum)
    free(minimum);
  return result;
}


static PyObject *Py_Maximum(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *labels = NULL;
  PyObject *labels_object = NULL, *indices_object, *result = NULL;
  double *maximum = NULL;
  int min_label, max_label, *result_indices = NULL, n_results;
  
  if (!PyArg_ParseTuple(args, "O&OO", NI_ObjectToInputArray, &input, 
                        &labels_object, &indices_object))
    goto exit;

  if (!_NI_GetIndices(indices_object, &result_indices, &min_label,
                      &max_label, &n_results))
    goto exit;

  if (!_NI_GetLabels(labels_object, &labels))
    goto exit;

  maximum = (double*)malloc(n_results * sizeof(double));
  if (!maximum) {
    PyErr_NoMemory();
    goto exit;
  }
  
  if (!NI_Statistics(input, labels, min_label, max_label, result_indices, 
                     n_results, NULL, NULL, NULL, NULL, maximum, NULL,
                     NULL))
    goto exit;

  result = _NI_BuildMeasurementResultDouble(n_results, maximum);

 exit:
  Py_XDECREF(input);
  Py_XDECREF(labels);
  if (result_indices)
    free(result_indices);
  if (maximum)
    free(maximum);
  return result;
}


static PyObject *Py_MinimumPosition(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *labels = NULL;
  PyObject *labels_object = NULL, *indices_object, *result = NULL;
  double *minimum = NULL;
  int *minimum_position = NULL;
  int min_label, max_label, *result_indices = NULL, n_results;
  
  if (!PyArg_ParseTuple(args, "O&OO", NI_ObjectToInputArray, &input, 
                        &labels_object, &indices_object))
    goto exit;

  if (!_NI_GetIndices(indices_object, &result_indices, &min_label,
                      &max_label, &n_results))
    goto exit;

  if (!_NI_GetLabels(labels_object, &labels))
    goto exit;

  minimum = (double*)malloc(n_results * sizeof(double));
  minimum_position = (int*)malloc(n_results * sizeof(int));
  if (!minimum || !minimum_position) {
    PyErr_NoMemory();
    goto exit;
  }
  
  if (!NI_Statistics(input, labels, min_label, max_label, result_indices, 
                     n_results, NULL, NULL, NULL, minimum, NULL, 
                     minimum_position, NULL))
    goto exit;

  result = _NI_BuildMeasurementResultInt(n_results, minimum_position);

 exit:
  Py_XDECREF(input);
  Py_XDECREF(labels);
  if (result_indices)
    free(result_indices);
  if (minimum_position)
    free(minimum_position);
  if (minimum)
    free(minimum);
  return result;
}


static PyObject *Py_MaximumPosition(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *labels = NULL;
  PyObject *labels_object = NULL, *indices_object, *result = NULL;
  double *maximum = NULL;
  int *maximum_position = NULL;
  int min_label, max_label, *result_indices = NULL, n_results;
  
  if (!PyArg_ParseTuple(args, "O&OO", NI_ObjectToInputArray, &input, 
                        &labels_object, &indices_object))
    goto exit;

  if (!_NI_GetIndices(indices_object, &result_indices, &min_label,
                      &max_label, &n_results))
    goto exit;

  if (!_NI_GetLabels(labels_object, &labels))
    goto exit;

  maximum = (double*)malloc(n_results * sizeof(double));
  maximum_position = (int*)malloc(n_results * sizeof(int));
  if (!maximum || !maximum_position) {
    PyErr_NoMemory();
    goto exit;
  }
  
  if (!NI_Statistics(input, labels, min_label, max_label, result_indices, 
                     n_results, NULL, NULL, NULL, NULL, maximum, NULL,
                     maximum_position))
    goto exit;

  result = _NI_BuildMeasurementResultInt(n_results, maximum_position);

 exit:
  Py_XDECREF(input);
  Py_XDECREF(labels);
  if (result_indices)
    free(result_indices);
  if (maximum_position)
    free(maximum_position);
  if (maximum)
    free(maximum);
  return result;
}


static PyObject *Py_Extrema(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *labels = NULL;
  PyObject *labels_object = NULL, *indices_object, *result = NULL;
  PyObject *min_result = NULL, *minp_result = NULL;
  PyObject *max_result = NULL, *maxp_result = NULL;
  double *minimum = NULL, *maximum = NULL;
  int *minimum_position = NULL, *maximum_position = NULL;
  int min_label, max_label, *result_indices = NULL, n_results;
  
  if (!PyArg_ParseTuple(args, "O&OO", NI_ObjectToInputArray, &input, 
                        &labels_object, &indices_object))
    goto exit;

  if (!_NI_GetIndices(indices_object, &result_indices, &min_label,
                      &max_label, &n_results))
    goto exit;

  if (!_NI_GetLabels(labels_object, &labels))
    goto exit;

  minimum = (double*)malloc(n_results * sizeof(double));
  maximum = (double*)malloc(n_results * sizeof(double));
  minimum_position = (int*)malloc(n_results * sizeof(int));
  maximum_position = (int*)malloc(n_results * sizeof(int));
  if (!minimum || !minimum_position || !maximum || !maximum_position) {
    PyErr_NoMemory();
    goto exit;
  }
  
  if (!NI_Statistics(input, labels, min_label, max_label, result_indices, 
                     n_results, NULL, NULL, NULL, minimum, maximum, 
                     minimum_position, maximum_position))
    goto exit;

  min_result = _NI_BuildMeasurementResultDouble(n_results, minimum);
  max_result = _NI_BuildMeasurementResultDouble(n_results, maximum);
  minp_result = _NI_BuildMeasurementResultInt(n_results,
                                              minimum_position);
  maxp_result = _NI_BuildMeasurementResultInt(n_results,
                                              maximum_position);
  if (!min_result || !minp_result || !max_result || !maxp_result)
    goto exit;
    
  result = Py_BuildValue("OOOO", min_result, max_result, minp_result, 
                         maxp_result);

 exit:
  Py_XDECREF(input);
  Py_XDECREF(labels);
  Py_XDECREF(min_result);
  Py_XDECREF(minp_result);
  Py_XDECREF(max_result);
  Py_XDECREF(maxp_result);
  if (result_indices)
    free(result_indices);
  if (minimum_position)
    free(minimum_position);
  if (minimum)
    free(minimum);
  if (maximum_position)
    free(maximum_position);
  if (maximum)
    free(maximum);
  return result;
}

static PyObject *Py_CenterOfMass(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *labels = NULL;
  PyObject *labels_object = NULL, *indices_object, *result = NULL;
  double *center_of_mass = NULL;
  int min_label, max_label, *result_indices = NULL, n_results;
  
  if (!PyArg_ParseTuple(args, "O&OO", NI_ObjectToInputArray, &input, 
                        &labels_object, &indices_object))
    goto exit;

  if (!_NI_GetIndices(indices_object, &result_indices, &min_label,
                      &max_label, &n_results))
    goto exit;

  if (!_NI_GetLabels(labels_object, &labels))
    goto exit;

  center_of_mass = (double*)malloc(input->nd * n_results * 
                                   sizeof(double));
  if (!center_of_mass) {
    PyErr_NoMemory();
    goto exit;
  }
  
  if (!NI_CenterOfMass(input, labels, min_label, max_label,
                       result_indices, n_results, center_of_mass))
    goto exit;

  result = _NI_BuildMeasurementResultDoubleTuple(n_results, input->nd,
                                                 center_of_mass);

 exit:
  Py_XDECREF(input);
  Py_XDECREF(labels);
  if (result_indices)
    free(result_indices);
  if (center_of_mass)
    free(center_of_mass);
  return result;
}

static PyObject *Py_Histogram(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *labels = NULL, **histograms = NULL;
  PyObject *labels_object = NULL, *indices_object, *result = NULL;
  int min_label, max_label, *result_indices = NULL, n_results, jj, nbins;
  double min, max;
  
  if (!PyArg_ParseTuple(args, "O&ddiOO", NI_ObjectToInputArray, &input, 
                        &min, &max, &nbins, &labels_object,
                        &indices_object))
    goto exit;

  if (!_NI_GetIndices(indices_object, &result_indices, &min_label,
                      &max_label, &n_results))
    goto exit;

  if (!_NI_GetLabels(labels_object, &labels))
    goto exit;

  histograms = (PyArrayObject**)malloc(input->nd * n_results *
                                       sizeof(PyArrayObject*));
  if (!histograms) {
    PyErr_NoMemory();
    goto exit;
  }
  for(jj = 0; jj < n_results; jj++) {
    histograms[jj] = NA_NewArray(NULL, tInt32, 1, nbins);
    if (!histograms[jj]) {
      PyErr_NoMemory();
      goto exit;
    }
  }
  
  if (!NI_Histogram(input, labels, min_label, max_label, result_indices, 
                    n_results, histograms, min, max, nbins))
    goto exit;

  result = _NI_BuildMeasurementResultArrayObject(n_results, histograms);

 exit:
  Py_XDECREF(input);
  Py_XDECREF(labels);
  if (result_indices)
    free(result_indices);
  if (histograms) {
    for(jj = 0; jj < n_results; jj++) {
      Py_XDECREF(histograms[jj]);
    }
    free(histograms);
  }
  return result;
}

/*********************************************************************/
/* new stuff */
/*********************************************************************/

/* Convert an Long sequence */
static int 
NI_ObjectToLongSequence(PyObject *object, maybelong **sequence)
{
  long *pa, ii;
  PyArrayObject *array = NA_InputArray(object, PyArray_LONG, C_ARRAY);
  
  *sequence = (maybelong*)malloc(NA_elements(array) * sizeof(maybelong));
  if (!*sequence) {
    PyErr_NoMemory();
    Py_XDECREF(array);
    return 0;
  }
  pa = (long*)NA_OFFSETDATA(array);
  for(ii = 0; ii < NA_elements(array); ii++)
    (*sequence)[ii] = pa[ii];
  Py_XDECREF(array);
  return 1;
}

static PyObject *Py_Correlate1D(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL, *weights = NULL;
  int axis, mode;
  long origin;
  double cval;
  
  if (!PyArg_ParseTuple(args, "O&O&iO&idl", NI_ObjectToInputArray, &input, 
                  NI_ObjectToInputArray, &weights, &axis, 
                  NI_ObjectToOutputArray, &output, &mode, &cval, &origin))
    goto exit;
  if (!NI_Correlate1D(input, weights, axis, output,
                      (NI_ExtendMode)mode, cval, origin))
    goto exit;
exit:
  Py_XDECREF(input);
  Py_XDECREF(weights);
  Py_XDECREF(output);
  return PyErr_Occurred() ? NULL : Py_BuildValue("");
}

static PyObject *Py_Correlate(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL, *weights = NULL;
  maybelong *origin = NULL;
  int mode;
  long frame;
  double cval;
  
  if (!PyArg_ParseTuple(args, "O&O&O&idO&l", NI_ObjectToInputArray, &input, 
          NI_ObjectToInputArray, &weights, NI_ObjectToOutputArray, &output, 
          &mode, &cval, NI_ObjectToLongSequence, &origin, &frame))
    goto exit;
  if (!NI_Correlate(input, weights, output, (NI_ExtendMode)mode, cval, 
                    origin, (UInt32)frame))
    goto exit;
exit:
  Py_XDECREF(input);
  Py_XDECREF(weights);
  Py_XDECREF(output);
  if (origin)
    free(origin);
  return PyErr_Occurred() ? NULL : Py_BuildValue("");
}

static PyObject *Py_UniformFilter1D(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL;
  int axis, mode;
  long filter_size, origin;
  double cval;
  
  if (!PyArg_ParseTuple(args, "O&liO&idl", NI_ObjectToInputArray, &input, 
                    &filter_size, &axis, NI_ObjectToOutputArray, &output,
                    &mode, &cval, &origin))
    goto exit;
  if (!NI_UniformFilter1D(input, filter_size, axis, output,
                                       (NI_ExtendMode)mode, cval, origin))
    goto exit;
exit:
  Py_XDECREF(input);
  Py_XDECREF(output);
  return PyErr_Occurred() ? NULL : Py_BuildValue("");
}

static PyObject *Py_MinOrMaxFilter1D(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL;
  int axis, mode, minimum;
  long filter_size, origin;
  double cval;
  
  if (!PyArg_ParseTuple(args, "O&liO&idli", NI_ObjectToInputArray, &input, 
                    &filter_size, &axis, NI_ObjectToOutputArray, &output,
                    &mode, &cval, &origin, &minimum))
    goto exit;
  if (!NI_MinOrMaxFilter1D(input, filter_size, axis, output,
                              (NI_ExtendMode)mode, cval, origin, minimum))
    goto exit;
exit:
  Py_XDECREF(input);
  Py_XDECREF(output);
  return PyErr_Occurred() ? NULL : Py_BuildValue("");
}

static PyObject *Py_MinOrMaxFilter(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL, *footprint = NULL;
  maybelong *origin = NULL;
  int mode, minimum;
  long frame;
  double cval;
  
  if (!PyArg_ParseTuple(args, "O&O&O&idO&li", NI_ObjectToInputArray,
                    &input, NI_ObjectToInputArray, &footprint,
                    NI_ObjectToOutputArray, &output, &mode, &cval,
                    NI_ObjectToLongSequence, &origin, &frame, &minimum))
    goto exit;
  if (!NI_MinOrMaxFilter(input, footprint, output, (NI_ExtendMode)mode,
                                    cval, origin, (UInt32)frame, minimum))
    goto exit;
exit:
  Py_XDECREF(input);
  Py_XDECREF(footprint);
  Py_XDECREF(output);
  if (origin)
    free(origin);
  return PyErr_Occurred() ? NULL : Py_BuildValue("");
}

static PyObject *Py_RankFilter(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL, *footprint = NULL;
  maybelong *origin = NULL;
  int mode, rank;
  long frame;
  double cval;
  
  if (!PyArg_ParseTuple(args, "O&iO&O&idO&l", NI_ObjectToInputArray,
                    &input, &rank, NI_ObjectToInputArray, &footprint,
                    NI_ObjectToOutputArray, &output, &mode, &cval,
                    NI_ObjectToLongSequence, &origin, &frame))
    goto exit;
  if (!NI_RankFilter(input, rank, footprint, output, (NI_ExtendMode)mode,
                                            cval, origin, (UInt32)frame))
    goto exit;
exit:
  Py_XDECREF(input);
  Py_XDECREF(footprint);
  Py_XDECREF(output);
  if (origin)
    free(origin);
  return PyErr_Occurred() ? NULL : Py_BuildValue("");
}

static int Py_Filter1DFunc(double *iline, maybelong ilen,
                           double *oline, maybelong olen, void *data)
{
  PyArrayObject *py_ibuffer = NULL, *py_obuffer = NULL; 
  PyObject *rv = NULL, *args = NULL, *tmp = NULL;
  maybelong ii;
  double *po = NULL;
  NI_PythonCallbackData *cbdata = (NI_PythonCallbackData*)data;
  
  py_ibuffer = NA_NewArray(iline, PyArray_DOUBLE, 1, (int)ilen);
  py_obuffer = NA_NewArray(NULL, PyArray_DOUBLE, 1, (int)olen);
  if (!py_ibuffer || !py_obuffer)
    goto exit;
  tmp = Py_BuildValue("(OO)", py_ibuffer, py_obuffer);
  if (!tmp)
    goto exit;
  args = PySequence_Concat(tmp, cbdata->extra_arguments);
  if (!args)
    goto exit;
  rv = PyObject_Call(cbdata->function, args, cbdata->extra_keywords);
  if (!rv)
    goto exit;
  po = (double*)NA_OFFSETDATA(py_obuffer);
  for(ii = 0; ii < olen; ii++)
    oline[ii] = po[ii];
exit:
  Py_XDECREF(py_ibuffer);
  Py_XDECREF(py_obuffer);
  Py_XDECREF(rv);
  Py_XDECREF(args);
  Py_XDECREF(tmp);
  return PyErr_Occurred() ? 0 : 1;
}

static PyObject *Py_GenericFilter1D(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL;
  PyObject *fnc = NULL, *extra_arguments = NULL, *extra_keywords = NULL;
  void *func = Py_Filter1DFunc, *data = NULL;
  NI_PythonCallbackData cbdata;
  int axis, mode;
  long origin, filter_size;
  double cval;
  
  if (!PyArg_ParseTuple(args, "O&OiiO&idlOO", NI_ObjectToInputArray, 
        &input, &fnc, &filter_size, &axis, NI_ObjectToOutputArray, 
        &output, &mode, &cval, &origin, &extra_arguments, &extra_keywords))
    goto exit;
  if (!PyTuple_Check(extra_arguments)) {
    PyErr_SetString(PyExc_RuntimeError,
                    "extra_arguments must be a tuple");
    goto exit;
  }
  if (!PyDict_Check(extra_keywords)) {
    PyErr_SetString(PyExc_RuntimeError,
                    "extra_keywords must be a dictionary");
    goto exit;
  }
  if (PyCObject_Check(fnc)) {
    func = PyCObject_AsVoidPtr(fnc);
    data = PyCObject_GetDesc(fnc);
  } else if (PyCallable_Check(fnc)) {
    if (extra_arguments && !PyTuple_Check(extra_arguments)) {
      PyErr_SetString(PyExc_RuntimeError,
                      "extra_arguments must be a tuple");
    }
    if (extra_keywords && !PyDict_Check(extra_keywords)) {
      PyErr_SetString(PyExc_RuntimeError,
                      "extra_keywords must be a dictionary");
    }
    cbdata.function = fnc;
    cbdata.extra_arguments = extra_arguments;
    cbdata.extra_keywords = extra_keywords;
    data = (void*)&cbdata;
  } else {
    PyErr_SetString(PyExc_RuntimeError,
                    "function parameter is not callable");
    goto exit;
  }
  if (!NI_GenericFilter1D(input, func, data, filter_size, axis, output,
                                        (NI_ExtendMode)mode, cval, origin))
    goto exit;
exit:
  Py_XDECREF(input);
  Py_XDECREF(output);
  return PyErr_Occurred() ? NULL : Py_BuildValue("");
}

static int Py_FilterFunc(double *buffer, maybelong filter_size,
                         double *output, void *data)
{
  PyArrayObject *py_buffer = NULL;
  PyObject *rv = NULL, *args = NULL, *tmp = NULL;
  NI_PythonCallbackData *cbdata = (NI_PythonCallbackData*)data;

  py_buffer = NA_NewArray(buffer, PyArray_DOUBLE, 1, filter_size);
  if (!py_buffer)
    goto exit;
  tmp = Py_BuildValue("(O)", py_buffer);
  if (!tmp)
    goto exit;
  args = PySequence_Concat(tmp, cbdata->extra_arguments);
  if (!args)
    goto exit;
  rv = PyObject_Call(cbdata->function, args, cbdata->extra_keywords);
  if (!rv)
    goto exit;
  *output = PyFloat_AsDouble(rv);
exit:
  Py_XDECREF(py_buffer);
  Py_XDECREF(rv);
  Py_XDECREF(args);
  Py_XDECREF(tmp);
  return PyErr_Occurred() ? 0 : 1;
}

static PyObject *Py_GenericFilter(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL, *footprint = NULL;
  PyObject *fnc = NULL, *extra_arguments = NULL, *extra_keywords = NULL;
  void *func = Py_FilterFunc, *data = NULL;
  NI_PythonCallbackData cbdata;
  int mode;
  long frame;
  maybelong *origin = NULL;
  double cval;
  
  if (!PyArg_ParseTuple(args, "O&OO&O&idO&lOO", NI_ObjectToInputArray, 
                        &input, &fnc, NI_ObjectToInputArray, &footprint,
                        NI_ObjectToOutputArray, &output, &mode, &cval,
                        NI_ObjectToLongSequence, &origin, &frame,
                        &extra_arguments, &extra_keywords))
    goto exit;
  if (!PyTuple_Check(extra_arguments)) {
    PyErr_SetString(PyExc_RuntimeError,
                    "extra_arguments must be a tuple");
    goto exit;
  }
  if (!PyDict_Check(extra_keywords)) {
    PyErr_SetString(PyExc_RuntimeError,
                    "extra_keywords must be a dictionary");
    goto exit;
  }
  if (PyCObject_Check(fnc)) {
    func = PyCObject_AsVoidPtr(fnc);
    data = PyCObject_GetDesc(fnc);
  } else if (PyCallable_Check(fnc)) {
    if (extra_arguments && !PyTuple_Check(extra_arguments)) {
      PyErr_SetString(PyExc_RuntimeError,
                      "extra_arguments must be a tuple");
    }
    if (extra_keywords && !PyDict_Check(extra_keywords)) {
      PyErr_SetString(PyExc_RuntimeError,
                      "extra_keywords must be a dictionary");
    }
    cbdata.function = fnc;
    cbdata.extra_arguments = extra_arguments;
    cbdata.extra_keywords = extra_keywords;
    data = (void*)&cbdata;
  } else {
    PyErr_SetString(PyExc_RuntimeError,
                    "function parameter is not callable");
    goto exit;
  }
  if (!NI_GenericFilter(input, func, data, footprint, output,
                                (NI_ExtendMode)mode, cval, origin, frame))
    goto exit;
exit:
  Py_XDECREF(input);
  Py_XDECREF(output);
  Py_XDECREF(footprint);
  if (origin)
    free(origin);
  return PyErr_Occurred() ? NULL : Py_BuildValue("");
}

static PyObject *Py_FourierFilter(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL, *parameters = NULL;
  int axis, filter_type;
  long n, frame;
  
  if (!PyArg_ParseTuple(args, "O&O&liO&li", NI_ObjectToInputArray, &input, 
                    NI_ObjectToInputArray, &parameters, &n, &axis,
                    NI_ObjectToOutputArray, &output, &frame, &filter_type))
    goto exit;   
                
  if (!NI_FourierFilter(input, parameters, n, axis, output,
                        (UInt32)frame, filter_type))
    goto exit;
    
exit:
  Py_XDECREF(input);
  Py_XDECREF(parameters);
  Py_XDECREF(output);
  return PyErr_Occurred() ? NULL : Py_BuildValue("");
}

static PyObject *Py_FourierShift(PyObject *obj, PyObject *args)
{
  PyArrayObject *input = NULL, *output = NULL, *shifts = NULL;
  int axis;
  long n, frame;
  
  if (!PyArg_ParseTuple(args, "O&O&liO&l", NI_ObjectToInputArray, &input, 
                    NI_ObjectToInputArray, &shifts, &n, &axis,
                    NI_ObjectToOutputArray, &output, &frame))
    goto exit;   
                
  if (!NI_FourierShift(input, shifts, n, axis, output, (UInt32)frame))
    goto exit;
    
exit:
  Py_XDECREF(input);
  Py_XDECREF(shifts);
  Py_XDECREF(output);
  return PyErr_Occurred() ? NULL : Py_BuildValue("");
}

static PyMethodDef methods[] = {
  {"correlate1d",         (PyCFunction)Py_Correlate1D,
   METH_VARARGS, ""},
  {"correlate",           (PyCFunction)Py_Correlate,
   METH_VARARGS, ""},
  {"uniform_filter1d",    (PyCFunction)Py_UniformFilter1D,
   METH_VARARGS, ""},
  {"min_or_max_filter1d", (PyCFunction)Py_MinOrMaxFilter1D,
    METH_VARARGS, ""},
  {"min_or_max_filter",   (PyCFunction)Py_MinOrMaxFilter,
    METH_VARARGS, ""},
  {"rank_filter",         (PyCFunction)Py_RankFilter,
   METH_VARARGS, ""},
  {"generic_filter",      (PyCFunction)Py_GenericFilter,
   METH_VARARGS, ""},
  {"generic_filter1d",    (PyCFunction)Py_GenericFilter1D,
   METH_VARARGS, ""},
  {"fourier_filter",      (PyCFunction)Py_FourierFilter,
   METH_VARARGS, ""},
  {"fourier_shift",      (PyCFunction)Py_FourierShift,
   METH_VARARGS, ""},
/*********************************************************************/
/* old stuff: */
/*********************************************************************/
  {"minimum_maximum_filter1d", (PyCFunction)Py_MinimumMaximumFilter1D, 
   METH_VARARGS, ""},
  {"minimum_maximum_filter", (PyCFunction)Py_MinimumMaximumFilter, 
   METH_VARARGS, ""},
  {"binary_erosion", (PyCFunction)Py_BinaryErosion, METH_VARARGS, ""},
  {"binary_erosion2", (PyCFunction)Py_BinaryErosion2, METH_VARARGS, ""},
  {"distance_transform_bf", (PyCFunction)Py_DistanceTransformBruteForce, 
   METH_VARARGS, ""},
  {"distance_transform_op", (PyCFunction)Py_DistanceTransformOnePass, 
   METH_VARARGS, ""},
  {"euclidean_feature_transform",
   (PyCFunction)Py_EuclideanFeatureTransform, METH_VARARGS, ""},
  {"watershed_ift", (PyCFunction)Py_WatershedIFT, METH_VARARGS, ""},
  {"label", (PyCFunction)Py_Label, METH_VARARGS, ""},
  {"find_objects", (PyCFunction)Py_FindObjects, METH_VARARGS, ""},
  {"spline_filter1d", (PyCFunction)Py_SplineFilter1D, METH_VARARGS, ""},
  {"affine_transform", (PyCFunction)Py_AffineTransform, METH_VARARGS, ""},
  {"geometric_transform", (PyCFunction)Py_GeometricTransform,
    METH_VARARGS, ""},
  {"map_coordinates", (PyCFunction)Py_MapCoordinates, METH_VARARGS, ""},
  {"zoom", (PyCFunction)Py_Zoom, METH_VARARGS, ""},
  {"shift", (PyCFunction)Py_Shift, METH_VARARGS, ""},
  {"sum", (PyCFunction)Py_Sum, METH_VARARGS, ""},
  {"mean", (PyCFunction)Py_Mean, METH_VARARGS, ""},
  {"variance", (PyCFunction)Py_Variance, METH_VARARGS, ""},
  {"minimum", (PyCFunction)Py_Minimum, METH_VARARGS, ""},
  {"maximum", (PyCFunction)Py_Maximum, METH_VARARGS, ""},
  {"minimum_position", (PyCFunction)Py_MinimumPosition, METH_VARARGS, ""},
  {"maximum_position", (PyCFunction)Py_MaximumPosition, METH_VARARGS, ""},
  {"extrema", (PyCFunction)Py_Extrema, METH_VARARGS, ""},
  {"center_of_mass", (PyCFunction)Py_CenterOfMass, METH_VARARGS, ""},
  {"histogram", (PyCFunction)Py_Histogram, METH_VARARGS, ""},
  {NULL, NULL, 0, NULL}
};


/* PyMODINIT_FUNC */
void
init_nd_image(void)
{
  Py_InitModule("_nd_image", methods);
  import_libnumarray();
}
