
/* This file is part of xtmodcoll-3.0, (c) xtmod.com
 * 
 * xtmodcoll-3.0 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * xtmodcoll-3.0 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser Public License
 * along with xtmodcoll-3.0.  If not, see <http://www.gnu.org/licenses/>.
 */


package com.xtmod.util.algo;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

import com.xtmod.util.algo.Feature.SortableFeature;



public class ListAlign<A, B, R> {

	// initial values
	protected List<A> aa;
	protected List<B> bb; 
	protected Feature<A, R> fa;
	protected Feature<B, R> fb;
	protected boolean doQuickAlign;
	
	
	public ListAlign (List<A> aa, List<B> bb, Feature<A, R> fa, Feature<B, R> fb) {
		this (aa, bb, fa, fb, false);
	}
	
	public ListAlign (List<A> aa, List<B> bb, Feature<A, R> fa, Feature<B, R> fb, boolean listsAreSorted) {
		this.aa = aa;
		this.bb = bb;
		this.fa = fa;
		this.fb = fb;
		
		this.doQuickAlign = listsAreSorted;
		this.doQuickAlign &= fa instanceof SortableFeature;
		this.doQuickAlign &= fb instanceof SortableFeature;
	}
	
	
	/**
	 * Returns a mapping of the elements of a list A to a list B. The
	 * mapping is an integer list with length |A|, where each integer denotes the
	 * respective index within list B, or <code>-1</code> if the element is not
	 * contained in B.
	 * 
	 * <p>
	 * <b>Example:</b>
	 * 
	 * <pre>
	 *  list A   mapping   list B     meaning
	 *   | a |    | -1 |     | c |     &quot;a doesn't exist within B&quot;
	 *   | c |    |  0 |     | e |     &quot;c is at index 0 within B&quot;
	 *   | d |    | -1 |     | g |     &quot;d doesn't exist within B&quot;
	 *   | e |    |  1 |     | k |     &quot;e is at index 1 within B&quot;
	 *   | k |    |  3 |     | q |     &quot;k is at index 3 within B&quot;
	 *   | m |    | -1 |               &quot;m doesn't exist within B&quot;
	 * </pre>
	 * 
	 * </p>
	 * 
	 * 
	 * @param a an unsorted list
	 * @param b an unsorted list
	 * @param c a comparator to compare elements of a with elements of b
	 * @return
	 */	
	public int[] align() {
		if (doQuickAlign)
			return align_quick();
		else
			return align_standard();
	}
	
	

	/**
	 * Standard implementation using hashmap.
	 */
	protected int[] align_standard () {

		int[] ret = new int[aa.size()];
		
		HashMap<Object, Integer> m = new HashMap<Object, Integer>();
		
		for (int i = 0; i < bb.size(); i++) {
			Object feature = fb.feature(bb.get(i));
			m.put(feature, i);
		}
		
		for (int i = 0; i < aa.size(); i++) {
			
			Object feature = fa.feature(aa.get(i));
			Object indexInB = m.get(feature);
			
			if (indexInB != null) {
				ret[i] = ((Integer)indexInB).intValue();
			} else {
				ret[i] = -1;
			}
		}
		
		return ret;
	}
	
	
	/**
	 * This is a quicker align implementation, it takes
	 * advantage of the fact that the lists are sorted.
	 * 
	 * @return the alignment
	 */
	protected int[] align_quick () {
		
		int[] ret = new int[aa.size()];

		int ib = 0;
		int ia = 0;

		for (;;) {

			if (ia == aa.size()) {
				break;
			}

			if (ib == bb.size()) {
				Arrays.fill(ret, ia, ret.length, -1);
				break;
			}

			A a = aa.get(ia);
			B b = bb.get(ib);
			Comparable a_ = (Comparable)fa.feature(a);
			Comparable b_ = (Comparable)fb.feature(b);

			int eq = a_.compareTo(b_);
			
			if (eq < 0) {
				/* a<b : no such b */
				ret[ia] = -1;
				ia += 1;
			} else if (eq == 0) {
				/* a=b : */
				ret[ia] = ib;
				ia += 1;
			} else if (eq > 0) {
				/* a>b : */
				ib += 1;
			}

		}

		return ret;
	}
		
}


