/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2008
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA 02111-1307  USA
 */
package alma.alarmsystem.core.alarms;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashSet;
import java.util.Vector;

import alma.acs.alarmsystem.generated.FaultCode;
import alma.acs.alarmsystem.generated.FaultMember;
import alma.acs.logging.AcsLogLevel;
import alma.alarmsystem.core.alarms.LaserCoreFaultState.LaserCoreFaultCodes;

import cern.laser.business.data.AlarmImpl;
import cern.laser.business.data.Category;
import cern.laser.business.data.Location;
import cern.laser.business.data.ResponsiblePerson;
import cern.laser.business.data.Source;
import cern.laser.business.data.Triplet;
import cern.laser.business.definition.data.SourceDefinition;

/**
 * The alarms generated by the alarm service.
 * <P>
 * The alarm definitions generated by this class are produced from the
 * source alarms defined in {@link LaserCoreFaultState}.
 * 
 * @author acaproni
 *
 */
public class LaserCoreAlarms {
	
	/**
	 * The empty String used for undefined String fields
	 */
	private static final String EMPTY_STRING="";
	
	/**
	 * Generate a vector with the definitions of all the alarms 
	 * that the alarm component can throw at runtime
	 * 
	 * @return A vector of alarm component alarms definitions
	 */
	public static Vector<AlarmImpl> generateAlarms() {
		/**
		 * The vector of the definition of all the alarms generated by the 
		 * alarm component
		 */
		Vector<AlarmImpl> alarms = new Vector<AlarmImpl>();
		
		// Iterate over the FCs defined in LaserCoreFaultState.LaserCoreFaultState
		for (LaserCoreFaultCodes laserFaultCode: LaserCoreFaultCodes.values()) {
			int FC=laserFaultCode.faultCode;
			int priority = laserFaultCode.priority;
			String action = EMPTY_STRING;
			String cause = laserFaultCode.cause;
			String consequence = laserFaultCode.consequences;
			String problemDesc = laserFaultCode.description;
			boolean instant = false;
			
			alma.acs.alarmsystem.generated.Location loc = new alma.acs.alarmsystem.generated.Location();
			loc.setBuilding(EMPTY_STRING);
			loc.setFloor(EMPTY_STRING);
			loc.setMnemonic(EMPTY_STRING);
			loc.setPosition(EMPTY_STRING);
			loc.setRoom(EMPTY_STRING);
		
			String FF = LaserCoreFaultState.FaultFamily;
			String FM = LaserCoreFaultState.FaultMember;
			
			AlarmImpl alarm = new AlarmImpl(); 
			
			alarm.setMultiplicityChildrenIds(new HashSet());
			alarm.setMultiplicityParentIds(new HashSet());
			alarm.setNodeChildrenIds(new HashSet());
			alarm.setNodeParentIds(new HashSet());
			
			alarm.setAction(action);
			alarm.setTriplet(new Triplet(FF, FM, FC));
			alarm.setCategories(new HashSet<Category>());
			alarm.setCause(cause);
			alarm.setConsequence(consequence);
			alarm.setProblemDescription(problemDesc);
			alarm.setHelpURL(null);
			
			alarm.setInstant(instant);
			Location location = new Location("0",loc.getFloor(),loc.getMnemonic(),loc.getPosition(),loc.getRoom());
			alarm.setLocation(location);
			alarm.setPiquetEmail(EMPTY_STRING);
			alarm.setPiquetGSM(EMPTY_STRING);
			alarm.setPriority(priority);
			ResponsiblePerson responsible = new ResponsiblePerson(
					1,EMPTY_STRING,EMPTY_STRING,EMPTY_STRING,EMPTY_STRING,EMPTY_STRING);
			alarm.setResponsiblePerson(responsible);
			
			SourceDefinition srcDef = new SourceDefinition("ALARM_SYSTEM_SOURCES","SOURCE",EMPTY_STRING,15,1);
			Source src = new Source(srcDef,responsible);
			alarm.setSource(src);
			alarm.setIdentifier(alarm.getTriplet().toIdentifier());
			
			alarms.add(alarm);
		}
		
		return alarms;
	}
}
