package cern.gp.beans.editors;

import java.awt.Color;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.FeatureDescriptor;
import java.beans.PropertyEditorSupport;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.border.LineBorder;

import org.openide.explorer.propertysheet.ExPropertyEditor;
import org.openide.explorer.propertysheet.PropertyEnv;
import org.openide.explorer.propertysheet.editors.EnhancedPropertyEditor;
import org.openide.nodes.Node;

/**
 * A PropertyEditor that puts a JButton into the corresponding Cell of the ExplorerTable
 * or PropertySheet.
 * As any other PropertyEditor, it is associated with a Bean Property:
 * <ul>
 * <li>the DisplayName of the property appears as the label fo the JButton </li>
 * <li>when user clicks on the JButton, this class calls setXxx(true) on the Bean.</li>
 * </ul>
 * 
 * For example, if you want to have a button with the label "accept",
 * <ul> 
 * <li>add a property "accept" of type <code>boolean</code> to your bean, define  
 * <code>void setAccept(boolean)</code> and <code>boolean getAccept(void)</code> accessor methods</li>
 * <li>define <code>ButtonEditor</code> by overriding the method 
 * <code>public PropertyInfo[] getPropertyInfo()</code> in your bean</li>
 * <li>in the setAccept() method, implement the code to be executed when the user clicks on the button</li>
 * </ul>
 * 
 *  
 * @author Vito Baggiolini
 *
 * @version $Revision: 1.2 $, $Date: 2006/09/25 08:52:36 $, $Author: acaproni $
 */
public class ButtonEditor extends PropertyEditorSupport implements EnhancedPropertyEditor, ExPropertyEditor {

    // the offset netbeans puts between the table cells and labels. 
    // without it, the label is drawn too much to the right.
    private static final int LABEL_OFFSET = 3;
    private JButton button; // singleton
    private PropertyEnv propertyEnv; // initialized by Netbeans

    /**
     * method called when the user clicks on the JButton.
     * By default, this implementation does a setValue(Boolean.True)
     * on the Bean property associated with the Button.
     * 
     * Override this if you need to do something else when the button is
     * clicked...
     * 
     * @param evt the ActionEvent generated by the JButton
     */
    protected void handleButtonClick(ActionEvent evt) {
        FeatureDescriptor fd = propertyEnv.getFeatureDescriptor();
        // we need to call the Node.Property directly because
        // this.setValue() does not propagate until the Bean.
        if (fd instanceof Node.Property) {
            try {
                ((Node.Property) fd).setValue(Boolean.TRUE);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        // nevertheless call this.setValue() to be coherent
        setValue(Boolean.TRUE);
    }

    //----------------------- overrides PropertyEditorSupport --------------------------
    public boolean isPaintable() {
        return true;
    }

    public void paintValue(java.awt.Graphics gfx, java.awt.Rectangle box) {
        gfx.translate(box.x - LABEL_OFFSET, box.y);
        getButton().setSize(box.width + LABEL_OFFSET, box.height);
        getButton().paint(gfx);
        gfx.translate(-box.x + LABEL_OFFSET, -box.y);
    }

    //----------------------- Implements EnhancedPropertyEditor ------------------------
    public Component getInPlaceCustomEditor() {
        JButton jb = getButton();
        jb.doClick();
        return jb;
    }

    public boolean hasInPlaceCustomEditor() {
        return true;
    }

    public boolean supportsEditingTaggedValues() {
        return false;
    }

    //----------------------- Implements ExPropertyEditor ------------------------------
    /* (non-Javadoc)
     * @see org.openide.explorer.propertysheet.ExPropertyEditor#attachEnv(org.openide.explorer.propertysheet.PropertyEnv)
     */
    public void attachEnv(PropertyEnv env) {
        propertyEnv = env;
    }

    //---------------------------- private methods -------------------------------------
    /**
     * lazy creator class. Should be called only after {@link #attachEnv(PropertyEnv)}
     * @return the JButton to be used both as Renderer and Editor
     */
    private JButton getButton() {
        if (button == null) {
            button = createButton();
        }
        return button;
    }

    private JButton createButton() {
        if (propertyEnv == null) {
            throw new RuntimeException("propertyEnv variable not yet initialized");
        }

        JButton jb = new JButton(propertyEnv.getFeatureDescriptor().getDisplayName());
        jb.setHorizontalTextPosition(JLabel.CENTER);
        jb.setHorizontalAlignment(JLabel.CENTER);
        jb.setBorder(new LineBorder(Color.gray));
        jb.setOpaque(false);
        jb.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                handleButtonClick(e);
            }
        });
        return jb;
    }

//    /**
//     * initializes the class variables 
//     *
//     */
//    private void initClass() {
//        button = getButton();
//    }

}
