/*
* Copyright (C) The Community OpenORB Project. All rights reserved.
*
* This software is published under the terms of The OpenORB Community Software
* License version 1.0, a copy of which has been included with this distribution
* in the LICENSE.txt file.
*/
package org.openorb.util;

/**
 * Profiler utility.
 *
 * @author Chris Wood
 * @version $Revision: 1.7 $ $Date: 2002/07/14 19:12:40 $ 
 */
public abstract class Profiler
{
    private static long s_begin;
    private static long [] s_sums;
    private static long [] s_sumsq;
    private static int s_upto;
    private static int s_count;
    private static boolean s_active = false;

    /**
     * Reset the values of the profiler.
     * 
     * @param points The number of profiling points to take.
     */
    public static void reset( int points )
    {
        s_sums = new long[ points ];
        s_sumsq = new long[ points ];
        s_upto = 0;
        s_count = 0;
        s_active = true;
        s_begin = System.currentTimeMillis();
    }

    /**
     * Get another sample.
     */
    public static void point()
    {
        if ( !s_active )
            return ;

        long total = System.currentTimeMillis() - s_begin;

        s_sums[ s_upto ] += total;

        s_sumsq[ s_upto ] += total * total;

        ++s_upto;

        s_begin = System.currentTimeMillis();
    }

    /**
     * Make another sample and return the current index.
     * 
     * @return The current sample index.
     */
    public static int next()
    {
        point();
        int ret = s_upto;
        ++s_count;
        s_upto = 0;
        s_begin = System.currentTimeMillis();
        return ret;
    }

    /**
     * Calculate the average of the sample points taken so far.
     * 
     * @return Return an array of average values.
     */
    public static double [] means()
    {
        double [] means = new double[ s_sums.length ];

        for ( int i = 0; i < means.length; ++i )
            means[ i ] = ( double ) s_sums[ i ] / ( double ) s_count;

        s_active = false;

        return means;
    }

    /**
     * Calculate the standard derivation.
     * 
     * @return The standard derivation of the sample points taken so far.
     */
    public static double [] stddev()
    {
        double [] stdev = new double[ s_sums.length ];
        double n = ( double ) s_count;

        for ( int i = 0; i < stdev.length; ++i )
            stdev[ i ] = Math.sqrt( ( n * ( double ) s_sumsq[ i ]
                  - ( double ) s_sums[ i ] * ( double ) s_sums[ i ] ) / ( n * n ) );

        s_active = false;

        return stdev;
    }
}
