/*
* Copyright (C) The Community OpenORB Project. All rights reserved.
*
* This software is published under the terms of The OpenORB Community Software
* License version 1.0, a copy of which has been included with this distribution
* in the LICENSE.txt file.
*/
package org.openorb.io;

import org.omg.CORBA.OctetSeqHolder;
import org.omg.CORBA.IntHolder;


/**
 * MarshalBuffers serve as a sink for data to be marshaled into. Fragmentation
 * of the data as it is marshaled can be controlled by the Listener registered
 * with the buffer at creation time.
 *
 * @author Unknown
 */
public class MarshalBuffer
{
    /**
     * Construct marshal buffer without listener. Use the fragment and
     * lastFragment methods to extract data from the buffer.
     */
    public MarshalBuffer()
    {
        m_head = m_tail = new Scrap();
        m_tail.fBuffer = new byte[ Scrap.DEFAULT_SCRAP ];
        m_tail.fOffset = 0;
        m_tail.fLength = 0;
        m_tail.fMode = Scrap.MODE_NORMAL;
        m_tail.fPosition = 0;
    }

    /**
     * Construct marshall buffer with listener. The listener is informed
     * when the buffer grows or is closed, and calls the fragment or
     * lastFragment methods respectivly.
     */
    public MarshalBuffer( Listener listener, Object listenerCookie )
    {
        this();

        m_listener = listener;
        m_listener_cookie = listenerCookie;
    }

    /**
     * Returns true if not connected to a listener. To get the data from
     * the buffer use lastFragment()
     */
    public boolean isStandalone()
    {
        return m_listener == null;
    }

    /**
     * count of all bytes inserted into the buffer, including previous
     * fragments
     */
    public int size()
    {
        if ( !prealloc() )
            return -1;

        return m_tail.fPosition;
    }

    /**
     * count of all bytes available for extracting into a fragment. 
     */
    public int available()
    {
        if ( !prealloc() )
            return -1;

        return m_tail.fPosition - m_head.fPosition + m_head.fLength;
    }

    /**
     * Allow or dissallow fragmentation. An opertunity to send a fragment will
     * occour when this is true only after the next append to the buffer.
     */
    public void setAllowFragment( boolean allowFragment )
    {
        m_allow_fragment = allowFragment;
    }

    /**
     * Test if fragmentation is currently enabled. This may be false even after
     * a call to setAllowFragment(true) if fragmentation is disabled for some
     * other reason. While this returns false availIncreaced will not be called
     * on the listener
     */
    public boolean getAllowFragment()
    {
        return m_allow_fragment && m_header_fragment && m_block_fragment && !m_in_fragment;
    }

    /**
     * Alocate space at end of buffer. The dest scrap is modified so
     * it's contents contain the allocated space. The allocated space is
     * considered to be scratch space, it's buffer is only avalable
     * until the next call.
     *
     * @param buf Out parameter, holds pointer to scratch space on
     * return. This space should not be stored. The pointer will be 
     * invalidated by setting it's value to null on next call.
     * @param off Out parameter, holds buffer offset on return.
     * @param len Length of requested buffer. 
     */
    public void alloc( OctetSeqHolder buf, IntHolder off, int len )
    {
        if ( !prealloc() )
        {
            if ( m_ignore_buffer == null || m_ignore_buffer.length < len )
                m_ignore_buffer = new byte[ len ];

            buf.value = m_ignore_buffer;

            off.value = 0;

            return ;
        }

        if ( m_in_listener && !m_in_fragment )
            throw new IllegalStateException( "Cannot modify buffer while calling listener" );

        if ( m_tail.fBuffer.length - m_tail.fLength - m_tail.fOffset >= len )
        {
            // tail contains both previously allocated space and
            // newly allocated space
            buf.value = m_tail.fBuffer;
            off.value = m_tail.fOffset + m_tail.fLength;

            m_tail.fLength += len;
            m_tail.fPosition += len;
        }
        else
        {
            // previously allocated space in old tail
            // newly allocated space in new tail
            Scrap nex = new Scrap();

            if ( len > Scrap.DEFAULT_SCRAP )
                nex.fBuffer = new byte[ len ];
            else
                nex.fBuffer = new byte[ Scrap.DEFAULT_SCRAP ];

            nex.fOffset = 0;

            nex.fLength = len;

            nex.fMode = Scrap.MODE_NORMAL;

            nex.fPosition = m_tail.fPosition + len;

            buf.value = nex.fBuffer;

            off.value = 0;

            m_tail.fNext = nex;

            m_tail = nex;
        }

        m_last_buffer = buf;
    }

    /**
     * Attach a readonly scrap to the end of the buffer. This will be
     * stored by reference and will not be copied unless the StorageBuffer
     * it spawns uses readWriteMode when iterating over itself.
     */
    public void append( byte [] buf, int off, int len )
    {
        if ( !prealloc() )
            return ;

        if ( m_in_listener && !m_in_fragment )
            throw new IllegalStateException( "Cannot modify buffer while calling listener" );

        if ( len == 0 )
            return ;

        if ( len < Scrap.DEFAULT_SCRAP )
        {
            // append by copy. Not worth doing otherwise realy.
            if ( m_tail.fBuffer.length - m_tail.fLength - m_tail.fOffset >= len )
            {
                // tail contains entire append buffer.
                System.arraycopy( buf, off, m_tail.fBuffer, m_tail.fOffset + m_tail.fLength, len );

                m_tail.fLength += len;
                m_tail.fPosition += len;
            }
            else
            {
                // use some of the old and some of the new.
                int rem = m_tail.fBuffer.length - m_tail.fLength - m_tail.fOffset;
                System.arraycopy( buf, off, m_tail.fBuffer, m_tail.fOffset + m_tail.fLength, rem );

                m_tail.fLength += rem;
                m_tail.fPosition += rem;

                Scrap nex = new Scrap();
                nex.fBuffer = new byte[ Scrap.DEFAULT_SCRAP ];
                nex.fOffset = 0;
                nex.fLength = len - rem;
                nex.fMode = Scrap.MODE_NORMAL;
                nex.fPosition = m_tail.fPosition + len - rem;

                System.arraycopy( buf, off + rem, nex.fBuffer, 0, len - rem );

                m_tail.fNext = nex;
                m_tail = nex;
            }

            return ;
        }

        /* invariant: Tail scrap is always a normal buffer. */

        if ( m_tail.fLength == 0 )
        {
            // a special scrap has just been allocated, leaving an empty
            // scrap at the tail. Put readonly data in tail and move tail's
            // contents one forward.
            Scrap nex = new Scrap();
            nex.fBuffer = m_tail.fBuffer;
            nex.fOffset = m_tail.fOffset;
            nex.fLength = 0;
            nex.fMode = Scrap.MODE_NORMAL;
            nex.fPosition = m_tail.fPosition + len;

            m_tail.fBuffer = buf;
            m_tail.fOffset = off;
            m_tail.fLength = len;
            m_tail.fMode = Scrap.MODE_READONLY;
            m_tail.fPosition = nex.fPosition;

            m_tail.fNext = nex;
            m_tail = nex;
        }
        else
        {
            // make current tail shared, append a readonly scrap then append
            // a normal scrap with the rest of the old tail's buffer.
            m_tail.fMode = Scrap.MODE_SHARED;

            Scrap ro = new Scrap();
            ro.fBuffer = buf;
            ro.fOffset = off;
            ro.fLength = len;
            ro.fMode = Scrap.MODE_READONLY;
            ro.fPosition = m_tail.fPosition + len;

            Scrap nex = new Scrap();
            nex.fBuffer = m_tail.fBuffer;
            nex.fOffset = m_tail.fOffset + m_tail.fLength;
            nex.fLength = 0;
            nex.fPosition = ro.fPosition;
            nex.fMode = Scrap.MODE_NORMAL;

            m_tail.fNext = ro;
            ro.fNext = nex;
            m_tail = nex;
        }

        postalloc();
    }

    /**
     * Insert padding.
     */
    public void pad( int len )
    {
        if ( !prealloc() )
            return ;

        if ( m_in_listener && !m_in_fragment )
            throw new IllegalStateException( "Cannot modify buffer while calling listener" );

        if ( len == 0 )
            return ;

        int rem = m_tail.fBuffer.length - m_tail.fLength - m_tail.fOffset;

        if ( rem >= len )
        {
            // all the padding fits in the current scrap
            m_tail.fLength += len;
            m_tail.fPosition += len;
        }
        else
        {
            // put some padding into current scrap
            m_tail.fLength += rem;
            m_tail.fPosition += rem;

            // and the rest into a new scrap.
            Scrap nex = new Scrap();
            nex.fBuffer = new byte[ Scrap.DEFAULT_SCRAP ];
            nex.fOffset = 0;
            nex.fLength = len - rem;
            nex.fMode = Scrap.MODE_NORMAL;
            nex.fPosition = m_tail.fPosition + nex.fLength;

            m_tail.fNext = nex;
            m_tail = nex;
        }

        postalloc();
    }

    /**
     * Add a header generator. Header generators generate data at a later time,
     * typically writing the length of the entire fragment or message when it
     * is about to be sent. Header generators also get an opportunity to write 
     * to the beginning of a new fragment when a fragment is taken from the buffer.
     *
     * @param gen the header generator
     * @param len the length of the data which will be written by the generator.
     * @param frag true if the header allows fragmentation to occour within it's
     *         range of authority.
     * @param cookie passed to the begin message and end message operations.
     */
    public void addHeader( HeaderGenerator gen, int len, boolean frag, Object cookie )
    {
        if ( m_in_listener && !m_in_fragment )
            throw new IllegalStateException( "Cannot modify buffer while calling listener" );

        if ( len < 0 )
            throw new IndexOutOfBoundsException();

        if ( gen == null )
            throw new NullPointerException();

        if ( !prealloc() )
            return ;

        HeaderData hd = new HeaderData();

        hd.fPrevious = m_last_header;

        m_last_header = hd;

        hd.fLastHeaderFragment = m_header_fragment;

        hd.fPosition = size();

        hd.fGenerator = gen;

        hd.fCookie = cookie;

        // disable fragmentation while we add the header
        if ( len != 0 )
        {
            m_header_fragment = false;
            OctetSeqHolder bufh = new OctetSeqHolder();
            IntHolder offh = new IntHolder();

            alloc( bufh, offh, len );

            hd.fBuffer = bufh.value;
            hd.fOffset = offh.value;
            hd.fLength = len;
        }

        // update header fragmentability.
        m_header_fragment = hd.fLastHeaderFragment && frag;
    }

    /**
     * Begin a block. Blocks unlike headers can overlap fragmentation boundaries
     * but can themselves be fragmented when the position in the buffer that they
     * hold is about to be sent. Every beginBlock operation is always paired with
     * an endBlock operation some time later in the stream.<p>
     * Nonfragmentable blocks can be nested, however fragmentable blocks cannot.
     * 
     * @param gen the block generator.
     * @param len the length of the data which will be written.
     * @param frag true if the block can be fragmented.
     * @param cookie passed to the fragment block and end block operation.
     */
    public void beginBlock( BlockGenerator gen, int len, boolean frag, Object cookie )
    {
        if ( m_in_listener && !m_in_fragment )
            throw new IllegalStateException( "Cannot modify buffer while calling listener" );

        if ( len < 0 )
            throw new IndexOutOfBoundsException();

        if ( gen == null )
            throw new NullPointerException();

        if ( !prealloc() )
            return ;

        BlockData bd = new BlockData();

        bd.fPrevious = m_last_block;

        bd.fIsFragment = ( m_last_block == null ) ? frag : false;

        m_last_block = bd;

        bd.fPosition = size();

        bd.fGenerator = gen;

        bd.fCookie = cookie;

        if ( len != 0 )
        {
            // disable fragmentation while we add the header
            m_block_fragment = false;
            OctetSeqHolder bufh = new OctetSeqHolder();
            IntHolder offh = new IntHolder();

            alloc( bufh, offh, len );

            bd.fBuffer = bufh.value;
            bd.fOffset = offh.value;
            bd.fLength = len;
        }

        // update block fragmentablilty.
        m_block_fragment = bd.fIsFragment;
    }

    /**
     * Call the endBlock operation on the last block written with the beginBlock
     * operation and remove the hold on the data.
     */
    public void endBlock()
    {
        if ( m_in_listener && !m_in_fragment )
            throw new IllegalStateException( "Cannot modify buffer while calling listener" );

        if ( m_last_block == null )
            throw new IllegalStateException( "End block without begin block" );

        m_last_block.fGenerator.endBlock( m_last_block.fBuffer, m_last_block.fOffset, 
                m_last_block.fLength, size() - m_last_block.fPosition, m_last_block.fCookie );

        m_last_block = m_last_block.fPrevious;

        m_block_fragment = ( m_last_block == null ) ? true : m_last_block.fIsFragment;
    }

    /**
     * Close. Calls close operation on listeners and frees all data
     * contained within the buffer. It is illegal to call close while within a 
     * block.
     */
    public void close()
    {
        if ( m_in_listener )
            throw new IllegalStateException( "Cannot close buffer while calling listener." );

        if ( m_last_block != null )
            throw new IllegalStateException( "All blocks must be closed before closing buffer" );

        if ( !prealloc() )
            return ;

        if ( m_listener != null )
        {
            m_in_listener = true;
            m_listener.bufferClosed( this, m_tail.fPosition, m_listener_cookie );
            m_in_listener = false;
        }

        m_head = m_tail = null;
        m_last_header = null;
    }

    /**
     * Cancel marshal. Calls cancel operation on listeners and frees all
     * data contained within the buffer. All following allocations will 
     * either be silently discarded (if the listener does not throw an exception)
     * or will be responded to with the exception passed.
     */
    public void cancel( org.omg.CORBA.SystemException ex )
    {
        if ( !prealloc() )
            return ;

        if ( m_listener != null )
        {
            m_in_listener = true;

            try
            {
                m_listener.bufferCanceled( this, ex, m_listener_cookie );
            }
            catch ( org.omg.CORBA.SystemException sex )
            {
                m_cancel_exception = sex;
            }

            m_in_listener = false;
        }
        else
            m_cancel_exception = ex;

        m_head = m_tail = null;

        m_last_header = null;

        m_last_block = null;

        if ( m_cancel_exception != null )
            throw m_cancel_exception;
    }

    /**
     * Prepare fragment. The returned storage buffer will be exactly the length
     * specified.
     */
    public StorageBuffer fragment( int len )
    {
        if ( !prealloc() )
            return null;

        if ( m_listener != null && !m_in_listener )
            throw new IllegalStateException( "Operation disallowed, must be called from listener" );

        if ( !getAllowFragment() )
            throw new IllegalStateException( "Buffer cannot currently be fragmented" );

        if ( len > m_tail.fPosition - m_head.fPosition + m_head.fLength )
            throw new IndexOutOfBoundsException();

        m_in_fragment = true;

        // call fragment on the fragmentable block (if any)
        if ( m_last_block != null )
        {
            BlockData bd = m_last_block;
            m_last_block = null;
            bd.fGenerator.fragmentBlock( bd.fBuffer, bd.fOffset, bd.fLength,
                    size() - bd.fPosition, this, bd.fCookie );
        }

        int endpos = len + m_head.fPosition - m_head.fLength;

        // call close on all the header blocks.
        HeaderData firstHeader = null;
        HeaderData tmp;

        while ( m_last_header != null )
        {
            m_last_header.fGenerator.endMessage( m_last_header.fBuffer,
                    m_last_header.fOffset, m_last_header.fLength, 
                    true, endpos - m_last_header.fPosition, m_last_header.fCookie );
            tmp = m_last_header.fPrevious;
            m_last_header.fPrevious = firstHeader;
            firstHeader = m_last_header;
            m_last_header = tmp;
        }

        // create the buffer which will be returned.

        Scrap mid;

        if ( m_tail.fPosition - m_tail.fLength < endpos )
            mid = m_tail;
        else
            for ( mid = m_head; mid.fPosition < endpos; mid = mid.fNext )

                ;

        Scrap nexthead;

        if ( mid.fPosition == endpos )
        {
            nexthead = mid.fNext;
            mid.fNext = null;
        }
        else
        {
            int olap = mid.fPosition - endpos;
            nexthead = new Scrap();
            nexthead.fMode = Scrap.MODE_SHARED | mid.fMode;
            nexthead.fBuffer = mid.fBuffer;
            nexthead.fPosition = mid.fPosition;
            nexthead.fOffset = mid.fOffset + mid.fLength - olap;
            nexthead.fLength = olap;
            nexthead.fNext = mid.fNext;

            mid.fLength = mid.fLength - olap;
            mid.fPosition = endpos;
            mid.fNext = null;
            mid.fMode = Scrap.MODE_SHARED | mid.fMode;
        }

        mid = null;
        StorageBuffer fragment = new StorageBuffer( m_head, len );

        // now begin the next message.
        m_head = m_tail = new Scrap();
        m_tail.fBuffer = new byte[ Scrap.DEFAULT_SCRAP ];
        m_tail.fOffset = 0;
        m_tail.fLength = 0;
        m_tail.fMode = Scrap.MODE_NORMAL;
        m_tail.fPosition = endpos;

        // call begin block on all the header blocks
        while ( firstHeader != null )
        {
            firstHeader.fGenerator.beginMessage( this, firstHeader.fCookie );
            firstHeader = firstHeader.fPrevious;
        }

        // invalidate any last buffer.
        prealloc();

        int addHeader = 0;

        // append the overlapping data.
        if ( nexthead != null )
        {
            addHeader = m_tail.fPosition + nexthead.fLength - nexthead.fPosition;

            if ( m_tail.fLength == 0 )
            {
                m_tail.fBuffer = nexthead.fBuffer;
                m_tail.fLength = nexthead.fLength;
                m_tail.fOffset = nexthead.fOffset;
                m_tail.fMode = nexthead.fMode;
                m_tail.fNext = nexthead.fNext;
                m_tail.fPosition += nexthead.fLength;
                nexthead = nexthead.fNext;
            }
            else
            {
                m_tail.fNext = nexthead;
            }

            // update the positions of the old buffers.

            while ( nexthead != null )
            {
                nexthead.fPosition += addHeader;
                m_tail = nexthead;
                nexthead = nexthead.fNext;
            }
        }

        BlockData bd = m_last_block;

        while ( bd != null )
        {
            bd.fPosition += addHeader;
            bd = bd.fPrevious;
        }

        m_in_fragment = false;

        // done, return the message fragment.
        return fragment;
    }

    /**
     * Return the last fragment. This also closes the buffer.
     */
    public StorageBuffer lastFragment()
    {
        if ( !prealloc() )
            return null;

        if ( m_listener != null && !m_in_listener )
            throw new IllegalStateException( 
                    "Operation disallowed, must be called from listener. Call close operation" );

        if ( m_last_block != null )
            throw new IllegalStateException( 
                    "Attempt to close buffer without closing all blocks" );

        // fill in the headers.
        while ( m_last_header != null )
        {
            m_last_header.fGenerator.endMessage( m_last_header.fBuffer,
                    m_last_header.fOffset, m_last_header.fLength, 
                    false, m_tail.fPosition - m_last_header.fPosition, m_last_header.fCookie );
            m_last_header = m_last_header.fPrevious;
        }

        Scrap msghead = m_head;
        int len = m_tail.fPosition;

        // go to closed state.
        m_head = m_tail = null;
        return new StorageBuffer( msghead, len );
    }

    /**
     * Interface HeaderGenerator.
     */
    public static interface HeaderGenerator
    {
        /**
         * Called when message is about to get sent. Modifications can be made
         * to the bytes allocated at the addHeader stage.
         *
         * @param buf buffer containing the reserved bytes. Not all of the buffer
         *            is considered to be read-write.
         * @param pos offset into buf of first modifiable byte.
         * @param len length of modifiable bytes.
         * @param fragment true if this is called as a response to the fragment
         *            operation.
         * @param length length in bytes between the position that addHeader was
         *            called and the end of the message.
         * @param cookie the cookie passed to the addHeader operation.
         */
        void endMessage( byte [] buf, int pos, int len,
                boolean fragment, int length, Object cookie );

        /**
         * Called to begin a new fragment. Writes may be made to the marshal buffer,
         * including adding a new header.
         *
         * @param buffer the buffer getting marshaled to.
         * @param cookie the cookie passed to the addHeader operation.
         */
        void beginMessage( MarshalBuffer buffer, Object cookie );
    }

    /**
     * Interface BlockGenerator.
     */
    public static interface BlockGenerator
    {
        /**
         * Called when endBlock operation is called. 
         *
         * @param buf buffer containing the reserved bytes. Not all of the buffer
         *            is considered to be read-write.
         * @param pos offset into buf of first modifiable byte.
         * @param len length of modifiable bytes.
         *
         * @param length length in bytes between the position that beginBlock was
         *            called and the end of the block.
         * @param cookie the cookie passed to the addHeader operation.
         */
        void endBlock( byte [] buf, int pos, int len,
                              int length, Object cookie );

        /**
         * Called when fragment is called and a block will be fragmented. Writes
         * may be made to the MarshalBuffer, including begining a new block.
         *
         * @param buf buffer containing the reserved bytes. Not all of the buffer
         *            is considered to be read-write.
         * @param pos offset into buf of first modifiable byte.
         * @param len length of modifiable bytes.
         *
         * @param length length in bytes between the position that beginBlock was
         *            called and the end of the block.
         * @param buffer the marshal buffer.
         * @param cookie the cookie passed to the addHeader operation.
         */
        void fragmentBlock( byte [] buf, int pos, int len,
                                   int length, MarshalBuffer buffer, Object cookie );
    }

    public static interface Listener
        extends java.util.EventListener
    {
        /**
         * called whenever the size of the buffer increaces or flush is called 
         * while fragemntation is enabled.
         */
        void availIncreaced( MarshalBuffer buffer, int available, Object cookie );

        /**
         * called when the buffer is closed.
         */
        void bufferClosed( MarshalBuffer buffer, int available, Object cookie );

        /**
         * Called when the marshal sequence is canceled by calling the cancel 
         * operation. This should either throw a system exception or simply return
         * if the cancel will be handled later on.
         */
        void bufferCanceled( MarshalBuffer buffer,
                org.omg.CORBA.SystemException ex, Object cookie );
    }

    private boolean prealloc()
    {
        if ( m_cancel_exception != null )
            throw m_cancel_exception;

        if ( m_head == null )
            return false;

        // invalidate last allocated buffer.
        if ( m_last_buffer != null )
        {
            m_last_buffer.value = null;
            m_last_buffer = null;
            postalloc();
        }

        return true;
    }

    private void postalloc()
    {
        // check for fragment request
        if ( m_listener != null
              && m_allow_fragment
              && m_header_fragment
              && m_block_fragment
              && !m_in_fragment )
        {
            m_in_listener = true;
            m_listener.availIncreaced( this, 
                  m_tail.fPosition - m_head.fPosition + m_head.fLength, m_listener_cookie );
            m_in_listener = false;
        }
    }

    private static class HeaderData
    {
        public HeaderData fPrevious;
        public boolean fLastHeaderFragment;

        public int fPosition;
        public HeaderGenerator fGenerator;
        public Object fCookie;

        public byte [] fBuffer;
        public int fOffset;
        public int fLength;
    }

    private static class BlockData
    {
        public BlockData fPrevious;
        public boolean fIsFragment;

        public int fPosition;
        public BlockGenerator fGenerator;
        public Object fCookie;

        public byte [] fBuffer;
        public int fOffset;
        public int fLength;
    }

    private org.omg.CORBA.SystemException m_cancel_exception = null;

    private Scrap m_head;
    private Scrap m_tail;

    private OctetSeqHolder m_last_buffer = null;

    private boolean m_allow_fragment = false;
    private boolean m_in_fragment = false;

    private boolean m_header_fragment = true;
    private HeaderData m_last_header = null;

    private boolean m_block_fragment = true;
    private BlockData m_last_block = null;

    // informed when size grows and buffer is closed.
    private Listener m_listener = null;
    private Object m_listener_cookie = null;
    private boolean m_in_listener = false;

    // dummy buffer to marshal data into once cancel has been called.
    private byte [] m_ignore_buffer = null;

    public static void main( String [] args )
    {
        MarshalBuffer mbuf = new MarshalBuffer();

        OctetSeqHolder buf = new OctetSeqHolder();
        IntHolder pos = new IntHolder();
        int len = 1;

        for ( int i = 0; i < 3000; i += len )
        {
            mbuf.alloc( buf, pos, len );

            for ( int j = 0; j < len; ++j )
                buf.value[ pos.value + j ] = ( byte ) ( ( i + j ) % 121 );
        }

        StorageBuffer sbuf = mbuf.lastFragment();

        System.out.println( "available = " + sbuf.available() );

        /*
        HexPrintStream hps = new HexPrintStream(System.out, HexPrintStream.FORMAT_MIXED);
        sbuf.mark();
        try {
          sbuf.writeTo(hps);
          hps.flush();
    }
        catch(java.io.IOException ex) {
          ex.printStackTrace(System.out);
          System.out.println(ex);
    }
        sbuf.reset();
        */

        /*
        sbuf.mark();
        IntHolder rlen = new IntHolder();
        for(int i = 0; i < 3000;) {
          rlen.value = 1024;
          len = sbuf.next(buf, pos, rlen);
          for(int j = 0; j < len; ++j)
            if(buf.value[pos.value+j] != (byte)((i+j)%121))
              System.out.println("Error at index " + (i+j));
          i += len;
    }
        sbuf.reset();
        */

        System.out.println( "available = " + sbuf.available() );
        byte [] lin = sbuf.linearize();

        for ( int i = 0; i < lin.length; ++i )
            if ( lin[ i ] != ( byte ) ( i % 121 ) )
                System.out.println( "Error at index " + i );
    }
}
