/*******************************************************************************
* E.S.O. - VLT project
*
* 
*
* who       when      what
* --------  --------  ----------------------------------------------
* bgustafs  05/03/97  modified error check
* bgustafs  21/02/97  changed to use fprintf instead of logMsg because of 
*                     stale buffer pointers
* imunoz  11/10/96  man page added
* imunoz  23/07/96  created
*/

/************************************************************************
*   NAME
*        lcubootLogInit
*        lcubootLogFinish
*        lcubootLogMsg
*
*        - log of boot messages from lcuboot into a WS file.
*
*   SYNOPSIS
*
*       STATUS lcubootLogInit(void);
*       STATUS lcubootLogFinish(void);
*       STATUS lcubootLogMsg(char *fmt, int arg1, int arg2, int arg3, 
*	          	     int arg4, int arg5, int arg6)
*
*
*   DESCRIPTION
*
*       These functions supports logging of messages generated while
*	executing the LCU's bootScript.
*	Messages are logged into the file lcubootLogFile, this file is
*	located in the boot WS in the directory $VLTDATA/ENVIRONMENTS/<lcuEnv>.
*	
*	lcubootLogInit -  Open the log file and adds its file
*               descriptor to the VxWorks logging system. To avoid that the
*               logfile size grows indefinetely, it is re-created
*		every time the boot process is started and the file
*               size is greater tha 0. 
*
*	lcubootLogFinish - Logs "lcuboot end" in the logFile,
*               closes it  and deletes the file descriptor from
*               the VxWorks's logging system.
*		If lcubootLogInit have not been called or if it have
*		failed nothing is done.
*
*	lcubootLogMsg - Behaves exactly like vxWorks's logMsg, but
*	        it flushes the buffer of the logfile.
*		If lcubootLogInit have not been called or if it have
*		failed nothing is done.
*
*   FILES
*
*   ENVIRONMENT
*
*   RETURN VALUES 
*
*       lcubootLogInit   -  lcubootERROR if it fails to open the
*                           logFile, otherwise  lcubootOK.
*       lcubootLogMsg    -  lcubootOK.
*	lcubootLogFinish -  lcubootOK. 
*
*
*   EXAMPLES
*          
*	   lcubooLogInit
*          lcubootLogMsg("hello %d %d %d testing ...",1,2,3,0,0,0)
*          lcubootLogFinish
*
*   SEE ALSO
*      logLib
*
*   BUGS   
*
*------------------------------------------------------------------------
*/

#include "vltPort.h"

static const char *rcsId="@(#) $Id: lcubootLogInit.c,v 1.2 2008/10/27 14:53:57 bjeram Exp $"; 

/* 
 * System Headers
 */

#include <stdio.h>
#include <fcntl.h>
#include <envLib.h>
#include <logLib.h>
#include <ioLib.h>
#include <stat.h>

/* 
 * Local Headers
 */
#include "lcuboot.h"


int lcubootLogFileFd;
FILE *lcubootLogFileFp;

/* 
 *  Init loggging to a WS/file, file name is $BOOTHOME/lcubootLogFile
 */
STATUS lcubootLogInit(void)
{
    char fname[80];
    int openMask = O_RDWR | O_CREAT | O_TRUNC;

    sprintf(fname, "%s/lcubootLogFile", getenv("BOOTHOME"));

    unlink(fname); /* delete lcubootLogFile */

    if ( (lcubootLogFileFd = open(fname, openMask, 0777)) < 0)
	{
	perror("open");
	return lcubootERROR;
	}

    if ( (lcubootLogFileFp = fdopen(lcubootLogFileFd, "w")) == NULL)
	{
	perror("fdopen");
	return lcubootERROR;
	}

    return lcubootOK;
}


/*
 * Flush lcubootLogFile pending writes, close the file and deleteit from 
 * the list of fd's to log.
 */
STATUS lcubootLogFinish(void)
{
    if (lcubootLogFileFd < 0)
	return lcubootOK;

    lcubootLogMsg("lcuboot end\n",0,0,0,0,0,0);
    fflush(lcubootLogFileFp);
    ioctl(lcubootLogFileFd, FIOSYNC,0);  
    fclose(lcubootLogFileFp);
    close(lcubootLogFileFd);
    
    return lcubootOK;
}


/*
 * Writes a logMsg to lcubootLogFile, flush its buffer
 */
STATUS lcubootLogMsg(const char *fmt, int arg1, int arg2, int arg3, 
		     int arg4, int arg5, int arg6)
{
    if (lcubootLogFileFd < 0)
	return lcubootOK;
 
    printf (fmt,arg1,arg2,arg3,arg4,arg5,arg6);
    fprintf (lcubootLogFileFp,fmt,arg1,arg2,arg3,arg4,arg5,arg6);
    fflush (lcubootLogFileFp);
    ioctl(lcubootLogFileFd, FIOSYNC,0); /* flush NFS buffer */

    return lcubootOK;
}

/*___oOo___*/
