/*******************************************************************************
* E.S.O. - VLT project
*
* ...
*
* who       when      what
* --------  --------  ----------------------------------------------
* bgustafs 2002-10-25 correction of search path, previous correction was not reliable
* bgustafs 2002-06-26 corrected serch path 
* bgustafs  15/07/01  modified man page, see SPR20010068
* bgustafs  18/07/97  adapted to Tornado
* imunoz  26/03/96  man page added
* imunoz  14/03/96  created
*/

/************************************************************************
*   NAME
*       lcubootErrorLoad,
*	lcubootErrorGetFormat
*	- LCU Error system functions.
*
*   SYNOPSIS
*       int lcubootErrorLoad(const char *modName)
*       int lcubootErrorGetFormat(const char *modName, 
*	                          int errNumber, 
*       		          char *severity, 
*		                  char *format) 
*
*	
*
*   DESCRIPTION
*       This functions provides the facilities to load ERROR files and
*       retrieve the information about error text and format for a
*       given module and error number. lcubootErrorLoad is called automatically
*       at module loading, but can be used by itself for test purposes. 
*
*	lcubootErrorLoad - load the ERROR definition file for the given
*                          module, if the module does not have ERROR
*                          files it just return. If the module has the
*                          ERROR file (<mod>_ERRORS) but the index
*                          file is not present (<mod>ERRORS.IDX) the
*                          script is aborted.
*
*       lcubootErrorGetFormat - retrieves severity and format
*                               definition for the pair
*                               (modName,errNumber). 
*               Returns values: lcubootOK if no problem
*                               lcubootError_MOD_NOT_FOUND if the
*                                      module was not found
*                               lcubootError_ERR_NOT_FOUND if the
*                                      error number was not found for
*                                      the module 
*                               lcubootError_INTERNAL if an internal
*                                      error occurs.
*   FILES
*	Error-defition files are accessed read-only under VLTROOT
*	respectively INTROOT:
*
*		ERRORS/<mod>_ERRORS
*		ERRORS/<mod>ERRORS.IDX
*
*	where <mod> is the name of the module, as given by `modName'.
*
*   ENVIRONMENT
*
*   COMMANDS
*
*   RETURN VALUES
*   
*   CAUTIONS 
*	Note that the name of the module and the prefix of the
*	error-definition files must be the same, otherwise the
*	files will not be found!
*
*   EXAMPLES
*       lcubootErrorLoad("mod");
*       lcubootErrorLoad("motci");
*
*	status = lcubootErrorGeFormat("mod",mod_ERR_NOT_FOUND,
*                                      myFormat, mySeveritty);
*       if (status != lcudrvOK) 
*	   switch (status) {
*	         case lcubootError_MOD_NOT_FOUND: ....
*	         case  lcubootError_ERR_NOT_FOUND: ...
*	         default: ...
*	   }
*
*   SEE ALSO
*       lcubootAutoEnv(1), lcubootAutoGen(1), lcubootAutoLcc(1),
*       loadLib(1), symLib(1)
*
*   BUGS   
* 
*------------------------------------------------------------------------
*/

#define ALLOCSIZE 20

#include "lcubootPrivate.h"
#include "ccs.h" /* only necessary for SUCCESS macro and ccsERROR */

static const char *rcsId="@(#) $Id: lcubootErrorLoad.c,v 1.2 2008/10/27 14:53:57 bjeram Exp $"; 

static struct errorIdx *errIdx = NULL;
static int modCount=0; 

int readIDXFile (int fd, errorIdxEntry **erridx, int *count) 
{
    FILE *fp;
    char buff[101];
    int errCount=0;
    errorIdxEntry *err;
   
    fp=fdopen(fd,"r");
    err=*erridx = calloc(ALLOCSIZE,sizeof(errorIdxEntry));
    while (fgets(buff, 101, fp) != NULL) 
	{
	if (buff[0] == '#' || buff[0]=='\n')
	    continue;
	sscanf(buff,"%d %c %ld",&err[errCount].errNumber,\
	       &err[errCount].severity, &err[errCount].offset);
	errCount++;
	if (errCount%ALLOCSIZE == 0)
	    err = *erridx =
		realloc(*erridx,(ALLOCSIZE+errCount)*sizeof(errorIdxEntry));
	}
    *count = errCount;
    return lcubootOK;
}

int readERRORSFile (int fd, char **txt) 
{
    int nr, offset;
    char buff[512], *errTxt;
    
    offset=lseek(fd,0L,SEEK_END);
    lseek(fd,0L,SEEK_SET);
    if (offset == ERROR) 
	return lcubootERROR;
    errTxt = *txt = calloc(offset+1,sizeof(char));
    offset = 0;
    while ( (nr = read(fd, buff, sizeof(buff))) )
	{
	memcpy(errTxt+offset,buff,nr);
	offset += nr;
	}
    *(errTxt+offset+1)=0;
    return lcubootOK;
}   

int lcubootErrorLoad(const char *modName)
{
    char searchPath[256], fileName[100];
    int fd1, fd2, i=0;
    
    /* Define search path */
     if ((getenv("INTROOT") != NULL) && (getenv("VLTROOT") != NULL))
        {
        sprintf(searchPath,"%s/ERRORS:%s/ERRORS",getenv("INTROOT"),
		getenv("VLTROOT"));
        }
     else if (getenv("INTROOT") != NULL)
        {
        sprintf(searchPath,"%s/ERRORS",getenv("INTROOT"));
        }
     else if (getenv("VLTROOT") != NULL)
             {
             sprintf(searchPath,"%s/ERRORS",getenv("VLTROOT"));
             }
     else
        {
        searchPath[0] = '\0';
        }

    /* open ERRORS file */
    strcpy(fileName, modName);
    strcat(fileName, "_ERRORS");
    fd2 = lcubootFileOpen(searchPath,fileName,O_RDONLY);
    if (fd2 < 0) /* no ERRORS file, is OK */
	return lcubootOK;
    
    /* open error IDX file */
    strcpy(fileName,modName);
    strcat(fileName,"ERRORS.IDX");
    fd1 = lcubootFileOpen(searchPath,fileName,O_RDONLY);
    if (fd1 < 0) 
	{
	close(fd2);
	RETURN_ABORT("Failed to open errors IDX file for module", modName);
	}
    /* allocate space for index table */
    if (errIdx == NULL) 
	{
	errIdx = calloc(ALLOCSIZE,sizeof(struct errorIdx));
	if (errIdx == NULL) 
	    RETURN_ABORT("Unable to alloc memory. Loading module", modName);
	}
    if (modCount>0 && modCount%ALLOCSIZE == 0) 
	{
	errIdx = realloc(errIdx,(ALLOCSIZE+modCount)*sizeof(struct errorIdx));
	if (errIdx == NULL) 
	    RETURN_ABORT("Unable to realloc memory. Loading module", modName);
	}
    /* Search if module already loaded */
    for(i=0; i<modCount && strcmp(errIdx[i].modName,modName); i++)
	;
    if (i>=modCount) /* not found, new module */
	i = modCount;
    
    /* load module errors+IDX file */
    errIdx[i].modName = calloc(strlen(modName)+1,sizeof(char));
    strcpy(errIdx[i].modName,modName);
    readIDXFile(fd1, &errIdx[i].idxEntry, &errIdx[i].errCount);
    close(fd1);
    readERRORSFile(fd2,&errIdx[i].txtEntry);
    close(fd2);
    if (i>=modCount) /* a new module has been loaded */
	modCount++;
    
    return lcubootOK;
}

int lcubootErrorGetFormat(const char *modName, 
		       int errNumber, 
		       char *severity, 
		       char *format) 
{
    int i=0, j=0;
    errorIdxEntry *err;
    char *s, *t;
    
    if (errNumber <=0 || *modName==0)
	return lcubootERROR;

    while (i<modCount && strcmp(modName,errIdx[i].modName)) 
	i++;
    if (i>=modCount) 
	{
	lcubootLogMsg("lcuboot: %s: %s: %d=0x%x=\"%s\"\n",\
		      (int)"Module Not Found", (int)modName,
		      errno,errno,(int)strerror(errno), 0);
	return lcubootError_MOD_NOT_FOUND;
	}
    
    err = errIdx[i].idxEntry;
    if (err==NULL) 
	{
	lcubootLogMsg("lcuboot: %s: %s: %d=0x%x=\"%s\"\n",\
	       (int)"Internal error: Null pointer to IDX struct",
	       (int)modName,errno,errno,(int)strerror(errno), 0);
	return lcubootError_INTERNAL;
	}
    
    while(err[j].errNumber != errNumber && j < errIdx[i].errCount ) 
	j++;
    if (j >= errIdx[i].errCount) 
	{
	lcubootLogMsg("lcuboot: %s: %s: %d=0x%x=\"%s\"\n",\
	       (int)"Error number not found",
	       (int)modName,errno,errno,(int)strerror(errno), 0);
	return lcubootError_ERR_NOT_FOUND;
	}
    
    t = format;
    s = errIdx[i].txtEntry+err[j].offset;
    while (*s != '\n' && *s) 
	{
	*t++=*s++;
	}
    *t = 0;
    *severity=err[j].severity;
    return lcubootOK;
}
/*___oOo___*/
