/******************************************************************************
* E.S.O. - VLT project
*
* "@(#) $Id: lcubootAutoGen.c,v 1.2 2008/10/27 14:53:57 bjeram Exp $"
*
* who       when      what
* --------  --------  ----------------------------------------------
* bgustafs 2003-09-15 abort script if loaded module has unresolved symbols
* bgustafs 2001-02-01 abort lcubootAutoLoad if binary not found
* bgustafs 2000-12-20 added new function lcubootAutoLoadNoAbort which 
*                     does not abort if function to load is not found
* bgustafs 2000-09-29 reverted change of lcubootAutoLoad, now it only 
*                     returns ERROR if module not found
* bgustafs 2000-07-04 added shell abort if module not found, spwn or 
*                     exec failed, SPR980421, new function 
*                     lcubooAutoExecorig which works as the old 
*                     lcubootAutoExec
* S.Sandrock 23.04.1995 New
*/

/************************************************************************
*   NAME
*	lcubootAutoGen,
*	lcubootAutoProbe,
*	lcubootAutoLoad,
*	lcubootAutoExec,
*       lcubootAutoExecorig,
*	lcubootAutoSpawn,
*	lcubootAutoCd, lcubootAutoCdBoot
*	- Automatic LCU Module Installation Facility
*
*   SYNOPSIS
*	int lcubootAutoProbe(char *probeAddr0, ... *probeAddr9)
*	int lcubootAutoLoad(BOOL cond, const char *moduleName, 
*			    int reldFlag, int symFlag)
*	int lcubootAutoLoadNoAbort(BOOL cond, const char *moduleName, 
*			           int reldFlag, int symFlag)
*	int lcubootAutoExec(BOOL cond, const char *funcName, int arg1, ...arg8)
*	int lcubootAutoExecorig(BOOL cond, const char *funcName, int arg1, 
*                                ...arg8)
*	int lcubootAutoSpawn(BOOL cond, const char *taskName, 
*			int priority, int options, int stackSize,
*			const char *funcName, int arg1, ..arg4)
*	int lcubootAutoCd(BOOL cond, const char *searchPath, 
*			  const char *fileName)
*	int lcubootAutoCdBoot(BOOL cond, const char *scriptName)
*
*   DESCRIPTION
*	These functions support the automatic installation of all general
*	LCU modules. All functions are intended to be
*	used directly from the VxWorks shell in a boot-script.
*
*	The script is aborted when a fatal error condition occurs, which
*	is signalled as a log message in the form:
*
*	<tid> (tShell): lcuboot: <message>: <faulty item>:<errno | faulty item>
*	<tid> (tShell): --- SCRIPT ABORTED ---
*
*	lcubootAutoProbe - tests whether any hardware is present at the given 
*		addresses. Byte-read accesses are executed to verify this.
*		Up to ten addresses can be given, the function stops after
*		the first non-successful access.
*		It returns the number of successful accesses.
*
*		The counting begins normally from zero each time the function
*		is called. If `probeAddr0' is NULL then the counting is
*		continued from the last value, which is useful when more
*		than ten addresses must be probed.
*
*		This can be used to determine the number of boards/devices
*		that are present in the system.
*
*		Note that it cannot be checked whether the hardware found
*		at a given address is actually the "correct" one.
*		Standard pre-defined board addresses must be used as an
*		essential precondition.
*
*	lcubootAutoLoad - if `cond' is TRUE, then it loads the binary module
*		`moduleName', applying the search-path in the environment 
*		variable BINPATH.
*
*		It also loads the module's error-definition files, provided 
*		they exist. It is silently ignored if no error-files are found.
*		See lcubootErrorLoad(1) for details.
*
*		No operation is done and OK is returned for FALSE `cond'.
*		The same is true if a module with this name is already loaded
*		and `reldFlag' is zero, otherwise the module is re-loaded.
*		`symFlag' is usually zero and can be omitted, see loadLib(1).
*		It returns OK if the module is found and loaded.
*		It returns ERROR if the module is not found, however,
*		the script is not aborted then. It aborts the script 
*		only if the module is found but cannot be loaded properly.
*
*		This can be used to conditionally/optionally load a module.
*
*	lcubootAutoLoadNoAbort - same as lcubootAutoLoad except it does not 
*               abort the shell script in case the module to load is not
*               found.
*
*	lcubootAutoExec - if `cond' is TRUE, then it executes the given 
*		function with the given arguments.
*
*		No operation is done and OK is returned for FALSE `cond'.
*		It returns the called function's return-value, or ERROR
*		if the function is not found in the symbol-table.
*
*		This can be used to conditionally/optionally install a module.
*               The shell script is aborted if the function is not found or
*               if the function does not return the value 0 or 2 (SUCCESS).
*
*	lcubootAutoExecorig - if `cond' is TRUE, then it executes the given 
*		function with the given arguments. This function works as
*               the old lcubootAutoExec - no script abort in case of error.
*
*		No operation is done and OK is returned for FALSE `cond'.
*		It returns the called function's return-value, or ERROR
*		if the function is not found in the symbol-table.
*
*		This can be used to conditionally/optionally install a module.
*
*	lcubootAutoSpawn - if `cond' is TRUE, then it spawns `funcName' as a
*		task with the name `taskName', using the given parameters,
*		as in taskSpawn.
*
*		No operation is done and OK is returned for FALSE `cond'.
*		It returns the return-value of the taskSpawn call, or 
*		ERROR if the function is not found in the symbol-table.
*
*		Note that fewer arguments can be passed to the task compared
*		to a direct call of taskSpawn.
*
*		This can be used to conditionally/optionally spawn a task.
*               The shell script is aborted if the function is not found or
*               if the task can't be spawned.
*
*	lcubootAutoCd - if `cond' is TRUE, then it changes to the directory
*		in which a given file is found. The file is searched applying
*		the given colon-separated directory search-path.
*
*		No operation is done and OK is returned for FALSE `cond'.
*		It returns the OK, or ERROR in case of any failure.
*		Note that after calling this function the current working
*		directory is not reset to it's previous value.
*
*	lcubootAutoCdBoot - is similar to `lcubootAutoCd', but specifically
*		intended for module-boot-scripts. It searches for `scriptName'
*		applying the search-path in the environment variable BOOTPATH.
*
*		The script is aborted when the module-boot-script could not
*		be found in any of these directories.
*
*   VARIABLES
*	lcubootProbeCount - counts the valid addresses in `lcubootAutoProbe'
*
*   FILES
*	See lcubootErrorLoad(1) for error-definition files.
*
*   ENVIRONMENT
*	BINPATH  - colon-separated directory searchpath for binary modules
*	BOOTPATH - colon-separated directory searchpath for module-boot-scripts
*
*   COMMANDS
*
*   RETURN VALUES
*
*   CAUTIONS 
*	Most functions are not reentrant and should therefore only be used
*	from LCU boot-scripts, where reentrancy is not important!
*
*   EXAMPLES
*	The first example searches for the module-boot-script of "lqs" 
*       and executes it under the shell:
*
*       lcubootAutoCdBoot 1,"lqs.boot"
*       < lqs.boot
*
*	This module-boot-script could then have the following contents.
*	It automatically loads and sets up the "lqs" module.
*	After that the found lqs task is spawned.
*
*	lcubootAutoLoad 1,"lqs"
*	
*	# Define Ping and Ack timeout(seconds)
*	lcubootAutoExec 1,"lqsInit",10,"rtap"
*	
*	lcubootAutoExec 1,"lqsAddEnvTbl","wte13qs","te13","134.171.12.222",2223
*	lcubootAutoExec 1,"lqsAddEnvTbl","wte19qs","te19","134.171.12.228",2223
*	lcubootAutoExec 1,"lqsAddEnvTbl","wte13",  "te13","134.171.12.222",2155
*	lcubootAutoExec 1,"lqsAddEnvTbl","wte16",  "te16","134.171.12.225",2167
*	 
*	# Spawn the lqs task
*	lcubootAutoSpawn 1, "tLqs", 90, 0x18, 20000, "lqs"
*	
*
*   SEE ALSO
*	lcubootAutoEnv(1), lcubootAutoDrv(1), lcubootAutoLcc(1), 
*	lcubootErrorLoad(1),
*	loadLib(1), symLib(1),
*	ld(2)
*
*   BUGS   
* 
*------------------------------------------------------------------------
*/

#include "lcubootPrivate.h"
#include "usrLib.h"
#include "unldLib.h"


/*
 * Global data:
 */
int lcubootProbeCount = 0; /* number of found addresses in lcubootAutoProbe */

int lcubootAutoProbe(char *probeAddr0,
		    char *probeAddr1,
		    char *probeAddr2,
		    char *probeAddr3,
		    char *probeAddr4,
		    char *probeAddr5,
		    char *probeAddr6,
		    char *probeAddr7,
		    char *probeAddr8,
		    char *probeAddr9)
{
    if (probeAddr0 != NULL) /* i.e. start counting form zero */
	{
	lcubootProbeCount = 0;
	if (lcubootProbeAddress(probeAddr0) != lcubootOK)
	    return lcubootProbeCount;
	++lcubootProbeCount;
	}
    if (probeAddr1 == NULL || lcubootProbeAddress(probeAddr1) != lcubootOK)
	return lcubootProbeCount;
    ++lcubootProbeCount;
    if (probeAddr2 == NULL || lcubootProbeAddress(probeAddr2) != lcubootOK)
	return lcubootProbeCount;
    ++lcubootProbeCount;
    if (probeAddr3 == NULL || lcubootProbeAddress(probeAddr3) != lcubootOK)
	return lcubootProbeCount;
    ++lcubootProbeCount;
    if (probeAddr4 == NULL || lcubootProbeAddress(probeAddr4) != lcubootOK)
	return lcubootProbeCount;
    ++lcubootProbeCount;
    if (probeAddr5 == NULL || lcubootProbeAddress(probeAddr5) != lcubootOK)
	return lcubootProbeCount;
    ++lcubootProbeCount;
    if (probeAddr6 == NULL || lcubootProbeAddress(probeAddr6) != lcubootOK)
	return lcubootProbeCount;
    ++lcubootProbeCount;
    if (probeAddr7 == NULL || lcubootProbeAddress(probeAddr7) != lcubootOK)
	return lcubootProbeCount;
    ++lcubootProbeCount;
    if (probeAddr8 == NULL || lcubootProbeAddress(probeAddr8) != lcubootOK)
	return lcubootProbeCount;
    ++lcubootProbeCount;
    if (probeAddr9 == NULL || lcubootProbeAddress(probeAddr9) != lcubootOK)
	return lcubootProbeCount;
    ++lcubootProbeCount;
    return lcubootProbeCount;
}


static int lcubootAutoLoadInternal(BOOL cond, BOOL abort,
				   const char *moduleName, int reldFlag, int symFlag)
{
    int fd = -1;
    MODULE_ID modId;

    /*
     * Don't do anything if `cond' is FALSE:
     */
    if (! cond) return lcubootOK;

    lcubootLogMsg("lcubootAutoLoad: Loading module %s, flags: %d,%d\n",
		  (int)moduleName,reldFlag, symFlag,0,0,0);
    /*
     * Check if a module with this name already exists:
     */
    modId = moduleFindByName((char *)moduleName);
    if (modId != NULL && reldFlag == 0)
	return lcubootOK;	/* do nothing if reloading is not requested */

    /*
     * Search for the object-file in BINPATH and open it for reading:
     */
    fd = lcubootFileOpen(getenv("BINPATH"), moduleName, O_RDONLY);
    if (fd < 0)
	{
	if (abort != TRUE)
	    {
	    return lcubootERROR; 
	    }
	else
	    {
	    RETURN_ABORT("Module not found: ", moduleName); 
	    }
	}

    /*
     * If reloading is requested then unload first (not for lcuboot itself):
     */
    if (modId != NULL && reldFlag != 0 && strcmp(moduleName,"lcuboot") != 0)
	{
	if (unldByModuleId(modId, 0) == ERROR)
	    RETURN_ABORT("unload failed for module", moduleName);
	}
    /*
     * Load the object-file's contents, first set errno to 0
     */
    errno = 0;
    modId = loadModule(fd, symFlag);
    close(fd);
    if (modId == NULL)
	RETURN_ABORT("load failed for module", moduleName);
    /*
     * Check if module loaded with unresolved sysmbols
     */
    if (errno == S_symLib_SYMBOL_NOT_FOUND)
	{
	unldByModuleId(modId, 0);
	RETURN_ABORT("unresolved symbols for module", moduleName);
	}
    /*  
     * Load module's ERRORS file   
     */
    lcubootErrorLoad(moduleName);
    return lcubootOK;
}

int lcubootAutoLoad(BOOL cond, 
		    const char *moduleName, int reldFlag, int symFlag)
{

    return lcubootAutoLoadInternal(cond, TRUE, moduleName, reldFlag, symFlag);
}

int lcubootAutoLoadNoAbort(BOOL cond, 
			   const char *moduleName, int reldFlag, int symFlag)
{

    return lcubootAutoLoadInternal(cond, FALSE, moduleName, reldFlag, symFlag);
}


int lcubootAutoExec(BOOL cond, const char *funcName, 
		   int arg1, int arg2, int arg3, int arg4, int arg5,
		   int arg6, int arg7, int arg8)
{
    FUNCPTR funcPtr = NULL;
    int stat;
    char msg[32];

    /*
     * Don't do anything if `cond' is FALSE:
     */
    if (! cond) return lcubootOK;

    lcubootLogMsg("Exec %s\n",(int)funcName,0,0,0,0,0);
    /*
     * Search for the function in the system symbol-table:
     */
    if (lccFindFunctionEntry(funcName,&funcPtr) == ERROR)
	RETURN_ABORT("Function not found: ", funcName);
/*	return lcubootERROR; */
    if (funcPtr == NULL) 
	RETURN_ABORT("Function not found: ", funcName);
/*return lcubootERROR; */

    /*
     * Call the function:
     */
/*    return */
    stat = (*funcPtr)(arg1,arg2,arg3,arg4,arg5,arg6,arg7,arg8);
    if ((stat != 0) && (stat != 2))
	{
	sprintf (msg, "%s, return code %i", funcName, stat);
	RETURN_ABORT("Function failed: ", msg);
	}
    
    return stat;
}

int lcubootAutoExecorig(BOOL cond, const char *funcName, 
			int arg1, int arg2, int arg3, int arg4, int arg5,
			int arg6, int arg7, int arg8)
{
    FUNCPTR funcPtr = NULL;

    /*
     * Don't do anything if `cond' is FALSE:
     */
    if (! cond) return lcubootOK;

    lcubootLogMsg("Exec %s\n",(int)funcName,0,0,0,0,0);
    /*
     * Search for the function in the system symbol-table:
     */
    if (lccFindFunctionEntry(funcName,&funcPtr) == ERROR)
        return lcubootERROR;
    if (funcPtr == NULL) return lcubootERROR;

    /*
     * Call the function:
     */
    return (*funcPtr)(arg1,arg2,arg3,arg4,arg5,arg6,arg7,arg8);
}


#define HUNDRED_KILO_BYTES 102400

int lcubootAutoSpawn(BOOL cond, const char *taskName, 
		     int priority, int options, int stackSize,
		     const char *funcName, 
		     int arg1, int arg2, int arg3, int arg4)
{
    FUNCPTR funcPtr = NULL;
    int tid;

    /*
     * Don't do anything if `cond' is FALSE:
     */
    if (! cond) return lcubootOK;

    lcubootLogMsg("Spawn %s Task:%s (Prio:%d Opt:%d Stack:%d)\n",
		  (int)taskName,(int)funcName,priority,options,stackSize,0);

    /*
     * Search for the function in the system symbol-table:
     */
    if (lccFindFunctionEntry((char *)funcName,&funcPtr) == ERROR || funcPtr == NULL)
	RETURN_ABORT("Function not found: ", funcName);

    /*
     * SPR20020284: Check stack size for '*InitAll' min. 100kB
     */
    if (strstr(funcName,"InitAll") != NULL)
	{
	if (stackSize < HUNDRED_KILO_BYTES) stackSize = HUNDRED_KILO_BYTES;
	}

    /*
     * Spawn the function as a task:
     */
    tid = taskSpawn((char *)taskName, priority, options, stackSize, funcPtr,
		  arg1, arg2, arg3, arg4, 0,0,0,0,0,0);
    if (tid == ERROR)
	RETURN_ABORT("Failed to spawn function: ", funcName);

    return tid;
}


int lcubootAutoCd(BOOL cond,
		  const char *searchPath,
		  const char *fileName)
{
    int fd, err;
    char absName[256] = "", *p;

    /*
     * Don't do anything if `cond' is FALSE:
     */
    if (! cond) return lcubootOK;

    /*
     * Search for the file and open it for reading:
     */
    fd = lcubootFileOpen(searchPath, fileName, O_RDONLY);
    if (fd < 0) return lcubootERROR;

    /*
     * Determine the absolute name and directory of the found file:
     */
    err = ioctl(fd, FIOGETNAME, (int)&absName);
    close(fd);
    if (err != OK) return lcubootERROR;
    p = strrchr(absName, '/');
    if (p == NULL) return lcubootERROR;

    /*
     * Isolate directory and do a cd to it:
     */
    *p = '\0';
    if (cd(absName) != OK) return lcubootERROR;

    return lcubootOK;
}


int lcubootAutoCdBoot(BOOL cond,
		      const char *scriptName)
{
    lcubootLogMsg("lcubootAutoCdBoot: %s\n",(int)scriptName,0,0,0,0,0);
    /*
     * search module-boot-script in BOOTPATH:
     */
    if (lcubootAutoCd(cond, getenv("BOOTPATH"), scriptName) == lcubootERROR)
	RETURN_ABORT("could not find module-boot-script", scriptName);

    return lcubootOK;
}


/*___oOo___*/
