/******************************************************************************
* E.S.O. - VLT project
*
* "@(#) $Id: lcubootAutoEnv.c,v 1.2 2008/10/27 14:53:57 bjeram Exp $"
*
* who        when       what
* ---------  ---------- ---------------------------------------------------
* S.Sandrock 23.04.1995 New
* S.Sandrock 07.11.1995 adapted to new VLTROOT/DATA structure
* S.Sandrock 04.12.1995 added CONFIGPATH
* S.Sandrock 17.03.1996 added MODROOT handling
* B.Gustafsson 2001-05-28 added ACSROOT and ACE_ROOT_VW to BINPATH
* B.Jeram    2004-09-20 change ACE+TAO library search path
* bgustafs   2004-10-05 added ACSROOT to BOOTPATH and CONFIGPATH
*/

/************************************************************************
*   NAME
*	lcubootAutoEnv,
*	lcubootAutoEnvInit
*	- Automatic LCU Environment Setup Facility
*
*   SYNOPSIS
*	int lcubootAutoEnvInit(void)
*
*   DESCRIPTION
*	These functions support the automatic setup of LCU shell environment.
*	All functions are intended to be used directly from the VxWorks shell 
*	in a boot-script.
*
*	The script is aborted when a fatal error condition occurs, which
*	is signalled as a log message in the form:
*
*	<tid> (tShell): lcuboot: <message>: <faulty item>:<errno | faulty item>
*	<tid> (tShell): --- SCRIPT ABORTED ---
*
*	lcubootAutoEnvInit - initialize shell environment variable from the
*		LCU boot-line settings and set the shell-prompt. 
*
*		The following variables are derived from the LCU boot-line
*		or derived from other variables, but only if they are not 
*		already defined:
*
*			LOCALHOST   := <bootParams.targetName>
*			LOCALIPADDR := <bootParams.ead>
*			LOCALENV    := l${LOCALHOST}
*			HOSTNAME    := <bootParams.hostName>
*			HOSTIPADDR  := <bootParams.had>
*			HOSTENV     := w${HOSTNAME}
*
*			BOOTHOME    (see below)
*			BOOTROOT    (see below)
*			BOOTDB      := ${BOOTHOME}/DB
*			LOGFILE     := ${BOOTHOME}/logfile
*
*		BOOTHOME corresponds to the directory of the startup-script
*		as specified in the boot-line. BOOTROOT is derived from that
*		by going two level up in the directory hierarchy. This should
*		be the same then as VLTDATA on the host workstation.
*
*		Default search-paths are also set for binary files (BINPATH)
*		and module-boot-scripts (BOOTPATH). The parts with INTROOT
*		respectively MODROOT are omitted when not defined. The 
*		CPU-dependent extension is omitted when CPU is not defined.
*
*		BINPATH  := $BOOTHOME:\
*			    [$MODROOT/bin:]\		(if MODROOT defined)
*			    [$INTROOT/vw/bin[/$CPU]:]\	(if INTROOT defined)
*			    $VLTROOT/vw/bin[/$CPU]
*
*		BOOTPATH := $BOOTHOME:\
*			    $BOOTROOT/config:\
*			    [$MODROOT/bin:]\		(if MODROOT defined)
*			    [$INTROOT/vw/bin/$CPU:]\	(if INTROOT/CPU def.)
*			    [$INTROOT/vw/bin:]\		(if INTROOT defined)
*			    $VLTROOT/vw/bin/$CPU:\	(if CPU defined)
*			    $VLTROOT/vw/bin
*
*		CONFIGPATH := $BOOTHOME:\
*			      $BOOTROOT/config:\
*			      [$MODROOT/config:]\	(if MODROOT defined)
*			      [$INTROOT/config:]\	(if INTROOT defined)
*			      $VLTROOT/config
*
*		The shell-prompt is derived from the local environment name.
*
*		Note that VLTROOT, INTROOT and MODROOT are not set!
*
*   VARIABLES
*
*   FILES
*
*   ENVIRONMENT
*	`lcubootAutoEnvInit' sets the above stated shell variables.
*	VLTROOT and optionally INTROOT/MODROOT must be set before calling it.
*
*	CPU can be optionally set to extend BINPATH and BOOTPATH.
*	Values are the same as for the VxWorks compiler, e.g. MC68040.
*
*   COMMANDS
*
*   RETURN VALUES
*
*   CAUTIONS 
*	Most functions are not reentrant and should therefore only be used
*	from LCU boot-scripts, where reentrancy is not important!
*
*   COMPATIBILITY
*	Observe the following changes after the JUL95 VLT SW Release:
*	- The structure under BOOTROOT has been changed.
*	- The default HOSTENV is no longer appended by "qs".
*
*   EXAMPLES
*
*   SEE ALSO
*	lcubootAutoGen(1), lcubootAutoDrv(1), lcubootAutoLcc(1), 
*	symLib(1)
*
*   BUGS   
* 
*------------------------------------------------------------------------
*/

#include "lcubootPrivate.h"


/*
 * Execute putenv for `str', abort at failure, log success:
 */
#define PUTENV(str)\
{\
	if (putenv(str) == ERROR)\
	    RETURN_ABORT("could not define variable", str);\
	printf("lcubootAutoEnvInit: \"%s\"\n", str);\
	lcubootLogMsg("lcubootAutoEnvInit: \"%s\"\n", (int)str,0,0,0,0,0);\
}

/*
 * Define `var' by concatenating `prefix' `member' `postfix':
 */
#define SETVAR(str,var,prefix,member,postfix)\
	strcpy(str, var),\
	strcat(str, "="),\
	strcat(str, prefix),\
	strcat(str, member),\
	strcat(str, postfix)\


int lcubootAutoEnvInit(void)
{
    extern BOOT_PARAMS sysBootParams;
    BOOT_PARAMS bootParams = sysBootParams;

    char *str = malloc(1024);
    const char *var;

    lcubootLogMsg("lcubootAutoEnvInit\n",0,0,0,0,0,0);

    if (str == NULL)
	RETURN_ABORT("could not allocate memory", "");

    /*
     * Set env-variables from boot-line:
     */
    if (getenv(var = "LOCALHOST") == NULL)
	{
	SETVAR(str,var,"",bootParams.targetName,"");
	PUTENV(str);
	}

    if (getenv(var = "LOCALIPADDR") == NULL)
	{
	SETVAR(str,var,"",bootParams.ead,"");
	PUTENV(str);
	}

    if (getenv(var = "LOCALENV") == NULL)
	{
	SETVAR(str,var,"l",getenv("LOCALHOST"),"");
	PUTENV(str);
	}

    if (getenv(var = "HOSTNAME") == NULL)
	{
	SETVAR(str,var,"",bootParams.hostName,"");
	PUTENV(str);
	}

    if (getenv(var = "HOSTIPADDR") == NULL)
	{
	SETVAR(str,var,"",bootParams.had,"");
	PUTENV(str);
	}

    if (getenv(var = "HOSTENV") == NULL)
	{
	SETVAR(str,var,"w",getenv("HOSTNAME"),"");
	PUTENV(str);
	}

    /*
     * Set other derived env-variables:
     */
    if (getenv(var = "BOOTHOME") == NULL)
	{
	char *tmp;
	/* isolate directory of the bootScript: */
	SETVAR(str,var,"",bootParams.startupScript,"");
	if ((tmp = strrchr(str, '/')) != NULL)
	    *tmp = EOS;
	PUTENV(str);
	}

    if (getenv(var = "BOOTROOT") == NULL)
	{
	char *tmp;
	/* from BOOTHOME, go 2 levels up: */
	SETVAR(str,var,getenv("BOOTHOME"),"","");
	if ((tmp = strrchr(str, '/')) != NULL)
	    *tmp = EOS;
	if ((tmp = strrchr(str, '/')) != NULL)
	    *tmp = EOS;
	PUTENV(str);
	}

    if (getenv(var = "BOOTDB") == NULL)
	{
	SETVAR(str,var,getenv("BOOTHOME"),"/DB","");
	PUTENV(str);
	}

    if (getenv(var = "LOGFILE") == NULL)
	{
	SETVAR(str,var,getenv("BOOTHOME"),"/logfile","");
	PUTENV(str);
	}

    /*
     * BINPATH   (see man-page)
     * BOOTPATH  (see man-page)
     */
    if (getenv(var = "BINPATH") == NULL)
	{
	const char *tmp;
	/* assemble colon-separated list of directories: */
	SETVAR(str,var,getenv("BOOTHOME"),"","");
	if ((tmp = getenv("MODROOT")) != NULL)
	    {
	    strcat(str, ":"), strcat(str, tmp), strcat(str, "/bin");
	    }
	if ((tmp = getenv("INTROOT")) != NULL)
	    {
	    strcat(str, ":"), strcat(str, tmp), strcat(str, "/vw/bin");
	    if ((tmp = getenv("CPU")) != NULL)
		strcat(str, "/"), strcat(str, tmp);
	    }
	if ((tmp = getenv("ACSROOT")) != NULL)
	    {
	    strcat(str, ":"), strcat(str, tmp), strcat(str, "/vw/bin");
	    if ((tmp = getenv("CPU")) != NULL)
		strcat(str, "/"), strcat(str, tmp);
	    }
	if ((tmp = getenv("ACE_ROOT_VW")) != NULL)
	    {
	    strcat(str, ":"), strcat(str, tmp), strcat(str, "/lib");
	    }
	if ((tmp = getenv("VLTROOT")) != NULL)
	    {
	    strcat(str, ":"), strcat(str, tmp), strcat(str, "/vw/bin");
	    if ((tmp = getenv("CPU")) != NULL)
		strcat(str, "/"), strcat(str, tmp);
	    }
	PUTENV(str);
	}

    if (getenv(var = "BOOTPATH") == NULL)
	{
	const char *tmp1, *tmp2;
	/* assemble colon-separated list of directories: */
	SETVAR(str,var,getenv("BOOTHOME"),":","");
	strcat(str, getenv("BOOTROOT")), strcat(str, "/config");
	if ((tmp1 = getenv("MODROOT")) != NULL)
	    {
	    strcat(str, ":"), strcat(str, tmp1), strcat(str, "/bin");
	    }
	if ((tmp1 = getenv("INTROOT")) != NULL)
	    {
	    strcat(str, ":"), strcat(str, tmp1), strcat(str, "/vw/bin");
	    if ((tmp2 = getenv("CPU")) != NULL)
		{
		strcat(str, "/"), strcat(str, tmp2);
		strcat(str, ":"), strcat(str, tmp1), strcat(str, "/vw/bin");
		}
	    }
	if ((tmp1 = getenv("ACSROOT")) != NULL)
	    {
	    strcat(str, ":"), strcat(str, tmp1), strcat(str, "/vw/bin");
	    if ((tmp2 = getenv("CPU")) != NULL)
		{
		strcat(str, "/"), strcat(str, tmp2);
		strcat(str, ":"), strcat(str, tmp1), strcat(str, "/vw/bin");
		}
	    }
	if ((tmp1 = getenv("VLTROOT")) != NULL)
	    {
	    strcat(str, ":"), strcat(str, tmp1), strcat(str, "/vw/bin");
	    if ((tmp2 = getenv("CPU")) != NULL)
		{
		strcat(str, "/"), strcat(str, tmp2);
		strcat(str, ":"), strcat(str, tmp1), strcat(str, "/vw/bin");
		}
	    }
	PUTENV(str);
	}

    if (getenv(var = "CONFIGPATH") == NULL)
	{
	const char *tmp;
	/* assemble colon-separated list of directories: */
	SETVAR(str,var,getenv("BOOTHOME"),":","");
	strcat(str, getenv("BOOTROOT")), strcat(str, "/config");
	if ((tmp = getenv("MODROOT")) != NULL)
	    {
	    strcat(str, ":"), strcat(str, tmp), strcat(str, "/config");
	    }
	if ((tmp = getenv("INTROOT")) != NULL)
	    {
	    strcat(str, ":"), strcat(str, tmp), strcat(str, "/config");
	    }
	if ((tmp = getenv("ACSROOT")) != NULL)
	    {
	    strcat(str, ":"), strcat(str, tmp), strcat(str, "/config");
	    }
	if ((tmp = getenv("VLTROOT")) != NULL)
	    {
	    strcat(str, ":"), strcat(str, tmp), strcat(str, "/config");
	    }
	PUTENV(str);
	}

    /*
     * Set LCU prompt:
     */
    strcpy(str, getenv("LOCALENV"));
    strcat(str, "->");
    shellPromptSet(str);
    printf("lcubootAutoEnvInit: shell-prompt = \"%s\"\n", str);
    lcubootLogMsg("lcubootAutoEnvInit: shell-prompt = \"%s\"\n", (int)str,
		  0,0,0,0,0);

    free(str);

    return lcubootOK;
}


/*___oOo___*/
