/* rlogLib.c - remote login library */

/* 
 * Copyright (c) 1984-2006, 2009 Wind River Systems, Inc. 
 *
 * The right to copy, distribute, modify or otherwise make use of this software
 * may be licensed only pursuant to the terms of an applicable Wind River
 * license agreement. 
 */

/*
modification history
--------------------
05l,11jun09,ggz  Defect WIND00168165. Backported the fix for defect WIND00169357.   
                 Fixed compiler warnings.
05k,02jun09,dlk  LP64 update. Also fix SMP session shutdown issue WIND00169357.
                 Move private types and functions out of rlogLib.h, to here.
		 Add warning that rlogin() is not task safe.
05j,14apr09,v_r  Included shellLibP.h for shellParserControl prototype.
05i,31mar09,zl   updated jobAdd() arguments.
05h,26dec06,kch  Removed references to the obsolete coreip virtual stack from
                 module description.
05g,12aug06,tkf  Clean up for integration with IPNET.
05f,04feb06,dlk  Fix user-side build warning.
05e,03jan06,dlk  Treat ECONNABORTED/ENOBUFS errnos from accept() as non-fatal.
05d,26aug05,vvv  fixed crash when connecting from PuTTY (SPR #111566)
05c,15jul05,vvv  fixed warning
05b,10jul05,vvv  fixed warnings
05a,28jun05,vvv  restored AI_V4MAPPED
04z,06jun05,vvv  fixed rlogin server to work even when kernel shell has not
                 been started (SPR #109449)
04y,23may05,vvv  removed AI_V4MAPPED
04x,16feb05,vvv  include netLib.h only in kernel
04w,03feb05,aeg  included sysLib.h & strings.h for RTP builds (SPR #105335)
04v,12jan05,vvv  osdep.h cleanup
04u,30aug04,dlk  Replace LOG_ERROR() with log_err(), etc..
04t,14sep04,niq  virtual stack related backwards compatibility changes
04s,24aug04,ann  merged from COMP_WN_IPV6_BASE6_ITER5_TO_UNIFIED_PRE_MERGE
04r,09jul04,rp   uninstalling of logout routine not required, REMOTE_STOP does
                 that job
04q,01jul04,vvv  fixed warnings
04p,22apr04,xli  fix gnu compiler warnings
04o,21apr04,wap  Change default rlogind task priority (SPR #96139)
04n,16apr04,asr  roll back 04m modification
04m,01apr04,asr  changes to ensure that rlogin works when compiled 
                 with -DINET6 even when IPv6 support may not be enabled 
                 in the stack.
04l,17mar04,rp   updated to build for 5.5
04k,24feb04,asr  updates resulting from review comments
04j,02feb04,pas  added a forward declaration for rlogindSessionPtysCreate
04i,22jan04,asr  1)changes to port rlogLib to user space.
                 2)introduced use of application logger.
                 3)updated doc markups according to apigen
04h,12dec03,rp   support for multiple rlogin sessions
04g,20nov03,niq  osdep.h cleanup
04e,06nov03,rlm  Ran batch header update for header re-org.
04d,08sep03,vvv  merged from ACCORDION_BASE6_MERGE_BASELINE
04c,14apr03,ijm  fixed diab compiler warnings
04b,30jul02,ppp  merging fixes done to version 03l of veloce
04a,03jun02,ann  ported to clarinet from, T2.2 ver 03m.
                 extended rlogin() to provide IPv6 support. created a new
		 LOCAL routine rhostnameToAddr() to perfrom host name 
		 translations.
*/

/*
DESCRIPTION
This library provides a remote login facility for VxWorks based on the UNIX 
`rlogin' protocol (as implemented in UNIX BSD 4.3).  On a VxWorks terminal, 
this command gives users the ability to log in to remote systems on the 
network.  

Reciprocally, the remote login daemon, rlogind(), allows remote users to 
log in to VxWorks.  The daemon is started by calling rlogInit(), which is 
called automatically when INCLUDE_RLOGIN is defined.  The remote login 
daemon accepts remote login requests from another VxWorks or UNIX system, 
and causes the shell's input and output to be redirected to the remote user.

Internally, rlogind() provides a tty-like interface to the remote
user through the use of the VxWorks pseudo-terminal driver ptyDrv.

INCLUDE FILES: rlogLib.h

SEE ALSO:
ptyDrv, telnetLib, UNIX BSD 4.3 manual entries for `rlogin', `rlogind', 
and `pty'
*/

#define LOG_LOCATION LOG_LOC_FUNC

#ifdef _WRS_KERNEL
#include <vxWorks.h>
#include <ptyDrv.h>
#include <tyLib.h>
#include <logLib.h>
#include <shellLib.h>
#include <private/shellLibP.h>
#include <private/excLibP.h>
#include <lstLib.h>
#endif

#include <sysLib.h>
#include <stdio.h>
#include <string.h>
#include <errnoLib.h>
#include <taskLib.h>
#include <ioLib.h>
#include <inetLib.h>
#include <remLib.h>
#include <sockLib.h>
#include <hostLib.h>
#include <rlogLib.h>
#include <applUtilLib.h>

#ifndef _WRS_KERNEL
#include <strings.h>		/* bzero, bcopy */
#else
#include <netLib.h>
#endif /* _WRS_KERNEL */

#if _WRS_VXWORKS_MAJOR < 6 || (_WRS_VXWORKS_MAJOR == 6 && _WRS_VXWORKS_MINOR <= 8)
typedef int _Vx_usr_arg_t; 
#define TASK_ID_ERROR ((TASK_ID)(-1)) 
#endif

#define LOGIN_SERVICE		513
#define MAX_CONNECT_TRIES	5

#define ZERO ((_Vx_usr_arg_t) 0)

#if 0
#define RLOGIND_DEBUG(x) do {x;} while ((0))
#else
#define RLOGIND_DEBUG(x) do {} while ((0))
#endif

#ifndef PTY_DEVICE_NAME_MAX_LEN
#define PTY_DEVICE_NAME_MAX_LEN 128 /* The maximum string length of a pty device */
#endif

/* The maximum string length of pty i/o task name */
#ifndef IO_TASK_NAME_MAX_LEN
#define IO_TASK_NAME_MAX_LEN 128
#endif

/* command interpreter control operations */

#ifndef REMOTE_START
#define REMOTE_START 0
#endif
#ifndef REMOTE_STOP
#define REMOTE_STOP 1
#endif

#ifdef _WRS_KERNEL

/* type of shellParserControl function */
#if (!defined (_WRS_VXWORKS_MAJOR) || (_WRS_VXWORKS_MAJOR < 6))
typedef STATUS (*SPC_FUNC) (UINT32 remoteEvent,
			    UINT32 sessionId,
			    UINT32 slaveFd);
#else
typedef STATUS (*SPC_FUNC) (UINT32 remoteEvent,
			    UINT32 sessionId,
			    UINT32 slaveFd,
			    VOIDFUNCPTR logoutFunc);
#endif

typedef UINT32 RLOGIND_REF_TYPE;

/* More than one of these flags can occur or'ed together */
#define RLOGIND_REF_MAIN        0x01U  /* Reference from rlogind() */
#define RLOGIND_REF_IN		0x02U  /* Reference from rlogInTask() */
#define RLOGIND_REF_OUT		0x04U  /* Reference from rlogOutTask() */
#define RLOGIND_REF_SHELL	0x08U  /* Disconnect call from rlogExit() */
#define RLOGIND_REF_LOGIN	0x10U  /* guards slaveFd during login */
#define RLOGIND_REF_EXCTASK	0x20U  /* Reference from tExcTask after
					  disconnect call from rlogExit() */

#define RLOGIND_REF_SOCKET	0x40U  /* socket needs shutdown() */

typedef enum _RLOGIND_LOGIN_STATE
    {
    NEVER_LOGGED_IN,
    LOGGED_IN_NOW,
    WAS_LOGGED_IN
    } RLOGIND_LOGIN_STATE;

/* rlogin server's session data. */
typedef struct
    {
    NODE        node;                /* for link-listing */
    char        ptyRemoteName[PTY_DEVICE_NAME_MAX_LEN];  
                                     /* name of the pty for this session */
    int         socket;              /* socket connection */
    int         inputFd;             /* input to command interpreter */
    int         outputFd;            /* output to command interpreter */
    int         slaveFd;             /* command interpreter side fd of pty */
    TASK_ID     outputTask;          /* Task ID to send data to socket */
    TASK_ID     inputTask;           /* Task ID to send data to socket */
    SPC_FUNC    parserControl;       /* source of input/output files */
    BOOL        busyFlag;            /* input/output tasks in use? */
    RLOGIND_LOGIN_STATE loggedIn;    /* Has a shell been started on this slot? */
    ULONG       userData;            /* A storage for application developers */
    TASK_ID     shellTaskId;         /* TaskID of shell associated with session */
    RLOGIND_REF_TYPE refs;
    } RLOGIND_SESSION_DATA;

/* rlogin server's task data */
typedef struct
    {
    RLOGIND_SESSION_DATA *pSession;     /* link to session data for tasks */
    } RLOGIND_TASK_DATA;

#endif /* _WRS_KERNEL */

extern char tyDeleteLineChar; /* No API to read this... */
extern char tyEofChar;	      /* No API to read this in VxWorks 6.x */
#define tyEOFGet() tyEofChar;

const char *rlogTermType	= "dumb/9600";	/* default terminal type */

/* rlogin task parameters */

int rlogTaskPriority	= 55;		/* task priority of rlogin tasks */
#ifdef _WRS_KERNEL
  int rlogTaskOptions	= VX_SUPERVISOR_MODE | VX_UNBREAKABLE;
#else
  int rlogTaskOptions	= 0;
#endif

int rlogTaskStackSize   = 8*BUFSIZ;     /* 8k, stack size of rlogin tasks */

TASK_ID rlogindId        = TASK_ID_ERROR;	/* rlogind task ID */
TASK_ID rlogChildTaskId  = TASK_ID_ERROR;	/* rlogChildTask task ID */

int rlogindCurrentClients = 0;

/* externals */

#ifdef _WRS_KERNEL
IMPORT	int logFdFromRlogin;		/* fd of pty for rlogin */
#endif /* _WRS_KERNEL */

/* local variables */

/*
 * rlogindSessionList is a pointer to a linked list of active sessions.
 * This is maintained for cleanup during exit
 */


#ifdef _WRS_KERNEL
LOCAL LIST      rlogindSessionList;

/* rlogindTaskList - an array of all sessions (active or inactive) */

LOCAL RLOGIND_TASK_DATA * rlogindTaskList = NULL;

LOCAL int rlogindMaxClients = 2; /* Default limit for simultaneous sessions. */

LOCAL SEM_ID rlogindMutexSem;

LOCAL SPC_FUNC rlogindParserControl = NULL; /* Accesses command interpreter. */
#endif /* _WRS_KERNEL */

LOCAL const char *ptyRlogName  = "/pty/rlogin";

LOCAL BOOL rlogInitDone = FALSE;/* FALSE = not done */
LOCAL int rloginSocket;		/* rlogin socket */

/* forward static functions */

#ifdef _WRS_KERNEL
LOCAL RLOGIND_SESSION_DATA *rlogindSessionAdd (void);
LOCAL void rlogindTaskDelete (int);
LOCAL void rlogindSessionDisconnect (RLOGIND_SESSION_DATA *, UINT32 refs);
LOCAL STATUS rlogindIoTasksCreate (RLOGIND_SESSION_DATA *);
LOCAL STATUS rlogindSessionPtysCreate (RLOGIND_SESSION_DATA *pSlot);
LOCAL void rlogExit (ULONG sessionId);
LOCAL STATUS recvStr (int sd, char *buf, size_t bufSiz);
LOCAL STATUS rlogindSessionPtysCreate (RLOGIND_SESSION_DATA *pSlot);

LOCAL void rlogInTask (RLOGIND_SESSION_DATA *);
LOCAL void rlogOutTask (RLOGIND_SESSION_DATA *);
#endif /* _WRS_KERNEL */

LOCAL STATUS rhostnameToAddr (char * pHost, struct sockaddr * pHostAddr);

LOCAL void rlogChildTask (void);

/* Rlogin server is supported in kernel only */

/*******************************************************************************
*
* rlogInit - initialize the remote login facility
*
* This routine initializes the remote login facility in the kernel. It creates
* a pty (pseudo tty) device and spawns rlogind(). If INCLUDE_RLOGIN is
* included, rlogInit() is called automatically at boot time.
*
* RETURNS: OK or ERROR.
*
* SEE ALSO: ptyDrv
*/

STATUS rlogInit (void)
    {

    int count;

    if (rlogInitDone)
	{
	log_info (RLOGIN_LOG, "already initialized.");
	return OK;
	}

#ifdef _WRS_KERNEL
    if (ptyDrv () == ERROR)
	{
	log_err (RLOGIN_LOG | LOG_ERRNO, "unable to initialize ptyDrv().");
	return ERROR;
	}

    rlogindParserControl = shellParserControl; /* Not ideal for scalability */

    rlogindMutexSem = semMCreate (SEM_Q_PRIORITY | SEM_DELETE_SAFE |
                                  SEM_INVERSION_SAFE );
    if (rlogindMutexSem == NULL)
        {
	log_err (RLOGIN_LOG, "unable to create mutex");
        return ERROR;
        }

    rlogindTaskList = (RLOGIND_TASK_DATA *)calloc ((size_t)rlogindMaxClients,
						   sizeof (RLOGIND_TASK_DATA));
    if (rlogindTaskList == NULL)
        {
        semDelete (rlogindMutexSem);
	log_err (RLOGIN_LOG, "unable to allocate memory for server task");
        return ERROR;
        }

    /* Allocate all of the session data structures and initialize them */

    for (count = 0; count < rlogindMaxClients; count++)
        {
        rlogindTaskList[count].pSession =
            (RLOGIND_SESSION_DATA *)calloc ((size_t)1, sizeof (RLOGIND_SESSION_DATA));

        if (rlogindTaskList[count].pSession == NULL)
            {
            rlogindTaskDelete (count);
	    log_err (RLOGIN_LOG, "unable to allocate memory for session");
            return ERROR;
            }

        /*
         * Initialize all elements of the structure to sane values.
         * Note that we use  'ERROR' is used in some cases so that we
         *  can differentiate from valid i/o file descriptors such as stdin.
         *  Also we need invalid task id's for initialization so we can
         *  differentiate from taskIdSelf() since 0 is implied as our taskId
         */

        rlogindTaskList[count].pSession->socket         = ERROR;
        rlogindTaskList[count].pSession->inputFd        = ERROR;
        rlogindTaskList[count].pSession->outputFd       = ERROR;
        rlogindTaskList[count].pSession->slaveFd        = ERROR;
        rlogindTaskList[count].pSession->outputTask     = TASK_ID_ERROR;
        rlogindTaskList[count].pSession->inputTask      = TASK_ID_ERROR;
        rlogindTaskList[count].pSession->parserControl  = NULL;
        rlogindTaskList[count].pSession->busyFlag       = FALSE;
        rlogindTaskList[count].pSession->loggedIn       = NEVER_LOGGED_IN;
        }

    rlogindId = (TASK_ID)taskSpawn ("tRlogind", rlogTaskPriority,
			   rlogTaskOptions, (size_t)rlogTaskStackSize,
			   (FUNCPTR) rlogind, 
			   ZERO, ZERO, ZERO, ZERO, ZERO,
			   ZERO, ZERO, ZERO, ZERO, ZERO);

    if (rlogindId == TASK_ID_ERROR)
	{
	log_err (RLOGIN_LOG, "unable to spawn rlogind.");
	rlogindTaskDelete (count);
	return ERROR;
	}

#endif /* _WRS_KERNEL */

    rlogInitDone = TRUE;

    return OK;
    }

/* Rlogin server is supported in kernel only. Following #ifdef 
 * encloses rlogind() and all of its helper functions.
 */
#ifdef _WRS_KERNEL 

/*******************************************************************************
*
* rlogind - the VxWorks remote login daemon
*
* This routine provides a facility for remote users to log in to VxWorks over
* the network.  If INCLUDE_RLOGIN is defined, rlogind() is spawned by
* rlogInit() at boot time.
*
* Remote login requests will cause `stdin', `stdout', and `stderr' to be
* directed away from the console.  When the remote user disconnects,
* `stdin', `stdout', and `stderr' are restored, and the shell is restarted.
* The rlogind() routine uses the remote user verification protocol specified
* by the UNIX remote shell daemon documentation, but ignores all the
* information except the user name, which is used to set the VxWorks remote
* identity (see the manual entry for iam()).
*
* The remote login daemon requires the existence of a pseudo-terminal
* device, which is created by rlogInit() before rlogind() is spawned.  The
* rlogind() routine creates two child processes, `tRlogInTask' and
* `tRlogOutTask', whenever a remote user is logged in.  These processes exit
* when the remote connection is terminated.
*
* RETURNS: N/A
*
* SEE ALSO: rlogInit(), iam()
*/

void rlogind (void)
    {
    char remUser [MAX_IDENTITY_LEN];	/* remote user trying to log in */
    char buf [BUFSIZ];
    int optval = ERROR;
#ifdef INET6
    struct sockaddr_in6 myAddress;
    struct sockaddr_in6 clientAddress;
    int  family = AF_INET6;
#else
    struct sockaddr_in myAddress;
    struct sockaddr_in clientAddress;
    int  family = AF_INET;
#endif
    RLOGIND_SESSION_DATA *pSlot = NULL;
    STATUS result = ERROR, rc = ERROR;
    int clientAddrLen = ERROR;
    int client = ERROR;
    int sd = ERROR;

    /* open a socket and wait for a client */

    sd = socket (family, SOCK_STREAM, 0);

    if (sd == ERROR)
       {
       log_err(RLOGIN_LOG | LOG_ERRNO, "in socket()");
       return;
       }

    bzero ((char *) &myAddress, sizeof (myAddress));

#ifdef INET6
    myAddress.sin6_family = AF_INET6;
    myAddress.sin6_port   = htons (LOGIN_SERVICE);
    myAddress.sin6_len   = sizeof (struct sockaddr_in6);
#else
    myAddress.sin_family = AF_INET;
    myAddress.sin_port   = htons (LOGIN_SERVICE);
    myAddress.sin_len   = sizeof (struct sockaddr_in);
#endif

    optval = 1;
    setsockopt (sd, SOL_SOCKET, SO_REUSEADDR,
		(char*) &optval, (socklen_t)sizeof (optval));

    if (bind (sd, (struct sockaddr *) &myAddress,
	      (socklen_t)sizeof (myAddress)) == ERROR)
	{
	log_err (RLOGIN_LOG | LOG_ERRNO, "in bind()");
	goto rlogind_exit;
	}

    rc = listen (sd, 1);

    if (rc == ERROR)
       {
       log_err(RLOGIN_LOG | LOG_ERRNO, "in listen()");
       goto rlogind_exit;
       }

    FOREVER
	{
	errnoSet (OK);		/* clear errno for pretty i() display */

	/* now accept connection */

	clientAddrLen = sizeof (clientAddress);
	client = accept (sd, (struct sockaddr *)&clientAddress, &clientAddrLen);

	if (client == ERROR)
	    {
	    /*
	     * ECONNABORTED is returned if the client resets
	     * the completed connection before accept() returns it.
	     * ENOBUFS might also be returned in some cases; we
	     * treat this as non-fatal also. In either case,
	     * the resources associated with the client connection
	     * will be cleaned up by the lower-level code.
	     */
	    if (errno == ECONNABORTED || errno == ENOBUFS)
		continue;

	    log_err (RLOGIN_LOG | LOG_ERRNO, "in accept()");
	    taskDelay (sysClkRateGet()); /* DoS potential */
	    continue;    
	    }

	/* turn on KEEPALIVE so if the client crashes, we'll know about it */

	optval = 1;
	setsockopt (client, SOL_SOCKET, SO_KEEPALIVE,
		    (char*) &optval, (socklen_t)sizeof (optval));


	/* read in initial strings from remote rlogin */
        /* ignore stderr */
	if ((recvStr (client, buf, (size_t)BUFSIZ) == ERROR) ||  
/* get local user name */
	    (recvStr (client, remUser, (size_t)MAX_IDENTITY_LEN) == ERROR) ||
/* ignore remote user name*/
	    (recvStr (client, buf, (size_t)BUFSIZ) == ERROR) ||
/* ignore terminal stuff */
	    (recvStr (client, buf, (size_t)BUFSIZ) == ERROR))
	    {
	    close (client);
            log_err (RLOGIN_LOG, "recvStr return ERROR");
            continue;
	    }
        
	/*
         * Some clients may send some additional information
	 * in the message. For example, the PuTTY client sends
	 * the window information even before the server has asked
	 * for it. To avoid confusing the input and output tasks
	 * we drain the socket here. We set it to non-blocking first
	 * to avoid pending forever if there is no more data.
	 *
	 * TODO: This doesn't seem reliable. How can we tell when
	 * the 'additional information' ends and user input begins?
	 * What if we call read() before the 'additional information'
	 * arrives?
	 */

	ioctl (client, FIONBIO, (_Vx_usr_arg_t) &optval);
	while ((read (client, buf, (size_t)BUFSIZ)) > 0);

	/* Restore blocking behavior. */

	optval = 0;
	ioctl (client, FIONBIO, (_Vx_usr_arg_t) &optval);

	/* acknowledge connection */

	write (client, "", (size_t)1);

	/* Get a session entry for this  connection */

	pSlot = rlogindSessionAdd ();

	/* Check if the maximum number of connections has been reached. */

	if (pSlot == NULL)
	    {
	    fdprintf (client, "\r\nSorry, session limit reached.\r\n");

            log_info (RLOGIN_LOG, "session limit reached");

	    /* Prevent denial of service attack by waiting 5 seconds (?) */
	    /*
	     * This is dubious. We can't service new connections while
	     * we wait, which may be the intent of the attack. This
	     * does possibly preserve CPU time for the already connected
	     * clients, however.
	     */
	    taskDelay (sysClkRateGet () * 5);
	    close (client);
	    continue;
	    }

	pSlot->socket = client;

	/*
	 * Create the pseudo terminal for a session. The input and output
	 * tasks transfer data between the client socket (connected to the
	 * remote machine) and the master side of the PTY; the shell reads
	 * from and writes to the slave side of the PTY.
	 */

	pSlot->refs = RLOGIND_REF_MAIN | RLOGIND_REF_SOCKET;

	result = rlogindSessionPtysCreate (pSlot);

	if (result == OK)
	    {
	    pSlot->refs |= (RLOGIND_REF_OUT | RLOGIND_REF_IN);
	    result = rlogindIoTasksCreate (pSlot);
	    }

	if (result == ERROR)
	    {
	    fdprintf (client, "\r\nrlogind: Sorry, session limit reached. Unable to spawn tasks. errno %#x\r\n", errnoGet());

            log_info (RLOGIN_LOG, "session limit reached");

	    /* Prevent denial of service attack by waiting 5 seconds */

	    taskDelay (sysClkRateGet () * 5);
	    rlogindSessionDisconnect (pSlot, RLOGIND_REF_MAIN);
	    continue;
	    }

	/* flush out pty device */

	ioctl (pSlot->slaveFd, FIOFLUSH, 0);

        /*
         * File descriptors are available. Save the corresponding
         * parser control routine.
         */

        pSlot->parserControl = rlogindParserControl;

        /*
         * Tell the parser control function to activate the shell
         *  for this session.
         */

	semTake (rlogindMutexSem, WAIT_FOREVER);
	pSlot->refs |= RLOGIND_REF_LOGIN;
	semGive (rlogindMutexSem);
	

#if (!defined (_WRS_VXWORKS_MAJOR) || (_WRS_VXWORKS_MAJOR < 6))
        result = (*rlogindParserControl) (REMOTE_START,
                                          (UINT32)pSlot,
                                          pSlot->slaveFd);
#else
        result = (*rlogindParserControl) (REMOTE_START,
                                          (ULONG)pSlot,
                                          (UINT32)pSlot->slaveFd,
                                          rlogExit);
#endif

        if (result == OK)
            {
#if (!defined (_WRS_VXWORKS_MAJOR) || (_WRS_VXWORKS_MAJOR < 6))
            shellLogoutInstall((FUNCPTR)rlogExit, (UINT32)pSlot);
#endif

	    semTake (rlogindMutexSem, WAIT_FOREVER);
	    /*
	     * Login was successful, the shell was spawned,
	     * and we will need to call shellParserControl (REMOTE_STOP, ...)
	     * later.
	     */
	    pSlot->loggedIn = LOGGED_IN_NOW;
	    pSlot->refs &= ~RLOGIND_REF_LOGIN;

	    /*
	     * This handles the case that the remote end
	     * disconnects before we took rlogindMutexSem()
	     * just above.
	     */
	    if ((pSlot->refs & RLOGIND_REF_SOCKET) == 0)
		result = ERROR;
	    else
		pSlot->refs &= ~RLOGIND_REF_MAIN;  /* not the last reference */

	    semGive (rlogindMutexSem);
	    }

        /*
         * Was the REMOTE_START of the shell successful?
         * Did we fail the login or did the remote user disconnect?
         */
        if (result == ERROR)
            rlogindSessionDisconnect (pSlot, RLOGIND_REF_MAIN | RLOGIND_REF_LOGIN);
	}

rlogind_exit:
    close (sd);
    }

/*******************************************************************************
*
* rlogOutTask - stdout to socket process
*
* This routine gets spawned by the rlogin daemon to move
* data between the client socket and the pseudo terminal.
* The task exits when the pty has been closed.
*
* \NOMANUAL
*/
LOCAL void rlogOutTask
    (
    RLOGIND_SESSION_DATA *pSlot  /* pointer to the connection information */
    )
    {
    ssize_t n;
    char buf [BUFSIZ];

    int sock = pSlot->socket;      /* socket to copy output to */
    int ptyMfd = pSlot->outputFd;       

    while ((n = read (ptyMfd, buf, sizeof (buf))) > 0)
	{
	if (write (sock, buf, (size_t)n) != n)
	    break;
	}

    rlogindSessionDisconnect (pSlot, RLOGIND_REF_OUT);
    }

/*******************************************************************************
*
* rlogInTask - socket to stdin process
*
* This routine gets spawned by the rlogin daemon to move
* data between the pseudo terminal and the client socket.
* The task exits, calling rlogExit(), when the client disconnects.
*
* \NOMANUAL
*/

LOCAL void rlogInTask
    (
    RLOGIND_SESSION_DATA *pSlot  /* pointer to the connection information */
    )
    {
    FAST ssize_t n;
    char buf [BUFSIZ];

    int sock = pSlot->socket;      /* socket to copy output to */
    int ptyMfd = pSlot->outputFd;  /* I.e., pSlot->inputFd, pty master side */

    /* Loop, reading from the socket and writing to the pty. */
      	
    while ((n = read (sock, buf, sizeof (buf))) > 0)
	{
	ssize_t m;
	do
	    {
	    m = write (ptyMfd, buf, (size_t)n);
	    if (m <= 0)
		break;
	    n -= m;
	    }
	while (0 < n);
	}

    /*
     * Send a Ctrl-U/Ctrl-D sequence to delete a partial line (if any),
     * and signal an end-of-file condition.
     * This should terminate a login.  The difficulty is, that
     * ptyDrv/tyLib do not provide reliable delivery of the EOF condition;
     * FIOFLUSH, done by loginPrompt(), can clear away the EOF. Also,
     * if the receive buffer is full, the EOF can be silently dropped.
     *
     * The other difficulty is that the client may have sent a correct
     * login followed by other data then closed, but the main server
     * task, which handles the login, may not have yet cleared
     * RLOGIND_REF_LOGIN. In this case, we really shouldn't flush good
     * data which the client may have delivered.  The below code is
     * an imperfect compromise.
     */

#if (_WRS_VXWORKS_MAJOR >= 6)
    ioctl (ptyMfd, PTYIOEOF, PTY_EOFSLAVEREAD | PTY_EOFSLAVEREAD_MOD);
#endif

    buf[0] = tyDeleteLineChar;
    buf[1] = tyEOFGet();
    write (ptyMfd, buf, (size_t)2);

    if ((pSlot->refs & RLOGIND_REF_LOGIN) != 0)
	{
	taskDelay (sysClkRateGet() / 4);

	if ((pSlot->refs & RLOGIND_REF_LOGIN) != 0)
	    {
	    /*
	     * We assume we're still really not logged in, so
	     * it's OK to flush, so that the EOF will reliably
	     * be delivered, unless loginPrompt() flushes it away...
	     * If that happens, we must wait for the loginPrompt()
	     * time-out.
	     */
	    ioctl (pSlot->slaveFd, FIOFLUSH, 0);
	    write (ptyMfd, buf + 1, (size_t)1); /* re-write EOF */
	    }
	}

    rlogindSessionDisconnect (pSlot, RLOGIND_REF_IN);
    }

/*******************************************************************************
*
* rlogExit - exit and cleanup for rlogind
*
* This is the support routine for logout().
* The client socket is closed, shell standard I/O is redirected
* back to the console, and the shell is restarted.
*/

LOCAL void rlogExit
    (
    ULONG sessionId    /* identifies the session to be deleted */
    )
    {
    RLOGIND_SESSION_DATA *pSlot = (RLOGIND_SESSION_DATA *)sessionId;

    /* This routine is called as a result of the logout() command
     * being issued from the shell.
     */

    rlogindSessionDisconnect (pSlot, RLOGIND_REF_SHELL);
    }

/*******************************************************************************
*
* rlogindSessionAdd - add a new entry to the rlogind session slot list
*
* Each of the rlogin clients is associated with an entry in the server's
* session list which records the session-specific context for each
* connection, including the file descriptors that access the command
* interpreter. This routine creates and initializes a new entry in the
* session list, unless the needed memory is not available or the upper
* limit for simultaneous connections is reached.
*
* RETURNS: A pointer to the session list entry, or NULL if none available.
*
* ERRNO: N/A
*
* \NOMANUAL
*
*/

LOCAL RLOGIND_SESSION_DATA *rlogindSessionAdd (void)
    {
    RLOGIND_SESSION_DATA * 	pSlot = NULL;
    int count = 0;
    STATUS rc = ERROR;
    SEM_ID finishSem;

    finishSem = semBCreate(SEM_Q_FIFO, SEM_EMPTY);
    if (finishSem == NULL)
	{
	log_err (RLOGIN_LOG | LOG_ERRNO, "no finishSem");
        return NULL;
	}

    rc = semTake (rlogindMutexSem, WAIT_FOREVER); 

    if (rc == ERROR)
        {
	log_err (RLOGIN_LOG | LOG_ERRNO, "in semTake()");
        goto fail;
        }

    if (rlogindCurrentClients >= rlogindMaxClients)
	goto no_more;

    for (count = 0; count < rlogindMaxClients; count++)
        if (!rlogindTaskList [count].pSession->busyFlag)
            break;
   
    /* Are there no more sessions available ? */ 

    if (count == rlogindMaxClients)
        {
no_more:
	log_err (RLOGIN_LOG, "max session reached");
	goto out;
        }
  
    pSlot = rlogindTaskList [count].pSession;
    pSlot->busyFlag = TRUE;
    pSlot->loggedIn = NEVER_LOGGED_IN;
    rlogindCurrentClients++;
    
    /* Add new entry to the list of active sessions. */

    lstAdd (&rlogindSessionList, &pSlot->node);

    RLOGIND_DEBUG (printf ("Allocated slot %p\n", pSlot));
out:
    semGive (rlogindMutexSem);
fail:
    if (pSlot == NULL)
	semDelete (finishSem);
    return pSlot;
    }

/*******************************************************************************
*
* rlogindSessionDisconnect - Shut down a session
*
* This routine removes a connection and all associated resources for it.
*
* This may be called because of login failure or end of a shell session
*
* \NOMANUAL
*/

LOCAL void rlogindSessionDisconnect 
     (
     RLOGIND_SESSION_DATA *pSlot,
     RLOGIND_REF_TYPE ref
     )
     {
     STATUS rc = ERROR;

     if (pSlot == NULL)
         return; /* should not happen */

     rc = semTake (rlogindMutexSem, WAIT_FOREVER); 

     if (rc == ERROR)
	 {
	 log_err (RLOGIN_LOG | LOG_ERRNO, "in semTake()");
	 return;
	 }

     RLOGIND_DEBUG (printf ("disconnect %p ref=0x%x\n", pSlot, ref));

     if (ref == RLOGIND_REF_SHELL) /* Was call from rlogExit()? */
	{
	/*
	 * Attempt to send "\r\n" to remote side. However, don't
	 * write directly to socket, let rlogOutTask() do it.
	 */
	if (write (pSlot->slaveFd, "\n", (size_t)1) != 1)
	    {
	    log_err (RLOGIN_LOG | LOG_ERRNO, "exit write CRLF");
	    }

	/* We can't do this here, because the REMOTE_STOP message to
	 * the shell parser causes the shell task to restart.  Defer
	 * to tExcTask.
	 */

	pSlot->refs |= RLOGIND_REF_EXCTASK;

	semGive (rlogindMutexSem);

	/*
	 * Release semaphore before posting exc job to avoid possible
	 * (though unlikely) deadlock if EXC queue is full.
	 */

#if (_WRS_VXWORKS_MAJOR < 6)
	excJobAdd (rlogindSessionDisconnect, (int)pSlot, RLOGIND_REF_EXCTASK,
		   0, 0, 0, 0);
#else
	jobAdd ((FUNCPTR)rlogindSessionDisconnect,
		(_Vx_usr_arg_t)pSlot,
		(_Vx_usr_arg_t)RLOGIND_REF_EXCTASK,
		ZERO, ZERO, ZERO, ZERO, NULL);
#endif
	return;
	}

     if (pSlot->refs & RLOGIND_REF_SOCKET)
	 {
	 /*
	  * Wake up rlogInTask() (or possibly rlogOutTask())
	  * waiting on socket.
	  */
	 if (pSlot->socket > STD_ERR)
	     shutdown (pSlot->socket, SHUT_RDWR);
	 pSlot->refs &= ~RLOGIND_REF_SOCKET;
	 }

    if (pSlot->loggedIn == LOGGED_IN_NOW)
	{
	pSlot->loggedIn = WAS_LOGGED_IN;

#if (!defined (_WRS_VXWORKS_MAJOR) || (_WRS_VXWORKS_MAJOR < 6))
	(*rlogindParserControl) (REMOTE_STOP, (UINT32)pSlot, 0);
#else
	(*rlogindParserControl) (REMOTE_STOP, (ULONG)pSlot, 0, NULL);
#endif
	}

    pSlot->refs &= ~ref; /* clear this reference */

    if ((pSlot->refs & RLOGIND_REF_LOGIN) == 0 && pSlot->slaveFd >= 0)
	{
	ioctl (pSlot->slaveFd, FIOFLUSH, 0);
	write (pSlot->slaveFd, "\n", (size_t)1);  /* Wake up the master side */
	close (pSlot->slaveFd);
	pSlot->slaveFd = -1;
	}

     if (pSlot->refs != 0)
	 goto rlogindSessionDisconnect_out;

     /* Close the i/o descriptors */

     if ((pSlot->outputFd) > STD_ERR)
	 close (pSlot->outputFd);

     /* Remove the pseudo terminal for the session. */

     if (ptyDevRemove (pSlot->ptyRemoteName) == ERROR)
	 log_err (RLOGIN_LOG, "ptyDevRemove failed."); 

     /* Close the socket connection to the client */

     if ((pSlot->socket) > STD_ERR)
	 {
	 if (pSlot->loggedIn != WAS_LOGGED_IN)
	     {
	     struct linger l;

	     l.l_onoff = 1;
	     l.l_linger = 0;

	     /* Reset the connection rather than close gracefully.
	      * The peer didn't log in successfully, so we won't bother
	      * to accumulate TCP TIME-WAIT state, which we otherwise might
	      * if we did the active close (e.g. in response to Ctrl-D).
	      */
	     setsockopt (pSlot->socket, SOL_SOCKET, SO_LINGER,
			 (char *) &l, (socklen_t)sizeof (l));

	     taskDelay (5 * sysClkRateGet()); /* dubious */
	     }
         close (pSlot->socket);
	 }
 
     /* Re-Initialize all elements of the structure to sane values */

     pSlot->socket         = ERROR;
     pSlot->inputFd        = ERROR;     
     pSlot->outputFd       = ERROR;
     pSlot->slaveFd        = ERROR;
     pSlot->outputTask     = TASK_ID_ERROR;
     pSlot->inputTask      = TASK_ID_ERROR;
     pSlot->parserControl  = NULL;
     pSlot->busyFlag       = FALSE;
     pSlot->loggedIn       = NEVER_LOGGED_IN;

     --rlogindCurrentClients;
     lstDelete (&rlogindSessionList, &pSlot->node);

     RLOGIND_DEBUG (printf ("Released slot %p\n", pSlot));

rlogindSessionDisconnect_out:

     semGive (rlogindMutexSem); 
     }

/*******************************************************************************
*
* rlogindIoTasksCreate - Create tasks to transferring i/o between socket and fd
* 
* Two tasks are created: An input task and an output task.  The name is based on
*    the pSlot argument for uniqueness.
* 
* RETURNS: OK if tasks are spawned successfully, or ERROR otherwise.
*
* \NOMANUAL
*/

LOCAL STATUS rlogindIoTasksCreate
    (
    RLOGIND_SESSION_DATA *pSlot
    )
    {
    char sessionTaskName[IO_TASK_NAME_MAX_LEN];
    char sessionInTaskName[IO_TASK_NAME_MAX_LEN];
    char sessionOutTaskName[IO_TASK_NAME_MAX_LEN];
    RLOGIND_REF_TYPE refs;
 
    /*
     * Spawn the input and output tasks which transfer data between
     * the socket and the i/o file descriptor. 
     *
     */

    sprintf (sessionTaskName, "_%lx", (ULONG)pSlot);
    sprintf (sessionInTaskName, "tRlogInTask%s", sessionTaskName);
    sprintf (sessionOutTaskName,"tRlogOutTask%s", sessionTaskName);
  
    /* spawn the process which transfers data from the master pty
     * to the socket;
     * spawn the process which transfers data from the client socket
     * to the master pty.
     */

    pSlot->outputTask = (TASK_ID) 
	taskSpawn (sessionOutTaskName, rlogTaskPriority,
		   rlogTaskOptions, (size_t)rlogTaskStackSize,
		   (FUNCPTR) rlogOutTask,
		   (_Vx_usr_arg_t)pSlot,
		   ZERO, ZERO, ZERO, ZERO, ZERO, ZERO, ZERO, ZERO, ZERO);

    if (pSlot->outputTask == TASK_ID_ERROR)
	{
	refs = RLOGIND_REF_OUT | RLOGIND_REF_IN;
	goto rlogindIoTasksCreate_fail;
	}

    pSlot->inputTask = (TASK_ID) 
	taskSpawn (sessionInTaskName, rlogTaskPriority,
		   rlogTaskOptions, (size_t)rlogTaskStackSize,
		   (FUNCPTR) rlogInTask,
		   (_Vx_usr_arg_t)pSlot,
		   ZERO, ZERO, ZERO, ZERO, ZERO, ZERO, ZERO, ZERO, ZERO);

    if (pSlot->inputTask == TASK_ID_ERROR)
        {
	refs = RLOGIND_REF_IN;
	goto rlogindIoTasksCreate_fail;
        }

    return OK;

rlogindIoTasksCreate_fail:

    semTake (rlogindMutexSem, WAIT_FOREVER);
    pSlot->refs &= ~refs;
    semGive (rlogindMutexSem);
    log_err (RLOGIN_LOG | LOG_ERRNO, "spawning task");
    return ERROR;
    }

/*******************************************************************************
*
* rlogindSessionPtysCreate - Create a pty pair and open them 
* 
* Two file descriptors are created: An input and an output fd.  The name is 
*    based on the pSlot argument for uniqueness.  The file descriptors are 
*    stored in the session structure.
* 
* RETURNS: OK if successfull, or ERROR otherwise.
*
* \NOMANUAL
*/

LOCAL STATUS rlogindSessionPtysCreate 
    (
    RLOGIND_SESSION_DATA *pSlot
    )
    {

    char sessionPtyRemoteName[PTY_DEVICE_NAME_MAX_LEN];
    char sessionPtyRemoteNameM[PTY_DEVICE_NAME_MAX_LEN];
    char sessionPtyRemoteNameS[PTY_DEVICE_NAME_MAX_LEN];
    int masterFd;

    /* Create unique names for the pty device */

    sprintf (sessionPtyRemoteName, "%s_%lx.", ptyRlogName, (ULONG)pSlot);
    sprintf (sessionPtyRemoteNameM, "%sM", sessionPtyRemoteName);
    sprintf (sessionPtyRemoteNameS, "%sS", sessionPtyRemoteName);

    /* pseudo tty device creation */

    if (ptyDevCreate (sessionPtyRemoteName,
		      (size_t)BUFSIZ, (size_t)BUFSIZ) == ERROR)
        {
	log_err (RLOGIN_LOG, "ptyDevCreate failed");
        return ERROR;
        }

    /* Master-side open of pseudo tty */

    strncpy(pSlot->ptyRemoteName, sessionPtyRemoteName,
	    (size_t)PTY_DEVICE_NAME_MAX_LEN);

    if ((masterFd = open (sessionPtyRemoteNameM, O_RDWR, 0)) == ERROR)
        {
        log_err (RLOGIN_LOG | LOG_ERRNO, "opening %s", sessionPtyRemoteNameM);
        return ERROR;
        }
    else
        {
        pSlot->inputFd = masterFd;
        pSlot->outputFd = masterFd;
        }

    /* Slave-side open of pseudo tty */

    if ((pSlot->slaveFd = open (sessionPtyRemoteNameS, O_RDWR, 0)) == ERROR)
        {
        log_err (RLOGIN_LOG | LOG_ERRNO, "opening %s", sessionPtyRemoteNameS);
        return ERROR;
        }

    /* setup the slave device to act like a terminal */

    ioctl (pSlot->slaveFd, FIOOPTIONS, OPT_TERMINAL);

    return OK;
    }

/*******************************************************************************
*
* rlogindTaskDelete - remove task data from task list
* The server uses this routine to clean up the (partial) task list if 
* an error occurs during startup.
*
* RETURNS: N/A
*
* \NOMANUAL
*/

LOCAL void rlogindTaskDelete
    (
    int 	numTasks 	/* number of entries to remove */
    )
    {
    int 	count;

    for (count = 0; count < numTasks; count++)
        if (rlogindTaskList [count].pSession != NULL)
            free (rlogindTaskList [count].pSession);
 
    free (rlogindTaskList);
    semDelete (rlogindMutexSem);
    }

/*******************************************************************************
*
* recvStr - receive a null terminated string from a socket
*
* Similar to fioRdString, but for sockets.
* TODO: Bound the time we wait for the string.
*
* RETURNS: OK or ERROR.
*/

LOCAL STATUS recvStr
    (
    FAST int sd,        /* socket */
    FAST char *buf,      /* where to put the string */
    size_t bufSiz       /* don't overflow */
    )
    {
    char * endPtr = buf + bufSiz;
 
    do
	{
	if ( (buf >= endPtr) || (read (sd, buf, (size_t)1) != 1) )
            {
            log_err (RLOGIN_LOG, "read failed");
	    return ERROR;
            }
	}
    while (*(buf++) != EOS);

    return (OK);
    }

#endif /* _WRS_KERNEL */


/*******************************************************************************
*
* rlogin - log in to a remote host
*
* This routine allows users to log in to a remote host. In the kernel it may
* be called from the VxWorks shell as follows:
*
* \cs
*    -> rlogin "remoteSystem"
* \ce
*
* where <remoteSystem> is either a host name, which has been previously added
* to the remote host table by a call to hostAdd(), or an Internet address in
* dot notation (e.g., "90.0.0.2").  The remote system will be logged into
* with the current user name as set by a call to iam().
*
* If a different userid is needed to connect to remote host then, remCurIdSet()
* should be used prior to rlogin call.
*
* \cs
*    -> remCurIdSet "user id"
* \ce
*
*
* In user space, an application can start an rlogin session through a program 
* This can be done as follows:
*
* \cs
*    remLibInit (WAIT_FOREVER);
*    rlogin ("remote hostname or IP address");
* \ce
*
* If a different userid is needed to connect to remote host then, setlogin()
* should be called prior to rlogin() call. This will set user id for the RTP.
*
* \cs
*    setlogin ("user id");
* \ce
*
* The user disconnects from the remote system by typing:
* \cs
*    ~.
* \ce
* as the only characters on the line, or by simply logging out from the remote
* system using logout().
*
* RETURNS:
* OK, or ERROR if the host is unknown, no privileged ports are available,
* the routine is unable to connect to the host, or the child process cannot
* be spawned.
*
* WARNING: This routine is callable only from one task context
* (i.e., from one shell) at a time. It is, however,
* not protected against multiple concurrent invocations. Use with caution.
*
* SEE ALSO: iam(), logout(), remCurIdSet(), setlogin()
*/

STATUS rlogin
    (
    char *host          /* name of host to connect to */
    )
    {
    char remUser [MAX_IDENTITY_LEN];
    char remPasswd [MAX_IDENTITY_LEN];
#ifdef INET6
    struct sockaddr_in6 hostAddr;
    int                 family = AF_INET6;
#else
    struct sockaddr_in  hostAddr;
    int                 family = AF_INET;
#endif
    int port = 0;
    char ch;
    int quitFlag = 0;
    int status = ERROR;

    if (rhostnameToAddr (host, (struct sockaddr *)&hostAddr) != OK)
        {
	log_err (RLOGIN_LOG, "rhostnameToAddr() failed");
	return ERROR;
        }

    #ifdef _WRS_KERNEL
      remCurIdGet (remUser, remPasswd);
    #else
      getlogin_r(remUser, MAX_IDENTITY_LEN-1);
      getpassword_r(remPasswd, MAX_IDENTITY_LEN-1);
    #endif

    /* get a privileged port */
    rloginSocket = rresvport_af (&port, family);

    if (rloginSocket == ERROR)
	{
	log_err (RLOGIN_LOG | LOG_ERRNO, "obtaining privileged port");
	return ERROR;
	}

    status = connect (rloginSocket, (struct sockaddr *)&hostAddr,
                      (socklen_t)sizeof(hostAddr));

    if (status == ERROR)
	{
	log_err (RLOGIN_LOG | LOG_ERRNO, "could not connect to host.");
	close (rloginSocket);
	return ERROR;
	}

    /* send a null (no seperate STDERR) */

    write (rloginSocket, "", (size_t)1);

    /* send the local and remote user names */

    write (rloginSocket, remUser, strlen (remUser) + 1);
    write (rloginSocket, remUser, strlen (remUser) + 1);

    /* send the terminal type */

    write (rloginSocket, (char *)rlogTermType, strlen (rlogTermType) + 1);

    /* spawn a process to handle stdin */
    
    rlogChildTaskId = (TASK_ID) taskSpawn ("tRlogChildTask", rlogTaskPriority, 
                                 rlogTaskOptions, (size_t)rlogTaskStackSize,
                                 (FUNCPTR) rlogChildTask,
			   	 ZERO, ZERO, ZERO, ZERO, ZERO,
				 ZERO, ZERO, ZERO, ZERO, ZERO);
    if (rlogChildTaskId == TASK_ID_ERROR)
        {
        log_err (RLOGIN_LOG | LOG_ERRNO, "trouble spawning child");
        close (rloginSocket);
	return ERROR;
	}

    /* set console to RAW mode except with XON-XOFF and 7 bit */

    ioctl (STD_IN, FIOOPTIONS, OPT_TANDEM | OPT_7_BIT);

    while ((rloginSocket != ERROR) && (read (STD_IN, &ch, (size_t)1) == 1))
	{
	/* track input of "<CR>~.<CR>" to terminate connection */

	if ((quitFlag == 0) && (ch == '~'))
	    quitFlag = 1;
	else if ((quitFlag == 1) && (ch == '.'))
	    quitFlag = 2;
	else if ((quitFlag == 2) && (ch == '\r'))
	    break;		/* got "<CR>~.<CR>" */
	else
	    quitFlag = (ch == '\r') ? 0 : -1;

	write (rloginSocket, &ch, (size_t)1);
	}

    /* wait for other tasks to finish up, i.e. rlogind */

    taskDelay (sysClkRateGet () / 2);

    if (rloginSocket != ERROR)
	{
	taskDelete ((int) rlogChildTaskId);
	close (rloginSocket);
	}

    /* reset console */

    ioctl (STD_IN, FIOOPTIONS, OPT_TERMINAL);
    printf ("\nClosed connection.\n");

    return OK;
    }



/*******************************************************************************
*
* rlogChildTask - rlogin child
*
* This routine gets spawned by rlogin() to read data from
* the rloginSocket and write it to stdout.
* The task exits when the client disconnects;
* rlogin is informed by setting rloginSocket to ERROR.
*
* but not LOCAL for i()
*
* \NOMANUAL
*/

LOCAL void rlogChildTask (void)
    {
    char buf [BUFSIZ];
    ssize_t n;

    while ((n = read (rloginSocket, buf, sizeof (buf))) > 0)
	write (STD_OUT, buf, (size_t)n);

    close (rloginSocket);

    /* indicate that client side caused termination,
     * and stop rlogin from reading stdin */

    rloginSocket = ERROR;

    ioctl (STD_IN, FIOCANCEL, 0);
    }

/******************************************************************************
*
* rhostnameToAddr - Convert the host name to a numeric address
* 
* This function is called by rlogin to convert the host name to a numeric
* address. This routine is common to both IPv4 and IPv6.
*
* RETURNS:    OK        upon success
*             ERROR     otherwise
*
* \NOMANUAL
*/

static STATUS rhostnameToAddr
    (
    char *            pHost,     /* Name of the host */
    struct sockaddr * pHostAddr  /* Host address to be returned */
    )
    {
    struct addrinfo     addrHints;
    struct addrinfo *   pAddrResults = NULL;
    struct addrinfo *   pAddrResultsTmp = NULL;
#ifdef INET6
    char                hostnameTmp[INET6_ADDRSTRLEN];
    BOOL                found = FALSE;
    struct sockaddr_in6 * pAddr = NULL;
    size_t              hostAddrLen = sizeof (struct sockaddr_in6);
#else
    char                  hostnameTmp[INET_ADDR_LEN];
    struct sockaddr_in *  pAddr = NULL;
    size_t                hostAddrLen = sizeof (struct sockaddr_in);
#endif

    if ( (pHost == NULL) || (pHostAddr == NULL) )
        {
	log_err (RLOGIN_LOG, "host or sockadd is NULL");
	return ERROR;
        }

    bzero ((char *) &addrHints, sizeof (struct addrinfo));

#ifdef INET6
    if (inet_addr (pHost) != (u_long)ERROR)
        {
        /* convert a dotted decimal string into an IPv4-mapped IPv6 address */
        sprintf (hostnameTmp, "0::ffff:%s", pHost);
        /* no address resolution necessary in this case */
        addrHints.ai_flags |= AI_NUMERICHOST;
        }
    else
#endif
        sprintf (hostnameTmp, "%s", pHost);

#ifdef INET6
    /* 
     * Return IPv4 addresses as IPv4-mapped IPv6 addresses if there
     * are no IPv6 addresses available for this host.
     */

    addrHints.ai_flags |= AI_V4MAPPED;
    addrHints.ai_family = AF_INET6;
#else
    addrHints.ai_family = AF_INET;
#endif

    if (getaddrinfo (hostnameTmp, NULL, &addrHints, &pAddrResults) == ERROR)
        {
	log_err (RLOGIN_LOG | LOG_ERRNO, "in getaddrinfo()");
	return ERROR;
        }

    bzero ((char *)pHostAddr, hostAddrLen);
    pAddrResultsTmp = pAddrResults;

#ifdef INET6
    do
        {
        if (pAddrResults->ai_family == AF_INET6)
            {
            bcopy ((char *)(struct sockaddr_in6 *) (pAddrResults->ai_addr),
                           (char *)pHostAddr, sizeof (struct sockaddr_in6));
            found = TRUE;
            break;
            }
        pAddrResults = pAddrResults->ai_next;
        } while (pAddrResults);

    pAddrResults = pAddrResultsTmp;

    if (!found)
#endif
        do
            {
            if (pAddrResults->ai_family == AF_INET)
                {
                bcopy ((char *)(struct sockaddr_in *)pAddrResults->ai_addr,
                        (char *)(struct sockaddr_in *)pHostAddr,
                                                  sizeof (struct sockaddr_in));
                 break;
                 }
            pAddrResults = pAddrResults->ai_next;
            } while (pAddrResults);

    freeaddrinfo (pAddrResultsTmp);        

#ifdef INET6
    pAddr = (struct sockaddr_in6 *)pHostAddr;
    pAddr->sin6_family = AF_INET6;
    pAddr->sin6_len = sizeof (struct sockaddr_in6);
    pAddr->sin6_port = htons (LOGIN_SERVICE);
#else
    pAddr = (struct sockaddr_in *)pHostAddr;
    pAddr->sin_family = AF_INET;
    pAddr->sin_len = sizeof (struct sockaddr_in);
    pAddr->sin_port = htons (LOGIN_SERVICE);
#endif

    return OK;
    }
