package alma.userrepository.shared.acs;

import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;

import javax.naming.NamingException;

import alma.userrepository.errors.UserRepositoryException;
import alma.userrepository.shared.DirectorySession;

public abstract class ConnectionManager {
    public static Object writeLock = new Object();

    private static Map<String, DirectorySession> connections = Collections
            .synchronizedMap(new ConcurrentHashMap<String, DirectorySession>());

    /**
     * Connects to the user repository using the given user id and password.
     * 
     * @param uid
     *            the user id of the user we connect as
     * @param password
     *            the password of the user
     * @return the hashcode referring to the connection.
     * 
     * @throws UserRepositoryException
     */
    public String connect(String uid, String password)
            throws UserRepositoryException {
        DirectorySession session = createSession(uid, password);

        // store connection using hashcode as index
        String hashCode = UUID.randomUUID().toString();
        connections.put(hashCode, session);

        return hashCode;
    }

    /**
     * Connect anonymously to the user repository.
     * 
     * @return hashcode referring to the connection
     * @throws UserRepositoryException
     */
    public String connect() throws UserRepositoryException {
        return connect("", "");
    }

    /**
     * Release the connection associated with the given hashcode.
     * 
     * @param hash
     *            the hashcode of the stored connection.
     * @throws NamingException
     *             if release of the connection fails.
     */
    public void release(String hashCode) throws UserRepositoryException {
        // don't release while we're writing a user. Should only happen if
        // client code calls write and release in parallel threads
        synchronized (writeLock) {
            DirectorySession session = connections.get(hashCode);
            if (session != null) {
                session.close();
                connections.remove(hashCode);
            }
        }
    }

    public void releaseAll() throws UserRepositoryException {
        Set<String> keys = connections.keySet();
        for (String hashcode : keys) {
            release(hashcode);
        }
    }

    protected DirectorySession getSession(String hashCode) {
        return connections.get(hashCode);
    }

    protected abstract DirectorySession createSession(String uid,
            String password) throws UserRepositoryException;

}
