/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 */
package alma.userrepository.roledirectory.acs;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;

import alma.UserRepositoryError.ObjectNotFoundErrorEx;
import alma.UserRepositoryError.PermissionErrorEx;
import alma.UserRepositoryError.UserRepositoryErrorEx;
import alma.UserRepositoryError.wrappers.AcsJObjectNotFoundErrorEx;
import alma.UserRepositoryError.wrappers.AcsJPermissionErrorEx;
import alma.UserRepositoryError.wrappers.AcsJUnknownCauseEx;
import alma.acs.component.ComponentImplBase;
import alma.acs.component.ComponentLifecycleException;
import alma.acs.container.ContainerServices;
import alma.userrepository.ApplicationRoleStruct;
import alma.userrepository.RoleDirectoryOperations;
import alma.userrepository.UserRole;
import alma.userrepository.UserRoleStruct;
import alma.userrepository.errors.ObjectNotFoundException;
import alma.userrepository.errors.PermissionException;
import alma.userrepository.errors.UserRepositoryException;
import alma.userrepository.roledirectory.Role;
import alma.userrepository.roledirectory.RoleDirectory;
import alma.userrepository.roledirectory.RoleDirectorySession;
import alma.userrepository.shared.DirectorySessionFactory;

/**
 * @author stewart
 * 
 */
public class RoleDirectoryImpl extends ComponentImplBase implements
        RoleDirectoryOperations {

    private ConnectionManager myConnectionManager;

    // ///////////////////////////////////////////////////////////
    // Implementation of ComponentLifecycle
    // ///////////////////////////////////////////////////////////

    public RoleDirectoryImpl() {
    }

    @Override
    public void initialize(ContainerServices inContainerServices)
            throws ComponentLifecycleException {
        super.initialize(inContainerServices);

        this.myConnectionManager = ConnectionManager.getInstance(this.m_logger);
    }

    /**
     * (non-Javadoc)
     * 
     * @see alma.acs.component.ComponentLifecycle#cleanUp()
     */
    @Override
    public void cleanUp() {
        m_logger.info("cleanUp() called...");
        try {
            myConnectionManager.releaseAll();
        } catch (UserRepositoryException e) {
            m_logger
                    .log(
                            Level.SEVERE,
                            "Exceptions thrown while cleaning up UserRepository connections",
                            e);
        }
        super.cleanUp();
    }

    // ///////////////////////////////////////////////////////////
    // Implementation of ACSComponent
    // ///////////////////////////////////////////////////////////

    /**
     * (non-Javadoc)
     * 
     * @see alma.userrepository.RoleDirectory#assignRole(java.lang.String,
     *      java.lang.String, java.lang.String)
     */
    public boolean assignRole(String inConnectionId, String inUID,
            UserRole inRole) throws ObjectNotFoundErrorEx, PermissionErrorEx,
            UserRepositoryErrorEx {
        RoleDirectorySession session = myConnectionManager
                .retrieve(inConnectionId);

        try {
            return session.assignRole(inUID, idlToJava(inRole));
        } catch (UserRepositoryException e) {
            if (e instanceof ObjectNotFoundException) {
                AcsJObjectNotFoundErrorEx acsE = new AcsJObjectNotFoundErrorEx(
                        e);
                throw acsE.toObjectNotFoundErrorEx();
            } else if (e instanceof PermissionException) {
                AcsJPermissionErrorEx acsE = new AcsJPermissionErrorEx(e);
                throw acsE.toPermissionErrorEx();
            } else {
                AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
                throw acsE.toUserRepositoryErrorEx();
            }
        }
    }

    /**
     * (non-Javadoc)
     * 
     * @see alma.userrepository.RoleDirectory#getAllRoles(java.lang.String)
     */
    public UserRoleStruct getAllRoles(String inConnectionId)
            throws UserRepositoryErrorEx {

        RoleDirectorySession session = myConnectionManager
                .retrieve(inConnectionId);
        try {
            return javaToIDL(session.getAllRoles());
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }
    }

    /**
     * (non-Javadoc)
     * 
     * @see alma.userrepository.RoleDirectory#getAllUserRoles(java.lang.String,
     *      java.lang.String)
     */
    public UserRoleStruct getAllUserRoles(String inConnectionId, String inUID)
            throws UserRepositoryErrorEx {

        RoleDirectorySession session = myConnectionManager
                .retrieve(inConnectionId);
        try {
            return javaToIDL(session.getAllUserRoles(inUID));
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }
    }

    /**
     * (non-Javadoc)
     * 
     * @see alma.userrepository.RoleDirectory#getExplicitUserRoles(java.lang
     *      .String, java.lang.String)
     */
    public UserRoleStruct getExplicitUserRoles(String inConnectionId,
            String inUID) throws UserRepositoryErrorEx {

        RoleDirectorySession session = myConnectionManager
                .retrieve(inConnectionId);
        try {
            return javaToIDL(session.getExplicitUserRoles(inUID));
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }
    }

    /**
     * (non-Javadoc)
     * 
     * @see alma.userrepository.RoleDirectory#revokeRole(java.lang.String,
     *      java.lang.String, UserRole)
     */
    public boolean revokeRole(String inConnectionId, String inUID,
            UserRole inRole) throws UserRepositoryErrorEx,
            ObjectNotFoundErrorEx, PermissionErrorEx {
        RoleDirectorySession session = myConnectionManager
                .retrieve(inConnectionId);

        try {
            return session.revokeRole(inUID, idlToJava(inRole));
        } catch (UserRepositoryException e) {
            if (e instanceof ObjectNotFoundException) {
                AcsJObjectNotFoundErrorEx acsE = new AcsJObjectNotFoundErrorEx(
                        e);
                throw acsE.toObjectNotFoundErrorEx();
            } else if (e instanceof PermissionException) {
                AcsJPermissionErrorEx acsE = new AcsJPermissionErrorEx(e);
                throw acsE.toPermissionErrorEx();
            } else {
                AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
                throw acsE.toUserRepositoryErrorEx();
            }
        }
    }

    /**
     * (non-Javadoc)
     * 
     * @see alma.userrepository.RoleDirectory#createConnection(java.lang.String,
     *      java.lang.String)
     */
    public String createConnection(String inUID, String inPassword)
            throws UserRepositoryErrorEx {
        String connectionId = null;
        try {
            connectionId = myConnectionManager.connect(inUID, inPassword);
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }

        return connectionId;
    }

    /**
     * (non-Javadoc)
     * 
     * @see alma.userrepository.RoleDirectory#releaseConnection(java.lang.String)
     */
    public void releaseConnection(String inConnectionId)
            throws UserRepositoryErrorEx {
        try {
            myConnectionManager.release(inConnectionId);
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.RoleDirectory#isAlive()
     */
    public boolean isAlive() {
        return DirectorySessionFactory.getInstance(this.m_logger).isAlive();
    }

    public static UserRoleStruct javaToIDL(RoleDirectory inDirectory) {
        UserRoleStruct outStruct = new UserRoleStruct();
        outStruct.userName = inDirectory.getUserName();
        List<ApplicationRoleStruct> list = new ArrayList<ApplicationRoleStruct>();
        for (String applicationName : inDirectory.getApplicationNames()) {
            List<UserRole> userRoleList = new ArrayList<UserRole>();
            for (Role role : inDirectory.getRoles(applicationName)) {
                userRoleList.add(javaToIDL(role));
            }
            ApplicationRoleStruct struct = new ApplicationRoleStruct(
                    applicationName, userRoleList.toArray(new UserRole[0]));
            list.add(struct);
        }
        outStruct.appRoleStructSeq = list.toArray(new ApplicationRoleStruct[0]);
        return outStruct;
    }

    public static RoleDirectory idlToJava(UserRoleStruct inStruct) {
        RoleDirectory outDir = new RoleDirectory(inStruct.userName);
        for (ApplicationRoleStruct appStruct : inStruct.appRoleStructSeq) {
            for (UserRole userRole : appStruct.roleSeq) {
                outDir
                        .add(new Role(userRole.roleName,
                                userRole.applicationName));
            }
        }
        return outDir;
    }

    public static UserRole javaToIDL(Role inRole) {
        UserRole outRole = new UserRole();
        outRole.applicationName = inRole.getApplicationName();
        outRole.roleName = inRole.getRoleName();
        return outRole;
    }

    public static Role idlToJava(UserRole inRole) {
        Role outRole = new Role(inRole.roleName, inRole.applicationName);
        return outRole;
    }
}
