/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 */
package alma.userrepository.projectdirectory.acs;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.omg.CORBA.StringHolder;

import alma.ACS.ComponentStates;
import alma.ProjectCodeModule.ProjectCodeStruct;
import alma.ProjectCodeModule.ProjectTypeCodeEnum;
import alma.UserRepositoryError.InvalidAttributeErrorEx;
import alma.UserRepositoryError.ObjectNotFoundErrorEx;
import alma.UserRepositoryError.PermissionErrorEx;
import alma.UserRepositoryError.UserRepositoryErrorEx;
import alma.UserRepositoryError.wrappers.AcsJInvalidAttributeErrorEx;
import alma.UserRepositoryError.wrappers.AcsJObjectNotFoundErrorEx;
import alma.UserRepositoryError.wrappers.AcsJPermissionErrorEx;
import alma.UserRepositoryError.wrappers.AcsJUnknownCauseEx;
import alma.acs.component.ComponentLifecycle;
import alma.acs.component.ComponentLifecycleException;
import alma.acs.container.ContainerServices;
import alma.archive.projectcode.ProjectCode;
import alma.archive.projectcode.TypeCode;
import alma.userrepository.MemberStruct;
import alma.userrepository.ProjectDirectoryJ;
import alma.userrepository.ProjectRoleEnum;
import alma.userrepository.ProjectRoleStruct;
import alma.userrepository.UserRole;
import alma.userrepository.errors.InvalidAttributeException;
import alma.userrepository.errors.ObjectNotFoundException;
import alma.userrepository.errors.PermissionException;
import alma.userrepository.errors.UserRepositoryException;
import alma.userrepository.projectdirectory.ProjectDirectorySession;
import alma.userrepository.projectdirectory.ProjectRole;
import alma.userrepository.projectdirectory.ProjectRoleDirectory;
import alma.userrepository.roledirectory.acs.RoleDirectoryImpl;
import alma.userrepository.shared.DirectorySessionFactory;
import alma.xmljbind.userrepository.user.User;

public class ProjectDirectoryImpl implements ComponentLifecycle,
        ProjectDirectoryJ {
    private ContainerServices m_containerServices;

    private Logger m_logger;

    protected Log log = LogFactory.getLog(this.getClass());

    private ConnectionManager myConnectionManager;

    // ///////////////////////////////////////////////////////////
    // Implementation of ComponentLifecycle
    // ///////////////////////////////////////////////////////////

    /*
     * (non-Javadoc)
     * 
     * @seealma.acs.component.ComponentLifecycle#initialize(alma.acs.container.
     * ContainerServices)
     */
    public void initialize(ContainerServices containerServices) {
        this.m_containerServices = containerServices;
        this.m_logger = this.m_containerServices.getLogger();
        this.myConnectionManager = ConnectionManager.getInstance(this.m_logger);
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.acs.component.ComponentLifecycle#cleanUp()
     */
    public void cleanUp() {
        try {
            myConnectionManager.releaseAll();
        } catch (UserRepositoryException e) {
            m_logger
                    .log(
                            Level.SEVERE,
                            "Exceptions thrown while cleaning up UserRepository connections",
                            e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.acs.component.ComponentLifecycle#aboutToAbort()
     */
    public void aboutToAbort() {
        cleanUp();
        m_logger.info("managed to abort...");
    }

    // ///////////////////////////////////////////////////////////
    // Implementation of ACSComponent
    // ///////////////////////////////////////////////////////////

    /*
     * (non-Javadoc)
     * 
     * @see alma.ACS.ACSComponentOperations#componentState()
     */
    public ComponentStates componentState() {
        return m_containerServices.getComponentStateManager().getCurrentState();
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.ACS.ACSComponentOperations#name()
     */
    public String name() {
        return m_containerServices.getName();
    }

    // ///////////////////////////////////////////////////////////
    // Implementation of UserRepositoryComponentJ
    // ///////////////////////////////////////////////////////////

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.ProjectDirectoryJ#assignProjectRole(java.lang.String,
     * java.lang.String, alma.userrepository.ProjectRole, java.lang.String)
     */
    public boolean addProjectUserMember(String connectionId, String member,
            ProjectRoleEnum role, ProjectCodeStruct projectId)
            throws ObjectNotFoundErrorEx, PermissionErrorEx,
            UserRepositoryErrorEx {
        ProjectDirectorySession session = myConnectionManager
                .retrieve(connectionId);

        try {
            return session.addProjectMember(member, idlToJava(projectId),
                    idlToJava(role));
        } catch (UserRepositoryException e) {
            if (e instanceof ObjectNotFoundException) {
                AcsJObjectNotFoundErrorEx acsE = new AcsJObjectNotFoundErrorEx(
                        e);
                throw acsE.toObjectNotFoundErrorEx();
            } else if (e instanceof PermissionException) {
                AcsJPermissionErrorEx acsE = new AcsJPermissionErrorEx(e);
                throw acsE.toPermissionErrorEx();
            } else {
                AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
                throw acsE.toUserRepositoryErrorEx();
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.ProjectDirectoryJ#assignProjectRole(java.lang.String,
     * alma.userrepository.UserRole, alma.userrepository.ProjectRole,
     * java.lang.String)
     */
    public boolean addProjectRoleMember(String connectionId,
            UserRole userRole, ProjectRoleEnum projectRole,
            ProjectCodeStruct projectId) throws ObjectNotFoundErrorEx,
            PermissionErrorEx, UserRepositoryErrorEx {
        ProjectDirectorySession session = myConnectionManager
                .retrieve(connectionId);

        try {
            return session.addProjectMember(RoleDirectoryImpl
                    .idlToJava(userRole), idlToJava(projectId),
                    idlToJava(projectRole));
        } catch (UserRepositoryException e) {
            if (e instanceof ObjectNotFoundException) {
                AcsJObjectNotFoundErrorEx acsE = new AcsJObjectNotFoundErrorEx(
                        e);
                throw acsE.toObjectNotFoundErrorEx();
            } else if (e instanceof PermissionException) {
                AcsJPermissionErrorEx acsE = new AcsJPermissionErrorEx(e);
                throw acsE.toPermissionErrorEx();
            } else {
                AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
                throw acsE.toUserRepositoryErrorEx();
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.ProjectDirectoryJ#createProject(java.lang.String,
     * java.lang.String, ProjectTypeCodeEnum)
     */
    public ProjectCodeStruct createProject(String connectionId,
            String inPIUserId, ProjectTypeCodeEnum typeCode)
            throws UserRepositoryErrorEx, InvalidAttributeErrorEx,
            PermissionErrorEx {
        ProjectDirectorySession session = myConnectionManager
                .retrieve(connectionId);

        try {
            return javaToIDL(session.createProject(inPIUserId,
                    idlToJava(typeCode)));
        } catch (UserRepositoryException e) {
            if (e instanceof InvalidAttributeException) {
                AcsJInvalidAttributeErrorEx acsE = new AcsJInvalidAttributeErrorEx(
                        e);
                throw acsE.toInvalidAttributeErrorEx();
            } else if (e instanceof PermissionException) {
                AcsJPermissionErrorEx acsE = new AcsJPermissionErrorEx(e);
                throw acsE.toPermissionErrorEx();
            } else {
                AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
                throw acsE.toUserRepositoryErrorEx();
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.ProjectDirectoryJ#exists(java.lang.String,
     * java.lang.String)
     */
    public boolean exists(String connectionId, ProjectCodeStruct projectId)
            throws UserRepositoryErrorEx {
        ProjectDirectorySession session = myConnectionManager
                .retrieve(connectionId);

        boolean result = false;

        try {
            result = session.exists(idlToJava(projectId));
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }

        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.ProjectDirectory#getProjectMembers(java.lang.String,
     * ProjectCodeStruct)
     */
    public ProjectRoleStruct getProjectMembers(String connectionId,
            ProjectCodeStruct projectId) throws UserRepositoryErrorEx {
        ProjectRoleStruct outStruct = null;

        ProjectDirectorySession session = myConnectionManager
                .retrieve(connectionId);

        try {
            ProjectRoleDirectory dir = session
                    .getProjectMembers(idlToJava(projectId));
            outStruct = javaToIDL(dir);
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }

        return outStruct;
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.ProjectDirectoryJ#listProjects(java.lang.String)
     */
    @Override
    public ProjectCodeStruct[] getAllProjects(String connectionId)
            throws UserRepositoryErrorEx {
        ProjectDirectorySession session = myConnectionManager
                .retrieve(connectionId);

        List<ProjectCodeStruct> structList = new ArrayList<ProjectCodeStruct>();
        try {
            List<ProjectCode> codeList = session.getProjects();
            for (ProjectCode code : codeList) {
                structList.add(javaToIDL(code));
            }
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }

        return structList.toArray(new ProjectCodeStruct[0]);
    }

    @Override
    public ProjectCodeStruct[] getProjects(String connectionId, int year,
            int semester) throws UserRepositoryErrorEx {
        ProjectDirectorySession session = myConnectionManager
                .retrieve(connectionId);

        List<ProjectCodeStruct> structList = new ArrayList<ProjectCodeStruct>();
        try {
            List<ProjectCode> codeList = session.getProjects(year, semester);
            for (ProjectCode code : codeList) {
                structList.add(javaToIDL(code));
            }
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }

        return structList.toArray(new ProjectCodeStruct[0]);
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.ProjectDirectoryJ#removeProject(
     * ProjectCodeStruct)
     */
    public void removeProject(String connectionId, ProjectCodeStruct projectCode)
            throws UserRepositoryErrorEx, InvalidAttributeErrorEx,
            PermissionErrorEx {
        ProjectDirectorySession session = myConnectionManager
                .retrieve(connectionId);

        try {
            session.removeProject(idlToJava(projectCode));
        } catch (UserRepositoryException e) {
            if (e instanceof InvalidAttributeException) {
                AcsJInvalidAttributeErrorEx acsE = new AcsJInvalidAttributeErrorEx(
                        e);
                throw acsE.toInvalidAttributeErrorEx();
            } else if (e instanceof PermissionException) {
                AcsJPermissionErrorEx acsE = new AcsJPermissionErrorEx(e);
                throw acsE.toPermissionErrorEx();
            } else {
                AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
                throw acsE.toUserRepositoryErrorEx();
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.ProjectDirectoryJ#revokeProjectRole(java.lang.String,
     * java.lang.String, alma.userrepository.ProjectRole, java.lang.String)
     */
    public boolean removeProjectUserMember(String connectionId, String userId,
            ProjectRoleEnum role, ProjectCodeStruct projectId)
            throws UserRepositoryErrorEx, ObjectNotFoundErrorEx,
            PermissionErrorEx {
        ProjectDirectorySession session = myConnectionManager
                .retrieve(connectionId);

        try {
            return session.removeProjectMember(userId, idlToJava(projectId),
                    idlToJava(role));
        } catch (UserRepositoryException e) {
            if (e instanceof ObjectNotFoundException) {
                AcsJObjectNotFoundErrorEx acsE = new AcsJObjectNotFoundErrorEx(
                        e);
                throw acsE.toObjectNotFoundErrorEx();
            } else if (e instanceof PermissionException) {
                AcsJPermissionErrorEx acsE = new AcsJPermissionErrorEx(e);
                throw acsE.toPermissionErrorEx();
            } else {
                AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
                throw acsE.toUserRepositoryErrorEx();
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.ProjectDirectoryJ#revokeProjectRole(java.lang.String,
     * alma.userrepository.UserRole, alma.userrepository.ProjectRole,
     * java.lang.String)
     */
    public boolean removeProjectRoleMember(String connectionId,
            UserRole userRole, ProjectRoleEnum projectRole,
            ProjectCodeStruct projectId) throws UserRepositoryErrorEx,
            ObjectNotFoundErrorEx, PermissionErrorEx {
        ProjectDirectorySession session = myConnectionManager
                .retrieve(connectionId);

        try {
            return session.removeProjectMember(RoleDirectoryImpl
                    .idlToJava(userRole), idlToJava(projectId),
                    idlToJava(projectRole));
        } catch (UserRepositoryException e) {
            if (e instanceof ObjectNotFoundException) {
                AcsJObjectNotFoundErrorEx acsE = new AcsJObjectNotFoundErrorEx(
                        e);
                throw acsE.toObjectNotFoundErrorEx();
            } else if (e instanceof PermissionException) {
                AcsJPermissionErrorEx acsE = new AcsJPermissionErrorEx(e);
                throw acsE.toPermissionErrorEx();
            } else {
                AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
                throw acsE.toUserRepositoryErrorEx();
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.ProjectDirectoryJ#resetSubmissionPeriod()
     */
    public void resetSubmissionPeriod(String connectionId)
            throws UserRepositoryErrorEx {
        ProjectDirectorySession session = myConnectionManager
                .retrieve(connectionId);

        try {
            session.resetSubmissionPeriod();
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.ProjectDirectoryJ#createConnection(java.lang.String,
     * java.lang.String)
     */
    public String createConnection(String uid, String password)
            throws UserRepositoryErrorEx {
        String connectionId = null;
        try {
            connectionId = myConnectionManager.connect(uid, password);
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }

        return connectionId;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.ProjectDirectoryJ#releaseConnection(java.lang.String)
     */
    public void releaseConnection(String connectionId)
            throws UserRepositoryErrorEx {
        try {
            myConnectionManager.release(connectionId);
        } catch (UserRepositoryException e) {
            AcsJUnknownCauseEx acsE = new AcsJUnknownCauseEx(e);
            throw acsE.toUserRepositoryErrorEx();
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.ProjectDirectoryJ#isAlive()
     */
    public boolean isAlive() {
        return DirectorySessionFactory.getInstance(this.m_logger).isAlive();
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.ProjectDirectoryJ#junkMethodSoWeGenerateBindingClass
     * (alma.xmljbind.userrepository.user.User, org.omg.CORBA.StringHolder)
     */
    public void junkMethodSoWeGenerateBindingClass(User junkIn,
            StringHolder junkOut) {
    }

    public static ProjectRoleEnum javaToIDL(ProjectRole inRole) {
        return ProjectRoleEnum.from_int(inRole.ordinal());
    }

    public static ProjectRole idlToJava(ProjectRoleEnum inRole) {
        return Enum.valueOf(ProjectRole.class, inRole.toString());
    }

    public static ProjectTypeCodeEnum javaToIDL(TypeCode inCode) {
        return ProjectTypeCodeEnum.from_int(inCode.ordinal());
    }

    public static TypeCode idlToJava(ProjectTypeCodeEnum inCode) {
        return Enum.valueOf(TypeCode.class, inCode.toString());
    }

    public static ProjectCodeStruct javaToIDL(ProjectCode inCode) {
        ProjectCodeStruct outStruct = new ProjectCodeStruct();
        outStruct.year = inCode.getYear();
        outStruct.semester = inCode.getSemester();
        outStruct.sequenceNumber = inCode.getSequenceNumber();
        outStruct.typeCode = javaToIDL(inCode.getTypeCode());
        return outStruct;
    }

    public static ProjectCode idlToJava(ProjectCodeStruct inStruct) {
        return new ProjectCode(inStruct.year, inStruct.semester,
                inStruct.sequenceNumber, idlToJava(inStruct.typeCode));
    }

    public static ProjectRoleStruct javaToIDL(ProjectRoleDirectory inDirectory) {
        ProjectRoleStruct outStruct = new ProjectRoleStruct();
        outStruct.projectCodeStruct = javaToIDL(inDirectory.getProjectCode());
        List<MemberStruct> list = new ArrayList<MemberStruct>();
        for (ProjectRole role : inDirectory.getAllRoles()) {
            MemberStruct struct = new MemberStruct();
            struct.projectRole = javaToIDL(role);
            struct.memberSeq = inDirectory.getMembers(role).toArray(
                    new String[0]);
            list.add(struct);
        }
        outStruct.memberSeq = list.toArray(new MemberStruct[0]);
        return outStruct;
    }

    public static ProjectRoleDirectory idlToJava(ProjectRoleStruct inStruct) {
        ProjectRoleDirectory outDir = new ProjectRoleDirectory(
                idlToJava(inStruct.projectCodeStruct));
        for (MemberStruct struct : inStruct.memberSeq) {
            for (String member : struct.memberSeq) {
                outDir.add(idlToJava(struct.projectRole), member);
            }
        }
        return outDir;
    }

    @Override
    public void execute() throws ComponentLifecycleException {
        // TODO Auto-generated method stub

    }
}
