/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 */
package alma.userrepository.addressbook.acs;

import java.util.ArrayList;
import java.util.List;

import alma.userrepository.AddressBookEntryJ;
import alma.userrepository.addressbook.ldap.beans.LdapAddressBookEntry;
import alma.userrepository.domainmodel.Address;
import alma.userrepository.domainmodel.AddressBookEntry;
import alma.userrepository.domainmodel.Preferences;
import alma.userrepository.domainmodel.User;

public class AcsAddressBookEntry implements AddressBookEntry {
	AddressBookEntryJ entry = new AddressBookEntryJ();

	private User user = null;
	private Preferences preferences = null;
	private List<Address> addresses;

	public AcsAddressBookEntry(AddressBookEntryJ entry) {
		this.entry = entry;
		user = new AcsUser(entry.thisUser);
		preferences = new AcsPreferences(entry.thisPreferences);
		addresses = new ArrayList<Address>();
		for (alma.xmljbind.userrepository.address.Address address : entry.addressArray) {
			AcsAddress addressWrapper = new AcsAddress(address);
			addresses.add(addressWrapper);
		}
	}

	public AcsAddressBookEntry(AddressBookEntry entry) {
		// castor won't serialize if the underlying xmljbind object if it's
		// been extended (or implements another interface?), so we have to
		// create the basic objects afresh
		this.entry = new AddressBookEntryJ();
		
		AcsUser userWrapper = new AcsUser(entry.getUser());
		this.user = userWrapper;
		this.entry.thisUser = userWrapper.getCastor();

		AcsPreferences preferencesWrapper = new AcsPreferences(entry.getPreferences());
		this.preferences = preferencesWrapper;
		this.entry.thisPreferences = preferencesWrapper.getCastor();
		
		addresses = new ArrayList<Address>();
		List<alma.xmljbind.userrepository.address.Address> acsAddresses = new ArrayList<alma.xmljbind.userrepository.address.Address>();
		for (alma.userrepository.domainmodel.Address address : entry
				.getAddresses()) {
			AcsAddress addressWrapper = new AcsAddress(address);
			addresses.add(addressWrapper);
			acsAddresses.add(addressWrapper.getCastor());
		}

		this.entry.addressArray = acsAddresses
				.toArray(new alma.xmljbind.userrepository.address.Address[acsAddresses
						.size()]);
	}

	public AddressBookEntry getLdapAddressBookEntry() {
		AcsUser acsUser = new AcsUser(entry.thisUser);
		AcsPreferences acsPreferences = new AcsPreferences(entry.thisPreferences);
		User user = acsUser.getLdapUser();
		Preferences preferences = acsPreferences.getLdapPreferences();

		List<Address> addresses = new ArrayList<Address>();
		for (alma.xmljbind.userrepository.address.Address address : entry.addressArray) {
			AcsAddress acsAddress = new AcsAddress(address);
			Address ldapAddress = acsAddress.getLdapAddress();
			addresses.add(ldapAddress);
		}
		
		AddressBookEntry ldapEntry = new LdapAddressBookEntry(user,addresses,preferences);
		return ldapEntry;
	}
	
	public String toString() {
		StringBuffer sb = new StringBuffer();
		sb.append("ACS User with uid: " + user.getUid() + ", name: "
				+ user.getFullName() + "\n");
		int counter = 0;
		for (Address address : addresses) {
			counter++;
			sb.append("ACS Address " + counter + " has uid: "
					+ address.getUid());
		}
		return sb.toString();
	}

	public AddressBookEntryJ getAddressBookEntryJ() {
		return entry;
	}

	public List<Address> getAddresses() {
		return addresses;
	}

	public User getUser() {
		return user;
	}

	public void setAddresses(List<Address> addresses) {
		for (alma.userrepository.domainmodel.Address address : addresses) {
			AcsAddress addressWrapper = new AcsAddress(address);
			addresses.add(addressWrapper);
		}
	}

	public void setUser(User user) {
		this.user = new AcsUser(user);
	}

	public Preferences getPreferences() {
		return preferences;
	}

	public void setPreferences(Preferences preferences) {
		this.preferences = new AcsPreferences(preferences);		
	}
}
