/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 */
package alma.userrepository.shared.ldap;

import java.util.Properties;
import java.util.logging.Logger;

import alma.archive.database.helpers.DBConfiguration;
import alma.archive.exceptions.general.DatabaseException;

public class ConnectionProperties {

    private enum PropertyData {
        // JNDI property keys
        INITIAL_FACTORY("archive.userrepository.factory.initial",
                "com.sun.jndi.ldap.LdapCtxFactory"), STATE_FACTORY(
                "archive.userrepository.factory.state",
                "alma.userrepository.addressbook.ldap.UserRepositoryStateFactory:com.sun.jndi.ldap.obj.LdapGroupFactory"), OBJECT_FACTORY(
                "archive.userrepository.factory.object",
                "alma.userrepository.addressbook.ldap.UserRepositoryObjectFactory:com.sun.jndi.ldap.obj.LdapGroupFactory"), AUTHENTICATION(
                "archive.userrepository.security.authentication", "simple"), PROVIDER_URL(
                "archive.userrepository.provider.url", "ldap://127.0.0.1:1389/"),
        // User and password to use for 'anonymous' access.
        READ_ONLY_USER_NAME("archive.userrepository.readonly.user.name",
                "uid=readUser,ou=master,dc=alma,dc=info"), READ_ONLY_USER_PASSWORD(
                "archive.userrepository.readonly.user.password", "oS!iMLDAP!");

        private String myName;

        private String myDefaultValue;

        PropertyData(String inName, String inDefaultValue) {
            this.myName = inName;
            this.myDefaultValue = inDefaultValue;
        }

        public String getName() {
            return this.myName;
        }

        public String getDefaultValue() {
            return this.myDefaultValue;
        }
    }

    private volatile static ConnectionProperties instance = null;

    private static Logger log = Logger.getAnonymousLogger();

    protected Properties myProperties = null;

    public static ConnectionProperties getInstance() {

        if (instance == null) {
            synchronized (ConnectionProperties.class) {
                if (instance == null) {
                    instance = new ConnectionProperties();
                }
            }
        }
        return instance;
    }

    private ConnectionProperties() {
        setProperties();
    }

    private void setProperties() {
        this.myProperties = new Properties();
        DBConfiguration config = null;
        try {
            config = DBConfiguration.instance(log);
        } catch (DatabaseException e) {
            log
                    .warning("Could not locate the Archive Configuration. Will use default property values.");
        }
        for (PropertyData key : PropertyData.values()) {
            String value = null;
            if (config != null) {
                value = config.get(key.getName());
            }
            if (value == null) {
                value = key.getDefaultValue();
            }
            this.myProperties.put(key.getName(), value);
        }
    }

    // -------------------------------------------------------------- Properties

    public String getInitialFactory() {
        return this.myProperties.getProperty(PropertyData.INITIAL_FACTORY
                .getName());
    }

    public String getObjectFactory() {
        return this.myProperties.getProperty(PropertyData.OBJECT_FACTORY
                .getName());
    }

    public String getStateFactory() {
        return this.myProperties.getProperty(PropertyData.STATE_FACTORY
                .getName());
    }

    public String getProviderUrl() {
        return this.myProperties
                .getProperty(PropertyData.PROVIDER_URL.getName());
    }

    public String getAuthentication() {
        return this.myProperties.getProperty(PropertyData.AUTHENTICATION
                .getName());
    }

    public String getReadOnlyUserName() {
        return this.myProperties.getProperty(PropertyData.READ_ONLY_USER_NAME
                .getName());
    }

    public String getReadOnlyUserPassword() {
        return this.myProperties
                .getProperty(PropertyData.READ_ONLY_USER_PASSWORD.getName());
    }
}
