/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 */
package alma.userrepository.shared;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public abstract class PropertiesLoader {
    static final String VALUE_SEPARATOR = ",";

    private static final String SUFFIX = ".properties";

    protected Log log = LogFactory.getLog(this.getClass());

    private String myResourceName;

    protected Properties myProperties = null;

    /**
     * Constructor taking the resource name as input. The resource is what will
     * be treated as the source for the fetched properties. The resource must
     * map to a file with .properties extension. The name is assumed to be
     * absolute and can use either "/" or "." for package segment separation
     * with an optional leading "/" and optional ".properties" suffix. Thus, the
     * following names refer to the same resource:
     * 
     * <pre>
     *   some.pkg.Resource
     *   some.pkg.Resource.properties
     *   some/pkg/Resource
     *   some/pkg/Resource.properties
     *   /some/pkg/Resource
     *   /some/pkg/Resource.properties
     * </pre>
     * 
     * @param inResourceName
     *            a String being the name of the resource to get the properties
     *            from.
     */
    protected PropertiesLoader(String inResourceName) {
        if (inResourceName == null)
            throw new IllegalArgumentException("inResourceName cannot be null");

        String name = inResourceName;

        if (name.startsWith("/")) {
            name = inResourceName.substring(1);
        }

        name = name.replace('.', '/');

        if (!name.endsWith(SUFFIX)) {
            name = name.concat(SUFFIX);
        }

        this.myResourceName = name;
    }

    abstract protected Properties getDefaultProperties();

    abstract protected void updatePropertiesFromSettings();

    abstract protected void updateSettingsFromProperties();

    /**
     * A convenience overload of {@link #getProperties(ClassLoader)} that uses
     * the current thread's context classloader.
     * 
     * @throws IllegalArgumentException
     *             if the resource was not found.
     */
    protected void getProperties() {
        getProperties(Thread.currentThread().getContextClassLoader());
    }

    /**
     * Looks up a resource with the name given in the constructor in the
     * classpath.
     * 
     * @param loader
     *            classloader through which to load the resource [null is
     *            equivalent to the application loader]
     * 
     * @throws IllegalArgumentException
     *             in case the resource was not found.
     * @throws IllegalStateException
     *             in case the resource does not conform to the expected data in
     *             an overridden {@link #validate()} method.
     */
    protected void getProperties(ClassLoader loader) {

        this.myProperties = new Properties(getDefaultProperties());

        InputStream in = null;
        try {
            if (loader == null) {
                loader = ClassLoader.getSystemClassLoader();
            }

            // Returns null on lookup failures
            in = loader.getResourceAsStream(this.myResourceName);
            if (in != null) {
                this.myProperties.load(in); // Can throw IOException
            }
            updateSettingsFromProperties();
            validate();
        } catch (Exception e) {
            throw new IllegalArgumentException("Error when reading resource ["
                    + this.myResourceName + "] (see cause for further details).", e);
        } finally {
            if (in != null)
                try {
                    in.close();
                } catch (Throwable ignore) {
                }
        }
    }

    protected void validate() {
    }

    public String getResourceName() {
        return this.myResourceName;
    }

    // ------------------------------------------------------- Utility Functions

    public static List<String> splitPropertyToList(String propertyValue) {
        String[] values = propertyValue.split(VALUE_SEPARATOR);
        List<String> outList = new ArrayList<String>();
        for (String value : values) {
            outList.add(value.trim());
        }
        return outList;
    }

    public static String concatToPropertyValue(List<String> l) {
        StringBuffer outBuffer = new StringBuffer();
        for (String s : l) {
            outBuffer.append(s);
            outBuffer.append(VALUE_SEPARATOR);
        }

        return outBuffer.toString();
    }

}