package alma.userrepository.shared;

import java.util.Hashtable;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.naming.AuthenticationException;
import javax.naming.Context;
import javax.naming.NamingException;
import javax.naming.directory.DirContext;
import javax.naming.directory.InitialDirContext;
import javax.naming.directory.SearchControls;

import alma.userrepository.addressbook.AddressBookSession;
import alma.userrepository.addressbook.ldap.LdapAddressBookSession;
import alma.userrepository.errors.PermissionException;
import alma.userrepository.errors.UserRepositoryException;
import alma.userrepository.projectdirectory.ProjectDirectorySession;
import alma.userrepository.projectdirectory.ldap.LdapProjectDirectorySession;
import alma.userrepository.roledirectory.RoleDirectorySession;
import alma.userrepository.roledirectory.ldap.LdapRoleDirectorySession;
import alma.userrepository.shared.ldap.DnDirectory;
import alma.userrepository.shared.ldap.ConnectionProperties;

public class LdapDirectorySessionFactory extends DirectorySessionFactory {
    protected static ConnectionProperties connectionProperties = null;

    public LdapDirectorySessionFactory(Logger inLogger) {
        super(inLogger);
        connectionProperties = ConnectionProperties.getInstance();
    }

    public AddressBookSession newAddressBookSession(String userId,
            String password) throws UserRepositoryException {
        String userDn = getUserDn(userId);
        DirContext context = getContext(userDn, password);
        RoleDirectorySession roleSession = newRoleDirectorySession(userId,
                password);
        return new LdapAddressBookSession(context, roleSession, this.myLogger);
    }

    public ProjectDirectorySession newProjectDirectorySession(String userId,
            String password) throws UserRepositoryException {
        String userDn = getUserDn(userId);
        DirContext context = getContext(userDn, password);
        return new LdapProjectDirectorySession(context, this.myLogger);
    }

    public RoleDirectorySession newRoleDirectorySession(String userId,
            String password) throws UserRepositoryException {
        String userDn = getUserDn(userId);
        DirContext context = getContext(userDn, password);
        return new LdapRoleDirectorySession(context, this.myLogger);
    }

    public boolean isAlive() {
        Hashtable<String, String> env = new Hashtable<String, String>();

        // override jndi properties with our own custom UR properties. We
        // don't need security properties as we're performing an anonymous
        // bind
        env.put(Context.INITIAL_CONTEXT_FACTORY, connectionProperties
                .getInitialFactory());
        env.put(Context.SECURITY_AUTHENTICATION, connectionProperties
                .getAuthentication());
        env
                .put(Context.STATE_FACTORIES, connectionProperties
                        .getStateFactory());
        env.put(Context.OBJECT_FACTORIES, connectionProperties
                .getObjectFactory());
        env.put(Context.PROVIDER_URL, connectionProperties.getProviderUrl());

        try {
            DirContext ctx = new InitialDirContext(env);
            SearchControls controls = new SearchControls();
            controls.setSearchScope(SearchControls.OBJECT_SCOPE);
            String searchFilter = "(objectClass=OpenLDAProotDSE)";
            myLogger.fine("Searching root context using the filter '" + searchFilter
                    + "'");
            ctx.search("", "(objectClass=OpenLDAProotDSE)", controls);
            ctx.close();
            return true;
        } catch (NamingException e) {
            myLogger.log(Level.SEVERE, "Unexpected exception: ", e);
            return false;
        }
    }

    protected String getUserDn(String inUserId) {
        String userDn = "";
        if ((inUserId != null) && (!inUserId.equals(""))) {
            userDn = DnDirectory.getUserDn(inUserId);
        }
        return userDn;
    }

    protected DirContext getContext(String userDn, String password)
            throws UserRepositoryException {
        Hashtable<String, String> env = new Hashtable<String, String>();

        if (userDn == null) {
            userDn = "";
        }
        if (password == null) {
            password = "";
        }

        if ("".equals(userDn)) {
            userDn = connectionProperties.getReadOnlyUserName();
            password = connectionProperties.getReadOnlyUserPassword();
        }

        // override jndi properties with our own custom UR properties
        env.put(Context.INITIAL_CONTEXT_FACTORY, connectionProperties
                .getInitialFactory());
        env.put(Context.SECURITY_AUTHENTICATION, connectionProperties
                .getAuthentication());
        env
                .put(Context.STATE_FACTORIES, connectionProperties
                        .getStateFactory());
        env.put(Context.OBJECT_FACTORIES, connectionProperties
                .getObjectFactory());
        env.put(Context.PROVIDER_URL, connectionProperties.getProviderUrl());

        // and add the username and password
        env.put(Context.SECURITY_PRINCIPAL, userDn);
        env.put(Context.SECURITY_CREDENTIALS, password);

        DirContext ctx;

        try {
            myLogger.fine("Connecting to server "
                    + connectionProperties.getProviderUrl()
                    + " with username '" + userDn + "'");
            ctx = new InitialDirContext(env);
            myLogger.finer("Connection established.");
        } catch (AuthenticationException e) {
            myLogger.info("Incorrect login attempt for user '" + userDn + "'.");
            throw new PermissionException("Invalid credentials", e);
        } catch (NamingException e) {
            myLogger.log(Level.SEVERE, "Unexpected error occurred: ", e);
            throw new UserRepositoryException(
                    "Unexpected error occurred while creating project directory session",
                    e);
        }

        return ctx;
    }

    @Override
    boolean authenticateImpl(String userId, String password)
            throws UserRepositoryException {
        boolean outResult = false;
        DirContext context = null;
        if (userId != null && password != null && !"".equals(userId)
                && !"".equals(password)) {
            try {
                context = getContext(getUserDn(userId), password);
                outResult = true;
            } catch (PermissionException e) {
                // do nothing.
            } finally {
                if (context != null) {
                    try {
                        context.close();
                    } catch (NamingException e) {
                        myLogger.log(Level.SEVERE, "Unexpected error occurred: ", e);
                        throw new UserRepositoryException(
                                "Unexpected error occurred while creating project directory session",
                                e);
                    }
                }
            }
        }
        return outResult;
    }

}
