/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 */
package alma.userrepository.roledirectory.ldap;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.naming.InvalidNameException;
import javax.naming.NameNotFoundException;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.NoPermissionException;
import javax.naming.directory.AttributeInUseException;
import javax.naming.directory.BasicAttributes;
import javax.naming.directory.DirContext;
import javax.naming.directory.NoSuchAttributeException;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.LdapName;
import javax.naming.ldap.Rdn;

import alma.userrepository.errors.ObjectNotFoundException;
import alma.userrepository.errors.PermissionException;
import alma.userrepository.errors.UserRepositoryException;
import alma.userrepository.roledirectory.Role;
import alma.userrepository.roledirectory.RoleDirectory;
import alma.userrepository.roledirectory.RoleDirectorySession;
import alma.userrepository.shared.ldap.DnDirectory;

public class LdapRoleDirectorySession implements RoleDirectorySession {
    protected Logger myLogger = null;

    private DirContext ctx = null;

    @SuppressWarnings("unused")
    private LdapRoleDirectorySession() {
    }

    public LdapRoleDirectorySession(DirContext ctx, Logger inLogger)
            throws UserRepositoryException {
        this.ctx = ctx;
        this.myLogger = inLogger;
    }

    /**
     * @see alma.userrepository.roledirectory.RoleDirectorySession#close()
     */
    public void close() throws UserRepositoryException {
        try {
            ctx.close();
        } catch (NamingException e) {
            myLogger.log(Level.SEVERE, "Couldn't close session: ", e);
            throw new UserRepositoryException("Couldn't close session: ", e);
        }
    }

    /**
     * @throws UserRepositoryException
     * @see alma.userrepository.roledirectory.RoleDirectorySession#getAllRoles
     */
    public RoleDirectory getAllRoles() throws UserRepositoryException {
        return translateList("ALL ROLES", listRoles());
    }

    /**
     * @see alma.userrepository.roledirectory.RoleDirectorySession#getAllUserRoles
     */
    public RoleDirectory getAllUserRoles(String inUserId)
            throws UserRepositoryException {
        return translateList(inUserId, listRoles(inUserId));
    }

    /**
     * @see alma.userrepository.roledirectory.RoleDirectorySession#getExplicitUserRoles
     */
    public RoleDirectory getExplicitUserRoles(String inUserId)
            throws UserRepositoryException {
        return translateList(inUserId, listExplicitRoles(inUserId));
    }

    /**
	 */
    private List<String> listRoles() throws UserRepositoryException {
        List<String> outRoles = new ArrayList<String>();

        SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.SUBTREE_SCOPE);
        controls.setReturningAttributes(new String[0]);
        String filter = "(cn=*)";

        myLogger.finer("Searching " + DnDirectory.ROLE_BASE + " with filter '"
                + filter + "'.");

        NamingEnumeration<SearchResult> ne;
        try {
            ne = ctx.search(DnDirectory.ROLE_BASE, filter, controls);
        } catch (NamingException e) {
            throw new UserRepositoryException(
                    "Unexpected error occurred while searching contents of 'roles' branch.",
                    e);
        }

        try {
            while (ne.hasMoreElements()) {
                SearchResult result = ne.next();
                outRoles.add(result.getNameInNamespace());
            }
        } catch (NamingException e) {
            throw new UserRepositoryException(
                    "Unexpected error occurred while fetching roles.", e);
        }

        return outRoles;
    }

    /**
	 */
    private List<String> listRoles(String uid) throws UserRepositoryException {
        String userDn = DnDirectory.getUserDn(uid);
        List<String> outList = new ArrayList<String>();
        try {
            recursiveFindRolesWithMember(userDn, outList);
        } catch (NamingException e) {
            throw new UserRepositoryException(
                    "Unexpected error occurred while listing roles for user with uid '"
                            + uid + "'.", e);
        }
        return outList;
    }

    private List<String> listExplicitRoles(String uid)
            throws UserRepositoryException {
        List<String> outRoles = new ArrayList<String>();

        SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.SUBTREE_SCOPE);
        controls.setReturningAttributes(new String[0]);
        controls.setDerefLinkFlag(true);
        String filter = "(member=" + DnDirectory.getUserDn(uid) + ")";

        NamingEnumeration<SearchResult> ne;
        try {
            ne = ctx.search(DnDirectory.ROLE_BASE, filter, controls);
        } catch (NamingException e) {
            throw new UserRepositoryException(
                    "Unexpected error occurred while searching for user '"
                            + uid + "' in roles.", e);
        }

        while (ne.hasMoreElements()) {
            try {
                SearchResult result = ne.next();
                outRoles.add(result.getNameInNamespace());
            } catch (NamingException e) {
                throw new UserRepositoryException(
                        "Unexpected error occurred while iterating results for 'match uid="
                                + uid + "' in roles'.", e);
            }
        }

        return outRoles;
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.roledirectory.RoleDirectorySession#assignRole(
     * java.lang.String, java.lang.String)
     */
    public boolean assignRole(String inUID, Role inRole)
            throws UserRepositoryException {
        return modifyRole(inUID, inRole, DirContext.ADD_ATTRIBUTE);
    }

    private boolean modifyRole(String inUID, Role inRole, int inOperation)
            throws UserRepositoryException {
        boolean outResult = true;

        String userDn = DnDirectory.getUserDn(inUID);
        String roleDn = DnDirectory.getRoleDn(inRole);

        try {
            BasicAttributes attrs = new BasicAttributes("member", userDn);
            ctx.modifyAttributes(roleDn, inOperation, attrs);
        } catch (AttributeInUseException e) {
            /*
             * In case the user already has the role to assign.
             */
            outResult = false;
        } catch (NoSuchAttributeException e) {
            /*
             * In case the user did not have the role to revoke.
             */
            outResult = false;
        } catch (NameNotFoundException e) {
            throw new ObjectNotFoundException("The role '" + inRole
                    + "' was not found.", e);
        } catch (NoPermissionException e) {
            throw new PermissionException(
                    "Insufficient permission to modify the role'" + inRole
                            + "' for user with uid '" + inUID + "'.", e);
        } catch (NamingException e) {
            throw new UserRepositoryException(
                    "Unexpected error occurred while modifying the role'"
                            + inRole + "' for user with uid '" + inUID + "'.",
                    e);
        }
        return outResult;
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.roledirectory.RoleDirectorySession#revokeRole
     * (java.lang.String, java.lang.String)
     */
    public boolean revokeRole(String inUID, Role inRole)
            throws UserRepositoryException {
        return modifyRole(inUID, inRole, DirContext.REMOVE_ATTRIBUTE);
    }

    private void recursiveFindRolesWithMember(String inMemberDn,
            List<String> inoutRoleList) throws NamingException {

        SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.SUBTREE_SCOPE);
        controls.setReturningAttributes(new String[0]);
        String filter = "(member=" + inMemberDn + ")";

        NamingEnumeration<SearchResult> ne = ctx.search(DnDirectory.ROLE_BASE,
                filter, controls);
        while (ne.hasMoreElements()) {
            SearchResult result = ne.next();
            String roleDN = result.getNameInNamespace();
            if (!inoutRoleList.contains(roleDN)) {
                inoutRoleList.add(roleDN);
                recursiveFindRolesWithMember(roleDN, inoutRoleList);
            }
        }
    }

    /**
     * Converts a List of DNs into a List of Role objects.
     * 
     * @param inList
     *            a List containing Strings being the role DN.
     * 
     * @return a RoleDirectory containing the corresponding Role objects.
     * @throws UserRepositoryException
     */
    private RoleDirectory translateList(String inUserName, List<String> inList)
            throws UserRepositoryException {
        RoleDirectory outDirectory = new RoleDirectory(inUserName);
        for (String roleDN : inList) {
            try {
                LdapName dn = new LdapName(roleDN);
                Rdn nameRdn = dn.getRdn(dn.size() - 1);
                String roleName = (String) nameRdn.getValue();
                Rdn groupRdn = dn.getRdn(dn.size() - 2);
                String roleApplication = (String) groupRdn.getValue();
                Role role = new Role(roleName, roleApplication);
                outDirectory.add(role);
            } catch (InvalidNameException e) {
                throw new UserRepositoryException(
                        "Highly unexpected exception!", e);
            }
        }
        return outDirectory;
    }
}
