/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 */
package alma.userrepository.projectdirectory.ldap;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.naming.InvalidNameException;
import javax.naming.NameAlreadyBoundException;
import javax.naming.NameNotFoundException;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.NoPermissionException;
import javax.naming.directory.Attribute;
import javax.naming.directory.AttributeInUseException;
import javax.naming.directory.Attributes;
import javax.naming.directory.BasicAttribute;
import javax.naming.directory.BasicAttributes;
import javax.naming.directory.DirContext;
import javax.naming.directory.NoSuchAttributeException;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;

import alma.archive.projectcode.ProjectCode;
import alma.archive.projectcode.TypeCode;
import alma.userrepository.errors.DuplicateObjectException;
import alma.userrepository.errors.InvalidAttributeException;
import alma.userrepository.errors.ObjectNotFoundException;
import alma.userrepository.errors.PermissionException;
import alma.userrepository.errors.UserRepositoryException;
import alma.userrepository.projectdirectory.ProjectDirectorySession;
import alma.userrepository.projectdirectory.ProjectRole;
import alma.userrepository.projectdirectory.ProjectRoleDirectory;
import alma.userrepository.roledirectory.Role;
import alma.userrepository.shared.ldap.DnDirectory;

/**
 * @author Stewart
 * 
 */
public class LdapProjectDirectorySession implements ProjectDirectorySession {

    private static final String ROLE_NAME_ATTR_ID = "almaProjectRoleName";

    protected Logger myLogger = null;

    /**
     * The DirContext for this session
     */
    private DirContext ctx = null;

    private Map<TypeCode, Integer> mySequenceCounterMap = new HashMap<TypeCode, Integer>();
    private Integer myYear;
    private Integer mySemester;

    @SuppressWarnings("unused")
    private LdapProjectDirectorySession() {
    }

    public LdapProjectDirectorySession(DirContext ctx, Logger inLogger) {
        this.ctx = ctx;
        this.myLogger = inLogger;
    }

    /*
     * (non-Javadoc)
     * 
     * @see ProjectDirectorySession#addProjectMember(String, ProjectCode,
     * ProjectRole)
     */
    public boolean addProjectMember(String inMember, ProjectCode inProjectCode,
            ProjectRole inProjectRole) throws UserRepositoryException {
        String member = DnDirectory.getUserDn(inMember);
        return modifyProjectMember(member, inProjectCode, inProjectRole,
                DirContext.ADD_ATTRIBUTE);
    }

    /*
     * (non-Javadoc)
     * 
     * @see ProjectDirectorySession#addProjectMember(Role, ProjectCode,
     * ProjectRole)
     */
    public boolean addProjectMember(Role inRole, ProjectCode inProjectCode,
            ProjectRole inProjectRole) throws UserRepositoryException {
        String member = DnDirectory.getRoleDn(inRole);
        return modifyProjectMember(member, inProjectCode, inProjectRole,
                DirContext.ADD_ATTRIBUTE);
    }

    /**
     * @throws UserRepositoryException
     */
    public void close() throws UserRepositoryException {
        try {
            ctx.close();
        } catch (NamingException e) {
            throw new UserRepositoryException("Couldn't close session: ", e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see ProjectDirectorySession#createProject(String, TypeCode)
     */
    public ProjectCode createProject(String inPI, TypeCode inTypeCode)
            throws UserRepositoryException {
        int attemptCount = 0;
        boolean done = false;
        ProjectCode projectCode = null;
        try {
            while (!done) {
                attemptCount++;
                projectCode = getNextProjectCode(inTypeCode);

                BasicAttributes attrs = new BasicAttributes();
                attrs.put("objectclass", "almaProject");
                attrs.put("almaProjectCode", projectCode.toString());

                BasicAttribute attr = new BasicAttribute(
                        "almaProjectRoleGroupReference");

                for (ProjectRole role : ProjectRole.values()) {
                    attr.add(DnDirectory.getProjectRoleGroupDn(role,
                            projectCode));
                }
                attrs.put(attr);

                try {
                    ctx
                            .bind(DnDirectory.getProjectDn(projectCode), null,
                                    attrs);

                    for (ProjectRole role : ProjectRole.values()) {
                        attrs = new BasicAttributes();
                        if (role == ProjectRole.ADMINISTRATOR) {
                            attr = new BasicAttribute("member");
                            attr.add(DnDirectory.getRoleDn(new Role(
                                    "ADMINISTRATOR", "MASTER")));
                            attr.add(DnDirectory.getRoleDn(new Role(
                                    "ASTRONOMER_ON_DUTY", "OMC")));
                            attrs.put(attr);
                        } else if (role == ProjectRole.PRIMARY_INVESTIGATOR) {
                            attrs.put("member", DnDirectory.getUserDn(inPI));
                        }
                        attrs.put("objectclass", "almaProjectRoleGroup");
                        attrs.put(ROLE_NAME_ATTR_ID, role.getRoleName());
                        ctx.bind(DnDirectory.getProjectRoleGroupDn(role,
                                projectCode), null, attrs);
                    }
                    done = true;
                } catch (NameAlreadyBoundException e) {
                    if (attemptCount < 2) {
                        myLogger
                                .info("Detected that project having code "
                                        + projectCode
                                        + " already exists. This suggests"
                                        + " that several instances are generating"
                                        + " project codes which in general is not a good idea."
                                        + " Will now reset the internal type code counters and try creating a project again.");
                        resetSequenceCounters(this.myYear, this.mySemester);
                    } else {
                        throw new DuplicateObjectException(
                                "Project "
                                        + projectCode
                                        + " already exists which indicates that"
                                        + " the system is set up with several sources for"
                                        + " generating project codes.", e);
                    }
                }
            }
        } catch (NoPermissionException e) {
            throw new PermissionException(
                    "Insufficient permission to create project", e);
        } catch (InvalidNameException e) {
            throw new InvalidAttributeException(
                    "Cannot create valid LDAP dn from project id", e);
        } catch (NamingException e) {
            throw new UserRepositoryException("Error creating project", e);
        }
        return projectCode;
    }

    /*
     * (non-Javadoc)
     * 
     * @see ProjectDirectorySession#exists(ProjectCode)
     */
    public boolean exists(ProjectCode projectCode)
            throws UserRepositoryException {
        SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.ONELEVEL_SCOPE);

        NamingEnumeration<SearchResult> users;
        try {
            users = ctx.search(DnDirectory.PROJECT_BASE, "("
                    + DnDirectory.getProjectRdn(projectCode) + ")", controls);
        } catch (NamingException e) {
            throw new UserRepositoryException("Error searching for project", e);
        }

        if (users.hasMoreElements()) {
            return true;
        } else {
            return false;
        }
    }

    public Map<TypeCode, Integer> getMaxSequenceNumbers(int inYear,
            int inSemester) throws UserRepositoryException {
        Map<TypeCode, Integer> outMap = new HashMap<TypeCode, Integer>();
        for (ProjectCode code : getProjects(inYear, inSemester)) {
            Integer seqNum = outMap.get(code.getTypeCode());
            if (seqNum == null || seqNum < code.getSequenceNumber()) {
                outMap.put(code.getTypeCode(), code.getSequenceNumber());
            }
        }
        return outMap;
    }

    private ProjectCode getNextProjectCode(TypeCode inTypeCode)
            throws UserRepositoryException {
        if (myYear == null) {
            resetSubmissionPeriod();
        }
        int seqNum = this.mySequenceCounterMap.get(inTypeCode) + 1;
        this.mySequenceCounterMap.put(inTypeCode, seqNum);
        return new ProjectCode(this.myYear, this.mySemester, seqNum, inTypeCode);
    }

    /*
     * (non-Javadoc)
     * 
     * @see ProjectDirectorySession#listMembers(ProjectCode, ProjectRole)
     */
    public ProjectRoleDirectory getProjectMembers(ProjectCode inProjectCode)
            throws UserRepositoryException {
        ProjectRoleDirectory outDir = new ProjectRoleDirectory(inProjectCode);

        // search controls matching and returning all members
        String searchFilter = "(member=*)";
        String[] searchAttributes = { "member", ROLE_NAME_ATTR_ID };
        SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.SUBTREE_SCOPE);
        controls.setReturningAttributes(searchAttributes);

        try {
            for (NamingEnumeration<SearchResult> ne = ctx.search(DnDirectory
                    .getProjectDn(inProjectCode), searchFilter, controls); ne
                    .hasMoreElements();) {
                SearchResult result = ne.next();

                Attributes attrs = result.getAttributes();
                Attribute attr = attrs.get(ROLE_NAME_ATTR_ID);
                ProjectRole role = Enum.valueOf(ProjectRole.class,
                        (String) attr.get());
                List<String> memberList = new ArrayList<String>();
                attr = attrs.get("member");
                for (NamingEnumeration<?> vals = attr.getAll(); vals.hasMore();) {
                    String member = (String) vals.next();
                    if (member.startsWith("uid")) {
                        // check that the member is not the "padding" member.
                        if (!member.startsWith("uid=nobody")) {
                            outDir.add(role, DnDirectory.getUserId(member));
                        }
                    } else {
                        recursiveFindMembersOfRole(member, memberList);
                    }
                }
                outDir.addAll(role, memberList);
            }
        } catch (NameNotFoundException e) {
            throw new ObjectNotFoundException("Project '" + inProjectCode
                    + "' not found", e);
        } catch (NamingException e) {
            e.printStackTrace();
            throw new UserRepositoryException("Error searching for members", e);
        }

        return outDir;
    }

    private void recursiveFindMembersOfRole(String inRoleDn,
            List<String> inoutList) throws NamingException {

        String[] attrIds = { "member" };
        Attributes attrs = ctx.getAttributes(inRoleDn, attrIds);
        Attribute attr = attrs.get("member");
        if (attr != null) {
            for (NamingEnumeration<?> vals = attr.getAll(); vals.hasMore();) {
                String member = (String) vals.next();
                if (member.startsWith("uid")) {
                    // check that the member is not the "padding" member.
                    if (!member.startsWith("uid=nobody")) {
                        String userId = DnDirectory.getUserId(member);
                        if (!inoutList.contains(userId)) {
                            inoutList.add(userId);
                        }
                    }
                } else {
                    recursiveFindMembersOfRole(member, inoutList);
                }
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see ProjectDirectorySession#listProjects()
     */
    public List<ProjectCode> getProjects() throws UserRepositoryException {
        return getProjects("*");
    }

    /*
     * (non-Javadoc)
     * 
     * @see ProjectDirectorySession#listProjects(int, int)
     */
    public List<ProjectCode> getProjects(int inYear, int inSemester)
            throws UserRepositoryException {
        String cn = inYear + "." + inSemester + ".*";
        return getProjects(cn);
    }

    private List<ProjectCode> getProjects(String inMatchingString)
            throws UserRepositoryException {
        SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.ONELEVEL_SCOPE);
        String[] searchAttributes = { "almaProjectCode" };
        controls.setReturningAttributes(searchAttributes);
        String filter = "(almaProjectCode=" + inMatchingString + ")";

        // search for objects satisfying the filter
        List<String> resultList;
        try {
            myLogger.fine("listProjects(): asking for projects with filter '"
                    + filter + "'");
            resultList = searchReturnResultList(DnDirectory.PROJECT_BASE,
                    filter, controls);
        } catch (NamingException e) {
            throw new UserRepositoryException("Error searching for projects", e);
        }

        List<ProjectCode> outResultList = new ArrayList<ProjectCode>();
        for (String result : resultList) {
            try {
                outResultList.add(new ProjectCode(result));
            } catch (IllegalArgumentException e) {
                myLogger.log(Level.SEVERE,
                        "Error when creating ProjectCode object using ["
                                + result + "].", e);
            }
        }
        return outResultList;
    }

    private boolean modifyProjectMember(String inMember,
            ProjectCode inProjectCode, ProjectRole inProjectRole,
            int inOperation) throws UserRepositoryException {
        boolean outResult = true;
        String groupDn = DnDirectory.getProjectRoleGroupDn(inProjectRole,
                inProjectCode);

        try {
            BasicAttributes attrs = new BasicAttributes("member", inMember);
            ctx.modifyAttributes(groupDn, inOperation, attrs);
        } catch (AttributeInUseException e) {
            /*
             * In case the user to add already exists.
             */
            outResult = false;
        } catch (NoSuchAttributeException e) {
            /*
             * In case the user to remove did not exist.
             */
            outResult = false;
        } catch (NameNotFoundException e) {
            throw new ObjectNotFoundException("The project " + inProjectCode
                    + " was not found.", e);
        } catch (NoPermissionException e) {
            throw new PermissionException(
                    "Insufficient permissions to modify membership of the project '"
                            + inProjectCode + "'", e);
        } catch (NamingException e) {
            throw new UserRepositoryException(
                    "Unexpected error occurred while modifying membership of the project",
                    e);
        }
        return outResult;
    }

    public void removeProject(ProjectCode projectCode)
            throws UserRepositoryException {
        SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.SUBTREE_SCOPE);
        controls.setReturningAttributes(new String[0]);
        String filter = "(objectclass=*)";

        NamingEnumeration<SearchResult> ne;
        try {
            ne = ctx.search(DnDirectory.getProjectDn(projectCode), filter,
                    controls);
        } catch (NameNotFoundException e) {
            return;
        } catch (NamingException e) {
            throw new UserRepositoryException(
                    "Unexpected error occurred while searching for project "
                            + projectCode, e);
        }

        /*
         * Now that we have the project node and all nodes under it we start
         * sorting them based on the number of parent nodes (found in the
         * distinguished name). A node can only be deleted if it is a leaf. By
         * sorting the nodes like this we can delete them in the right order.
         */
        HashMap<Integer, List<String>> deleteMap = new HashMap<Integer, List<String>>();
        try {
            while (ne.hasMoreElements()) {
                SearchResult result = ne.next();
                String name = result.getNameInNamespace();
                int length = name.split(",").length;
                List<String> nameList = deleteMap.get(length);
                if (nameList == null) {
                    nameList = new ArrayList<String>();
                    deleteMap.put(length, nameList);
                }
                nameList.add(name);
            }
        } catch (NamingException e) {
            throw new UserRepositoryException(
                    "Unexpected error occurred while fetching roles.", e);
        }

        try {
            List<Integer> keys = new ArrayList<Integer>(deleteMap.keySet());
            Collections.sort(keys);
            Collections.reverse(keys);
            for (Integer key : keys) {
                for (String name : deleteMap.get(key)) {
                    ctx.unbind(name);
                }
            }
        } catch (NameNotFoundException e) {
        } catch (NoPermissionException e) {
            throw new PermissionException(
                    "Insufficient permission to remove project", e);
        } catch (NamingException e) {
            throw new UserRepositoryException(
                    "Unexpected error occurred while removing project", e);
        }

    }

    private void resetSequenceCounters(int inYear, int inSemester)
            throws UserRepositoryException {
        this.mySequenceCounterMap = getMaxSequenceNumbers(inYear, inSemester);
        for (TypeCode code : TypeCode.class.getEnumConstants()) {
            if (this.mySequenceCounterMap.get(code) == null) {
                this.mySequenceCounterMap.put(code, new Integer(0));
            }
        }
    }

    public void resetSubmissionPeriod() throws UserRepositoryException {
        Attributes attributes;
        try {
            attributes = ctx.getAttributes("cn=currentSubmissionPeriod,"
                    + DnDirectory.PROJECT_BASE);
        } catch (NamingException e) {
            throw new UserRepositoryException(
                    "Error searching for submission period.", e);
        }

        String value = null;
        int year;
        int semester;
        try {
            value = attributes.get("almaSubmissionYear").get().toString();
            year = Integer.parseInt(value);
        } catch (NumberFormatException e) {
            throw new IllegalArgumentException(
                    "Non-integer value for year in LDAP; found [" + value + "]");
        } catch (NamingException e) {
            throw new IllegalStateException(
                    "Expected attribute almaSubmissionYear not found.");
        }
        try {
            value = attributes.get("almaSubmissionSemester").get().toString();
            semester = Integer.parseInt(value);
        } catch (NumberFormatException e) {
            throw new IllegalArgumentException(
                    "Non-integer value for semester in LDAP; found [" + value
                            + "]");
        } catch (NamingException e) {
            throw new IllegalStateException(
                    "Expected attribute almaSubmissionSemester not found.");
        }
        if (year < 2009 || year > 2200) {
            throw new IllegalArgumentException(
                    "Unrealistic value of year in LDAP; found [" + year
                            + "], value between 2009 and 2200 expected.");
        }
        if (semester < 1 || semester > 9) {
            throw new IllegalArgumentException(
                    "Illegal value of semester in LDAP; found [" + semester
                            + "], value should be between 1 and 9.");
        }
        this.myYear = year;
        this.mySemester = semester;
        resetSequenceCounters(year, semester);
    }

    /*
     * (non-Javadoc)
     * 
     * @see ProjectDirectorySession#removeProjectMember(String, ProjectCode,
     * ProjectRole)
     */
    public boolean removeProjectMember(String inMember,
            ProjectCode inProjectCode, ProjectRole inProjectRole)
            throws UserRepositoryException {
        String member = DnDirectory.getUserDn(inMember);
        return modifyProjectMember(member, inProjectCode, inProjectRole,
                DirContext.REMOVE_ATTRIBUTE);
    }

    /*
     * (non-Javadoc)
     * 
     * @see ProjectDirectorySession#removeProjectMember(Role, ProjectCode,
     * ProjectRole)
     */
    public boolean removeProjectMember(Role inRole, ProjectCode inProjectCode,
            ProjectRole inProjectRole) throws UserRepositoryException {
        String member = DnDirectory.getRoleDn(inRole);
        return modifyProjectMember(member, inProjectCode, inProjectRole,
                DirContext.REMOVE_ATTRIBUTE);
    }

    private List<String> searchReturnResultList(String searchBase,
            String searchFilter, SearchControls controls)
            throws NamingException {
        // search for objects satisfying the filter
        List<String> results = new ArrayList<String>();
        for (NamingEnumeration<SearchResult> ne = ctx.search(searchBase,
                searchFilter, controls); ne.hasMoreElements();) {
            SearchResult result = ne.next();

            Attributes resultAttrs = result.getAttributes();
            for (NamingEnumeration<? extends Attribute> resultAttr = resultAttrs
                    .getAll(); resultAttr.hasMoreElements();) {
                Attribute attr = resultAttr.nextElement();
                for (NamingEnumeration<?> vals = attr.getAll(); vals.hasMore();) {
                    String s = (String) vals.next();
                    results.add(s);
                }
            }
        }
        return results;
    }
}
