/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 */
package alma.userrepository.projectdirectory;

import java.util.List;
import java.util.Map;

import alma.archive.projectcode.ProjectCode;
import alma.archive.projectcode.TypeCode;
import alma.userrepository.errors.UserRepositoryException;
import alma.userrepository.roledirectory.Role;
import alma.userrepository.shared.DirectorySession;

/**
 * @author Stewart
 * 
 */
public interface ProjectDirectorySession extends DirectorySession {

    /**
     * Adds a member to the stated role of a project.
     * 
     * @param inMember
     *            a String stating the user name to add
     * 
     * @param inProjectCode
     *            the project code
     * 
     * @param inProjectRole
     *            the project role
     * 
     * @return a boolean indicating if the stated member was added;
     *         <code>true</code> if the user did not exist and was added,
     *         <code>false</code> otherwise.
     * 
     * @throws PermissionException
     *             in case the user lacks the permission to perform the
     *             operation.
     * @throws ObjectNotFoundException
     *             in case the stated project does not exist.
     * @throws UserRepositoryException
     *             if any other error occurs during the operation
     */
    public boolean addProjectMember(String inMember, ProjectCode inProjectCode,
            ProjectRole inProjectRole) throws UserRepositoryException;

    /**
     * Adds a role to the stated role of a project.
     * 
     * @param inRole
     *            a Role object being the role to add
     * 
     * @param projectCode
     *            the project code
     * 
     * @param projectRole
     *            the project role
     * 
     * @return a boolean indicating if the stated member was added;
     *         <code>true</code> if the user did not exist and was added,
     *         <code>false</code> otherwise.
     * 
     * @throws PermissionException
     *             in case the user lacks the permission to perform the
     *             operation.
     * @throws ObjectNotFoundException
     *             in case the stated project does not exist.
     * @throws UserRepositoryException
     *             if any other error occurs during the operation
     */
    public boolean addProjectMember(Role inRole, ProjectCode inProjectCode,
            ProjectRole inProjectRole) throws UserRepositoryException;

    /**
     * Creates a new project of the given type. Attempts to create a project
     * without sufficient privileges will result in a PermissionException.
     * 
     * @param inPI
     *            a String stating the user name being the Primary Investigator.
     * @param inTypeCode
     *            a TypeCode object stating the type of project to create.
     * 
     * @return a ProjectCode object being the project code of the newly created
     *         project.
     * 
     * @throws UserRepositoryException
     *             if an error occurs during the operation
     */
    public ProjectCode createProject(String inPI, TypeCode inTypeCode)
            throws UserRepositoryException;

    /**
     * Removes the project with the given project code. Attempts to remove a
     * project without sufficient privileges will result in a
     * PermissionException. Attempts to remove a project that does not exist
     * will NOT generate any exceptions.
     * 
     * @param projectCode
     *            the project code
     * 
     * @throws UserRepositoryException
     *             if an error occurs during the operation
     */
    public void removeProject(ProjectCode projectCode)
            throws UserRepositoryException;

    /**
     * Determines whether a project with the given code exists in the directory.
     * 
     * @param projectCode
     *            the project code to search for
     * 
     * @return true if a project with the given ID exists
     * 
     * @throws UserRepositoryException
     *             if an error occurs during the operation
     */
    public boolean exists(ProjectCode projectCode)
            throws UserRepositoryException;

    /**
     * Retrieves the maximum sequence number for each of the different type
     * codes for the given year and semester. In case no project codes, and
     * hence no sequence numbers, are found for a particular type code the type
     * code is excluded from the returned map.
     * 
     * @param inYear
     *            an int being the year of interest
     * @param inSemester
     *            an int being the semester of interest
     * 
     * @return a Map with various TypeCodes as keys and the value being the
     *         corresponding maximum sequence number as an Integer.
     * 
     * @throws UserRepositoryException
     *             if an error occurs during the operation
     */
    public Map<TypeCode, Integer> getMaxSequenceNumbers(int inYear,
            int inSemester) throws UserRepositoryException;

    /**
     * Gets all members and their role(s) in the stated project.
     * 
     * @param projectCode
     *            a ProjectCode identifying the project of interest.
     * 
     * @return a ProjectRoleDirectory containing the roles and members.
     * 
     * @throws UserRepositoryException
     *             if an error occurs during the operation
     */
    public ProjectRoleDirectory getProjectMembers(ProjectCode projectCode)
            throws UserRepositoryException;

    /**
     * Lists all projects. This function returns a list containing the IDs of
     * all projects in the directory. The returned list is NOT sorted.
     * 
     * @return a list of project codes
     * 
     * @throws UserRepositoryException
     *             if an error occurs during the operation
     */
    public List<ProjectCode> getProjects() throws UserRepositoryException;

    /**
     * Lists all projects registered for the given year and semester. This
     * function returns a list containing the IDs of all projects in the
     * directory. The returned list is NOT sorted.
     * 
     * @param inYear
     *            an int being the year of interest
     * @param inSemester
     *            an int being the semester of interest
     * 
     * @return a list of project codes
     * 
     * @throws UserRepositoryException
     *             if an error occurs during the operation
     */
    public List<ProjectCode> getProjects(int inYear, int inSemester)
            throws UserRepositoryException;

    /**
     * Removes a member from the stated role of a project.
     * 
     * @param inMember
     *            a String stating the user name to remove
     * 
     * @param inProjectCode
     *            the project code
     * 
     * @param inProjectRole
     *            the project role
     * 
     * @return a boolean indicating if the stated member was removed;
     *         <code>true</code> if the user existed and was removed,
     *         <code>false</code> otherwise.
     * 
     * @throws PermissionException
     *             in case the user lacks the permission to perform the
     *             operation.
     * @throws ObjectNotFoundException
     *             in case the stated project does not exist.
     * @throws UserRepositoryException
     *             if any other error occurs during the operation
     */
    public boolean removeProjectMember(String inMember,
            ProjectCode inProjectCode, ProjectRole inProjectRole)
            throws UserRepositoryException;

    /**
     * Removes a member from the stated role of a project.
     * 
     * @param inRole
     *            a Role object being the role to remove
     * 
     * @param inProjectCode
     *            the project code
     * 
     * @param inProjectRole
     *            the project role
     * 
     * @return a boolean indicating if the stated member was removed;
     *         <code>true</code> if the user existed and was removed,
     *         <code>false</code> otherwise.
     * 
     * @throws PermissionException
     *             in case the user lacks the permission to perform the
     *             operation.
     * @throws ObjectNotFoundException
     *             in case the stated project does not exist.
     * @throws UserRepositoryException
     *             if any other error occurs during the operation
     */
    public boolean removeProjectMember(Role inRole, ProjectCode inProjectCode,
            ProjectRole inProjectRole) throws UserRepositoryException;

    /**
     * Resets the internal values in the project code generator to whatever
     * values are stored in LDAP.
     * 
     * @throws UserRepositoryException
     *             if an error occurs during the operation
     */
    public void resetSubmissionPeriod() throws UserRepositoryException;
}
