/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 */
package alma.userrepository.addressbook.ldap.beans;

import static alma.userrepository.addressbook.ldap.AttributeUtilities.MODIFY_ATTR;

import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.BasicAttribute;
import javax.naming.directory.BasicAttributes;

import alma.userrepository.addressbook.ldap.AttributeUtilities;
import alma.userrepository.addressbook.ldap.UserInfoProperties;
import alma.userrepository.addressbook.ldap.UserAccountProperties;
import alma.userrepository.domainmodel.User;
import alma.userrepository.errors.UserRepositoryException;
import alma.userrepository.roledirectory.Role;
import alma.userrepository.roledirectory.RoleDirectory;

public class LdapUser implements User {
    private String firstName;
    private String fullName;
    private String modifiedTimestamp;
    private String notes;
    private String surname;
    private String uid;

    /**
     * The roles associated with a user. This is a convenience attribute: the
     * roles cannot be retrieved from an almaUser LDAP entry, they have to be
     * determined by querying the RoleDirectory.
     */
    private RoleDirectory myRoleDirectory;

    private boolean isNewEntry = false;

    private static UserInfoProperties ldapMapping = UserInfoProperties
            .getInstance();
    private static UserAccountProperties posixAccountProperties = UserAccountProperties
            .getInstance();

    // - constructors ---------------------------------------------------------

    public LdapUser() {
    }

    public LdapUser(User user) {
        if (user != null) {
            this.setFirstName(user.getFirstName());
            this.setFullName(user.getFullName());
            this.setNotes(user.getNotes());
            this.setSurname(user.getSurname());
            this.setUid(user.getUid());
        }
    }

    public LdapUser(Attributes attrs) throws UserRepositoryException {
        this.setFirstName(AttributeUtilities.getAttributeDefinition(attrs,
                ldapMapping.getUserFirstName()));
        this.setFullName(AttributeUtilities.getAttributeDefinition(attrs,
                ldapMapping.getUserFullName()));
        this.setNotes(AttributeUtilities.getAttributeDefinition(attrs,
                ldapMapping.getUserNotes()));
        this.setSurname(AttributeUtilities.getAttributeDefinition(attrs,
                ldapMapping.getUserSurname()));
        this.setUid(AttributeUtilities.getAttributeDefinition(attrs,
                ldapMapping.getUserUid()));
        this.setModifiedTimestamp(AttributeUtilities.getAttributeDefinition(
                attrs, MODIFY_ATTR));
    }

    // - public methods -------------------------------------------------------

    /**
     * @return The LdapUser expressed as a set of Attributes
     */
    public Attributes toAttributes() {
        Attributes attrs = new BasicAttributes(true);

        BasicAttribute oc = new BasicAttribute("objectclass");
        for (String objectClass : UserInfoProperties
                .splitPropertyToList(ldapMapping.getUserObjectClasses())) {
            oc.add(objectClass);
        }
        attrs.put(oc);

        // add user attributes
        AttributeUtilities.addPropertyToAttributes(ldapMapping
                .getUserFirstName(), this.getFirstName(), attrs);
        AttributeUtilities.addPropertyToAttributes(ldapMapping
                .getUserFullName(), this.getFullName(), attrs);
        AttributeUtilities.addPropertyToAttributes(ldapMapping.getUserNotes(),
                this.getNotes(), attrs);
        AttributeUtilities.addPropertyToAttributes(
                ldapMapping.getUserSurname(), this.getSurname(), attrs);
        AttributeUtilities.addPropertyToAttributes(ldapMapping.getUserUid(),
                this.getUid(), attrs);

        // if this is a new entry, we should also add the posixaccount and
        // shadowaccount attributes
        if (isNewEntry == true) {
            attrs = addPosixAccountAttributes(attrs);
        }

        // we don't set modifyTimestamp, as this is handled by the LDAP server

        return attrs;
    }

    // - private methods ------------------------------------------------------

    private Attributes addPosixAccountAttributes(Attributes attrs) {
        Attribute oc = attrs.get("objectclass");
        for (String objectClass : UserInfoProperties
                .splitPropertyToList(ldapMapping.getPosixAccountObjectClasses())) {
            oc.add(objectClass);
        }

        // the POSIX details never change, so we can pull them straight from the
        // defaults
        addDefaultToAttributes(UserAccountProperties.GID_NUMBER_KEY,
                posixAccountProperties.getGidNumber(), attrs);
        addDefaultToAttributes(UserAccountProperties.HOME_DIRECTORY_KEY,
                posixAccountProperties.getHomeDirectory(), attrs);
        addDefaultToAttributes(UserAccountProperties.LOGIN_SHELL_KEY,
                posixAccountProperties.getLoginShell(), attrs);
        addDefaultToAttributes(UserAccountProperties.PREFERRED_LANGUAGE_KEY,
                posixAccountProperties.getPreferredLanguage(), attrs);
        addDefaultToAttributes(UserAccountProperties.SHADOW_EXPIRE_KEY,
                posixAccountProperties.getShadowExpire(), attrs);
        addDefaultToAttributes(UserAccountProperties.SHADOW_FLAG_KEY,
                posixAccountProperties.getShadowFlag(), attrs);
        addDefaultToAttributes(UserAccountProperties.SHADOW_INACTIVE_KEY,
                posixAccountProperties.getShadowInactive(), attrs);
        addDefaultToAttributes(UserAccountProperties.SHADOW_MAX_KEY,
                posixAccountProperties.getShadowMax(), attrs);
        addDefaultToAttributes(UserAccountProperties.SHADOW_MIN_KEY,
                posixAccountProperties.getShadowMin(), attrs);
        addDefaultToAttributes(UserAccountProperties.SHADOW_WARNING_KEY,
                posixAccountProperties.getShadowWarning(), attrs);
        addDefaultToAttributes(UserAccountProperties.UID_NUMBER_KEY,
                posixAccountProperties.getUidNumber(), attrs);

        return attrs;
    }

    private static void addDefaultToAttributes(String attrId, String value,
            Attributes attrs) {
        if ((value != null) && (!value.equals(""))) {
            BasicAttribute attr = new BasicAttribute(attrId);
            attr.add(value);
            attrs.put(attr);
        }
    }

    // - generated getters and setters ----------------------------------------

    public String getFirstName() {
        return firstName;
    }

    public void setFirstName(String firstName) {
        this.firstName = firstName;
    }

    public String getFullName() {
        return fullName;
    }

    public void setFullName(String fullName) {
        this.fullName = fullName;
    }

    public String getModifiedTimestamp() {
        return modifiedTimestamp;
    }

    public void setModifiedTimestamp(String modifiedTimestamp) {
        this.modifiedTimestamp = modifiedTimestamp;
    }

    public String getNotes() {
        return notes;
    }

    public void setNotes(String notes) {
        this.notes = notes;
    }

    public void setRoles(RoleDirectory roles) {
        this.myRoleDirectory = roles;
    }

    public RoleDirectory getRoles() {
        return myRoleDirectory;
    }

    public String getSurname() {
        return surname;
    }

    public void setSurname(String surname) {
        this.surname = surname;
    }

    public String getUid() {
        return uid;
    }

    public void setUid(String uid) {
        this.uid = uid;
    }

    public void isNewEntry(boolean status) {
        this.isNewEntry = status;
    }

    @Override
    public boolean hasRole(Role inRole) {
        return this.myRoleDirectory.hasRole(inRole);
    }
}
