/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 */
package alma.userrepository.addressbook.ldap.beans;

import javax.naming.directory.Attributes;
import javax.naming.directory.BasicAttribute;
import javax.naming.directory.BasicAttributes;

import alma.userrepository.addressbook.ldap.AttributeUtilities;
import alma.userrepository.addressbook.ldap.UserInfoProperties;
import alma.userrepository.domainmodel.Address;
import alma.userrepository.domainmodel.User;
import alma.userrepository.errors.UserRepositoryException;

public class LdapAddress implements Address {
    private String city;
    private String country;
    private String department;
    private String email;
    private String fax;
    private String jobTitle;
    private String mobile;
    private String organization;
    private String pager;
    private String phone;
    private String postcode;
    private String region;
    private String streetAddress;
    private String uid;
    private String website;

    private String firstName;
    private String surname;
    private String fullname;

    private String modifiedTimestamp;

    private static UserInfoProperties ldapMapping = UserInfoProperties
            .getInstance();

    // - constructors ---------------------------------------------------------

    public LdapAddress() {
    }

    public LdapAddress(Address address) {
        if (address != null) {
            city = address.getCity();
            country = address.getCountry();
            department = address.getDepartment();
            email = address.getEmail();
            fax = address.getFax();
            jobTitle = address.getJobTitle();
            mobile = address.getMobile();
            organization = address.getOrganization();
            pager = address.getPager();
            phone = address.getPhone();
            postcode = address.getPostcode();
            region = address.getRegion();
            streetAddress = address.getStreetAddress();
            uid = address.getUid();
            website = address.getWebsite();
        }
    }

    public LdapAddress(Attributes attrs) throws UserRepositoryException {
        city = AttributeUtilities.getAttributeDefinition(attrs, ldapMapping
                .getCity());
        country = AttributeUtilities.getAttributeDefinition(attrs, ldapMapping
                .getCountry());
        department = AttributeUtilities.getAttributeDefinition(attrs,
                ldapMapping.getDepartment());
        email = AttributeUtilities.getAttributeDefinition(attrs, ldapMapping
                .getEmail());
        fax = AttributeUtilities.getAttributeDefinition(attrs, ldapMapping
                .getFax());
        jobTitle = AttributeUtilities.getAttributeDefinition(attrs, ldapMapping
                .getJobTitle());
        mobile = AttributeUtilities.getAttributeDefinition(attrs, ldapMapping
                .getMobile());
        organization = AttributeUtilities.getAttributeDefinition(attrs,
                ldapMapping.getOrganization());
        pager = AttributeUtilities.getAttributeDefinition(attrs, ldapMapping
                .getPager());
        phone = AttributeUtilities.getAttributeDefinition(attrs, ldapMapping
                .getPhone());
        postcode = AttributeUtilities.getAttributeDefinition(attrs, ldapMapping
                .getPostcode());
        region = AttributeUtilities.getAttributeDefinition(attrs, ldapMapping
                .getRegion());
        streetAddress = AttributeUtilities.getAttributeDefinition(attrs,
                ldapMapping.getStreetAddress());
        uid = AttributeUtilities.getAttributeDefinition(attrs, ldapMapping
                .getAddressUid());
        website = AttributeUtilities.getAttributeDefinition(attrs, ldapMapping
                .getWebsite());
    }

    // - public methods -------------------------------------------------------

    public void syncName(User user) {
        firstName = user.getFirstName();
        surname = user.getSurname();
        fullname = user.getFullName();
    }

    public Attributes toAttributes() {
        Attributes attrs = new BasicAttributes(true);

        BasicAttribute oc = new BasicAttribute("objectclass");
        for (String objectClass : UserInfoProperties
                .splitPropertyToList(ldapMapping.getAddressObjectClasses())) {
            oc.add(objectClass);
        }
        attrs.put(oc);

        // add address attributes
        AttributeUtilities.addPropertyToAttributes(ldapMapping.getAddressUid(),
                this.getUid(), attrs);
        AttributeUtilities.addPropertyToAttributes(ldapMapping.getCity(), this
                .getCity(), attrs);
        AttributeUtilities.addPropertyToAttributes(ldapMapping.getCountry(),
                this.getCountry(), attrs);
        AttributeUtilities.addPropertyToAttributes(ldapMapping.getDepartment(),
                this.getDepartment(), attrs);
        AttributeUtilities.addPropertyToAttributes(ldapMapping.getEmail(), this
                .getEmail(), attrs);
        AttributeUtilities.addPropertyToAttributes(ldapMapping.getFax(), this
                .getFax(), attrs);
        AttributeUtilities.addPropertyToAttributes(ldapMapping.getJobTitle(),
                this.getJobTitle(), attrs);
        AttributeUtilities.addPropertyToAttributes(ldapMapping.getMobile(),
                this.getMobile(), attrs);
        AttributeUtilities.addPropertyToAttributes(ldapMapping
                .getOrganization(), this.getOrganization(), attrs);
        AttributeUtilities.addPropertyToAttributes(ldapMapping.getPager(), this
                .getPager(), attrs);
        AttributeUtilities.addPropertyToAttributes(ldapMapping.getPhone(), this
                .getPhone(), attrs);
        AttributeUtilities.addPropertyToAttributes(ldapMapping.getPostcode(),
                this.getPostcode(), attrs);
        AttributeUtilities.addPropertyToAttributes(ldapMapping.getRegion(),
                this.getRegion(), attrs);
        AttributeUtilities.addPropertyToAttributes(ldapMapping
                .getStreetAddress(), this.getStreetAddress(), attrs);
        AttributeUtilities.addPropertyToAttributes(ldapMapping.getWebsite(),
                this.getWebsite(), attrs);

        // and user name attributes if necessary
        AttributeUtilities.addPropertyToAttributes(ldapMapping
                .getAddressFirstName(), firstName, attrs);
        AttributeUtilities.addPropertyToAttributes(ldapMapping
                .getAddressSurname(), surname, attrs);
        AttributeUtilities.addPropertyToAttributes(ldapMapping
                .getAddressFullName(), fullname, attrs);

        return attrs;
    }

    // - generated getters and setters ----------------------------------------

    public String getCity() {
        return city;
    }

    public void setCity(String city) {
        this.city = city;
    }

    public String getCountry() {
        return country;
    }

    public void setCountry(String country) {
        this.country = country;
    }

    public String getDepartment() {
        return department;
    }

    public void setDepartment(String department) {
        this.department = department;
    }

    public String getEmail() {
        return email;
    }

    public void setEmail(String email) {
        this.email = email;
    }

    public String getFax() {
        return fax;
    }

    public void setFax(String fax) {
        this.fax = fax;
    }

    public String getJobTitle() {
        return jobTitle;
    }

    public void setJobTitle(String jobTitle) {
        this.jobTitle = jobTitle;
    }

    public String getMobile() {
        return mobile;
    }

    public void setMobile(String mobile) {
        this.mobile = mobile;
    }

    public String getModifiedTimestamp() {
        return modifiedTimestamp;
    }

    public void setModifiedTimestamp(String modifiedTimestamp) {
        this.modifiedTimestamp = modifiedTimestamp;
    }

    public String getOrganization() {
        return organization;
    }

    public void setOrganization(String organization) {
        this.organization = organization;
    }

    public String getPager() {
        return pager;
    }

    public void setPager(String pager) {
        this.pager = pager;
    }

    public String getPhone() {
        return phone;
    }

    public void setPhone(String phone) {
        this.phone = phone;
    }

    public String getPostcode() {
        return postcode;
    }

    public void setPostcode(String postcode) {
        this.postcode = postcode;
    }

    public String getRegion() {
        return region;
    }

    public void setRegion(String region) {
        this.region = region;
    }

    public String getStreetAddress() {
        return streetAddress;
    }

    public void setStreetAddress(String streetAddress) {
        this.streetAddress = streetAddress;
    }

    public String getUid() {
        return uid;
    }

    public void setUid(String uid) {
        this.uid = uid;
    }

    public String getWebsite() {
        return website;
    }

    public void setWebsite(String website) {
        this.website = website;
    }

    public String getFirstName() {
        return firstName;
    }

    public void setFirstName(String firstName) {
        this.firstName = firstName;
    }

    public String getFullname() {
        return fullname;
    }

    public void setFullname(String fullname) {
        this.fullname = fullname;
    }

    public String getSurname() {
        return surname;
    }

    public void setSurname(String surname) {
        this.surname = surname;
    }

}
