/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 */
package alma.userrepository.addressbook.ldap;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.logging.Logger;

import javax.naming.AuthenticationNotSupportedException;
import javax.naming.InvalidNameException;
import javax.naming.Name;
import javax.naming.NameClassPair;
import javax.naming.NameNotFoundException;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.NoPermissionException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.BasicAttribute;
import javax.naming.directory.DirContext;
import javax.naming.directory.InvalidSearchFilterException;
import javax.naming.directory.ModificationItem;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.LdapName;

import alma.userrepository.addressbook.AddressBookSession;
import alma.userrepository.addressbook.ldap.beans.LdapAddress;
import alma.userrepository.addressbook.ldap.beans.LdapAddressBookEntry;
import alma.userrepository.addressbook.ldap.beans.LdapPreferences;
import alma.userrepository.addressbook.ldap.beans.LdapUser;
import alma.userrepository.domainmodel.Address;
import alma.userrepository.domainmodel.AddressBookEntry;
import alma.userrepository.domainmodel.Preferences;
import alma.userrepository.domainmodel.User;
import alma.userrepository.errors.DataExpiredException;
import alma.userrepository.errors.DuplicateObjectException;
import alma.userrepository.errors.InvalidAttributeException;
import alma.userrepository.errors.ObjectNotFoundException;
import alma.userrepository.errors.PermissionException;
import alma.userrepository.errors.UserRepositoryException;
import alma.userrepository.roledirectory.RoleDirectory;
import alma.userrepository.roledirectory.RoleDirectorySession;
import alma.userrepository.shared.ldap.DnDirectory;

/**
 * @author Stewart
 * 
 */
public class LdapAddressBookSession implements AddressBookSession {
    public static final String ADDRESS_RDN_KEY = "uid=";

    /**
     * The DirContext used by this session
     */
    private DirContext ctx = null;

    private RoleDirectorySession roleSession = null;

    protected Logger myLogger = null;

    @SuppressWarnings("unused")
    private LdapAddressBookSession() {
    }

    public LdapAddressBookSession(DirContext ctx,
            RoleDirectorySession roleSession, Logger inLogger) {
        this.ctx = ctx;
        this.roleSession = roleSession;
        this.myLogger = inLogger;
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.addressbook.AddressBookSession#close()
     */
    public void close() throws UserRepositoryException {
        try {
            ctx.close();
        } catch (NamingException e) {
            throw new UserRepositoryException("Couldn't close session", e);
        }
    }

    private void createNewAddresses(AddressBookEntry newEntry)
            throws UserRepositoryException {
        // new addresses will not have a UID
        // search the user's branch for almaAddresses, retrieving the UID
        SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.SUBTREE_SCOPE);
        String[] attributesToReturn = { "uid" };
        controls.setReturningAttributes(attributesToReturn);
        List<String> addressUids = new ArrayList<String>();
        try {
            String userDn = DnDirectory.getUserDn(newEntry);
            List<Map<String, String>> searchResults = searchReturnResults(
                    userDn, "(objectclass=almaAddress)", controls);
            for (Map<String, String> map : searchResults) {
                addressUids.add(map.get("uid"));
            }
        } catch (NamingException e) {
            throw new UserRepositoryException(
                    "Could not determine if almaAddresses exist", e);
        }

        for (Address address : newEntry.getAddresses()) {
            // if this is the first address, it should be merged with the
            // almaUser entry, therefore it does not need a new UID
            if (newEntry.getAddresses().indexOf(address) == 0) {
                continue;
            }

            // a new address will not have a UID, while an address that has been
            // unset as the default will have the same UID as the user. This
            // is always true as we do not allow the user entry UID to change.
            String uid = address.getUid();
            if (uid == null || uid.equals("")
                    || uid.equals(newEntry.getUser().getUid())) {
                myLogger.finer("Calculating UUID for new address entry.");
                String uuid = UUID.randomUUID().toString();
                address.setUid(uuid);
            }

            // if the address does not exist..
            if (!addressUids.contains(address.getUid())) {
                try {
                    // bind address to context
                    LdapAddress newAddress = new LdapAddress(address);
                    newAddress.syncName(newEntry.getUser());
                    Name addressDn = new LdapName(DnDirectory.getAddressDn(
                            address, newEntry));
                    myLogger.fine("Binding new address to '"
                            + addressDn.toString() + "'.");
                    ctx.bind(addressDn, newAddress);
                } catch (NoPermissionException e) {
                    throw new PermissionException(e);
                } catch (InvalidNameException e) {
                    throw new InvalidAttributeException("The UID '"
                            + address.getUid() + "' is an invalid LDAP DN: ", e);
                } catch (NamingException e) {
                    throw new UserRepositoryException(e);
                }
            } // end if does not exist test
        }
    }

    private void createNewUser(AddressBookEntry newEntry)
            throws UserRepositoryException {
        LdapUser user = new LdapUser(newEntry.getUser());
        user.isNewEntry(true);
        String userUid = newEntry.getUser().getUid();

        try {
            myLogger.info("Creating new user with uid '" + userUid + "'.");
            String userDn = DnDirectory.getUserDn(newEntry);
            ctx.bind(userDn, user);
        } catch (NoPermissionException e) {
            throw new PermissionException("No permission to create new user: ",
                    e);
        } catch (NamingException e) {
            throw new UserRepositoryException(
                    "Unexpected error occurred while creating new user:", e);
        }
    }

    private void deleteOldAddresses(AddressBookEntry newEntry,
            AddressBookEntry oldEntry) throws UserRepositoryException {
        // addresses that should be deleted from the directory will exist in the
        // old entry but not in the new one
        List<String> newUids = new ArrayList<String>();
        for (Address address : newEntry.getAddresses()) {
            newUids.add(address.getUid());
        }
        for (Address address : oldEntry.getAddresses()) {
            String uid = address.getUid();
            // don't try to delete address attached to user entry
            if (uid.equals(oldEntry.getUser().getUid())) {
                continue;
            }

            if (!newUids.contains(uid)) {
                // delete this address from the context
                String addressDn = DnDirectory.getAddressDn(address, oldEntry);
                try {
                    Name name = new LdapName(addressDn);
                    myLogger.fine("Removing " + name.toString()
                            + " from directory");
                    ctx.unbind(name);
                } catch (InvalidNameException e) {
                    throw new UserRepositoryException(
                            "Error deleting old address:" + oldEntry, e);
                } catch (NamingException e) {
                    throw new UserRepositoryException(
                            "Unexpected error occurred deleting old Address entry",
                            e);
                }
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @seealma.userrepository.addressbook.AddressBookSession#deleteUser(
     * AddressBookEntry)
     */
    public void deleteEntry(AddressBookEntry user)
            throws UserRepositoryException {
        deleteEntryByUid(user.getUser().getUid());
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.addressbook.AddressBookSession#deleteUser(java.lang
     * .String)
     */
    public void deleteEntryByUid(String uid) throws UserRepositoryException {
        // must delete child entries before parent
        deleteUserChildren(uid);

        SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.ONELEVEL_SCOPE);

        String searchFilter = "(&(objectClass=almaUser)("
                + DnDirectory.getUserRdn(uid) + "))";
        List<String> userDns = searchGetDn(searchFilter, controls);

        if (userDns.isEmpty()) {
            throw new ObjectNotFoundException("User with uid=" + uid
                    + " not found");
        }

        try {
            for (String userDn : userDns) {
                myLogger.fine("Deleting entry " + userDn + ".");
                ctx.unbind(userDn);
            }
        } catch (NoPermissionException e) {
            throw new PermissionException(
                    "No permission to delete user with uid=" + uid + ".", e);
        } catch (NamingException e) {
            throw new UserRepositoryException(
                    "Unexpected error occurred while deleting user: ", e);
        }
    }

    /**
     * Delete child entries of the given AddressBookEntry from the LDAP server.
     * 
     * @param AddressBookEntry
     *            the AddressBookEntry to delete child entries from
     * @throws UserRepositoryException
     */
    private void deleteUserChildren(String uid) throws UserRepositoryException {
        SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.ONELEVEL_SCOPE);

        List<String> userDns = searchGetLocation("("
                + DnDirectory.getUserRdn(uid) + ")");

        for (String userDn : userDns) {
            try {
                // then search for Address below the user
                for (NamingEnumeration<SearchResult> ne = ctx.search(userDn,
                        "(objectClass=almaAddress)", controls); ne
                        .hasMoreElements();) {
                    NameClassPair ncp = (NameClassPair) ne.nextElement();
                    // .. constructing the full dn of the object..
                    String addressDn = ncp.getName() + "," + userDn;
                    // .. and delete the address object.
                    myLogger.fine("Deleting entry " + addressDn + ".");
                    ctx.unbind(addressDn);
                }
            } catch (NameNotFoundException e) {
                throw new ObjectNotFoundException("Entry not found: ", e);
            } catch (NoPermissionException e) {
                throw new PermissionException(
                        "No permission to delete Address", e);
            } catch (NamingException e) {
                throw new UserRepositoryException(
                        "Unexpected error occurred while deleting Address", e);
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.addressbook.AddressBookSession#exists(java.lang.String
     * )
     */
    public boolean exists(String uid) throws UserRepositoryException {
        // We just do a ONELEVEL search as all users should be in the root level
        // of the user branch
        String searchFilter = "(&(objectClass=almaUser)("
                + DnDirectory.getUserRdn(uid) + "))";
        SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.ONELEVEL_SCOPE);

        myLogger.finer("Searching onelevel scope using the filter '"
                + searchFilter + "'.");
        List<String> results = searchGetDn(searchFilter, controls);

        if (results.isEmpty()) {
            myLogger.fine("No user with uid=" + uid + " found.");
            return false;
        } else {
            myLogger.fine(results.size() + " user with uid=" + uid + " found.");
            return true;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.addressbook.AddressBookSession#getUser(java.lang.
     * String)
     */
    public AddressBookEntry getEntry(String uid) throws UserRepositoryException {
        // 1) retrieve the user
        // 2) find addresses
        // 3) find preferences
        // 4) attach them to the AddressBookEntry structure
        // 5) return the result

        // step 1: retrieve the user
        AddressBookEntry entry = retrieveUserEntry(uid);
        User user = entry.getUser();

        // step 2: find additional addresses
        List<Address> addresses = getAddresses(uid);

        // step 3: find preferences
        Preferences preferences = getPreferences(uid);

        // now we're at step 4, insert the newly retrieved entries into the
        // object
        AddressBookEntry result = new LdapAddressBookEntry(user, addresses,
                preferences);

        return result;
    }

    private List<Address> getAddresses(String uid)
            throws UserRepositoryException {
        // first address is part of user object
        AddressBookEntry entry = retrieveUserEntry(uid);
        User user = entry.getUser();
        List<Address> addresses = entry.getAddresses();

        // now search for additional addresses below user
        SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.ONELEVEL_SCOPE);
        String userDn = DnDirectory.getUserDn(user);

        try {
            // step 2: search for almaAddresses below the user we just
            // retrieved.
            // for each almaAddress..
            for (NamingEnumeration<?> ne = ctx.search(userDn,
                    "(objectClass=almaAddress)", controls); ne
                    .hasMoreElements();) {
                NameClassPair ncp = (NameClassPair) ne.nextElement();
                // .. construct the full dn of the entry..
                String addressDn = ncp.getName() + "," + userDn;
                // .. and retrieve it..
                myLogger
                        .fine("Retrieving Address with dn '" + addressDn + "'.");
                Address address = (Address) ctx.lookup(addressDn);
                // before adding it to the Address list.
                addresses.add(address);
            }
        } catch (NameNotFoundException e) {
            throw new ObjectNotFoundException("Entry not found: ", e);
        } catch (NoPermissionException e) {
            throw new PermissionException(e);
        } catch (NamingException e) {
            throw new UserRepositoryException(e);
        }

        return addresses;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.addressbook.AddressBookSession#getUser(java.lang.
     * String)
     */
    public User getUser(String uid) throws UserRepositoryException {
        AddressBookEntry entry = retrieveUserEntry(uid);
        return entry.getUser();
    }

    public Preferences getPreferences(String uid)
            throws UserRepositoryException {
        AddressBookEntry entry = retrieveUserEntry(uid);
        return entry.getPreferences();
    }

    private AddressBookEntry retrieveUserEntry(String uid)
            throws UserRepositoryException {
        // construct search parameters thaw will identify user with given uid
        String searchFilter = "(&(objectClass=almaUser)("
                + DnDirectory.getUserRdn(uid) + "))";
        SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.ONELEVEL_SCOPE);
        myLogger.fine("Searching for users using the filter '" + searchFilter
                + "'.");
        List<String> results = searchGetDn(searchFilter, controls);

        // result should now contain one dn: that of the user with the given uid
        if (results.size() == 0) {
            myLogger.fine("No entries were found with the uid '" + uid + "'");
            throw new ObjectNotFoundException(
                    "No users were found with uid='" + uid + "'");
        }
        String userDn = results.get(0);

        // now we
        // 1) retrieve the user
        // 2) find the roles of the user
        // 3) attach the roles to the User object
        // 4) return the result
        AddressBookEntry entry = null;
        try {
            // step 1: retrieve the user
            myLogger.fine("Retrieving user entry with dn '" + userDn + "'.");
            Object o = ctx.lookup(userDn);
            if (o instanceof User) {
                // user entry is not an address? should we complain here?
                throw new UserRepositoryException(
                        "User without address is unhandled");
            } else if (o instanceof AddressBookEntry) {
                entry = (AddressBookEntry) o;
            } else {
                throw new UserRepositoryException(
                        "Cannot handle object factory result");
            }
        } catch (NameNotFoundException e1) {
            throw new ObjectNotFoundException("Entry not found: ", e1);
        } catch (NoPermissionException e2) {
            throw new PermissionException(e2);
        } catch (NamingException lastE) {
            throw new UserRepositoryException(lastE);
        }

        // now we're at step 2, find the roles
        User user = entry.getUser();
        if (user instanceof LdapUser) {
            RoleDirectory roles = roleSession.getAllUserRoles(user.getUid());
            // step 3: attach the roles to the User
            ((LdapUser) user).setRoles(roles);
        }

        return entry;
    }

    /*
     * (non-Javadoc)
     * 
     * @see alma.userrepository.addressbook.AddressBookSession#listUsers()
     */
    public List<String> listUsers() throws UserRepositoryException {
        // search and retrieve uid of all almaUsers. We just do a ONELEVEL
        // search as all users should be in the root of the user branch
        String searchFilter = "(objectClass=almaUser)";
        SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.ONELEVEL_SCOPE);
        String[] searchAttributes = { "uid" };
        controls.setReturningAttributes(searchAttributes);

        myLogger.finer("listUsers(): using search filter '" + searchFilter
                + "'.");
        List<Map<String, String>> results = searchGetAttributes(searchFilter,
                controls);
        List<String> uids = new ArrayList<String>();

        for (Map<String, String> map : results) {
            uids.add(map.get("uid"));
        }

        return uids;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.addressbook.AddressBookSession#searchGetAttributes
     * (java.lang.String, java.util.List)
     */
    public List<Map<String, String>> searchGetAttributes(String searchFilter,
            List<String> attributes) throws UserRepositoryException {
        String searchBase = DnDirectory.PEOPLE_BASE;
        return searchLocationGetAttributes(searchBase, searchFilter, attributes);
    }

    /*
     * (non-Javadoc)
     * 
     * @seealma.userrepository.addressbook.AddressBookSession#
     * searchLocationGetAttributes(java.lang.String, java.lang.String,
     * java.util.List)
     */
    public List<Map<String, String>> searchLocationGetAttributes(
            String searchBase, String searchFilter, List<String> attributes)
            throws UserRepositoryException {
        SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.SUBTREE_SCOPE);
        controls.setReturningAttributes(attributes
                .toArray(new String[attributes.size()]));

        return searchGetAttributes(searchBase, searchFilter, controls);
    }

    /**
     * Return attributes from the people branch for objects matching the given
     * search filter.
     * 
     * @param searchFilter
     *            the LDAP filter to match
     * @param controls
     *            the SearchControls containing the attributes and search scope
     * @return a list of attributes of matching objects
     * @throws UserRepositoryException
     */
    private List<Map<String, String>> searchGetAttributes(String searchFilter,
            SearchControls controls) throws UserRepositoryException {
        String searchBase = DnDirectory.PEOPLE_BASE;
        return searchGetAttributes(searchBase, searchFilter, controls);
    }

    /**
     * Return attributes for objects matching the given search filter, beginning
     * the search from the given location.
     * 
     * @param searchBase
     *            the location from where to start the search
     * @param searchFilter
     *            the LDAP filter to match
     * @param controls
     *            the SearchControls containing the attributes and search scope
     * @return a list of attributes of matching objects
     * @throws UserRepositoryException
     */
    private List<Map<String, String>> searchGetAttributes(String searchBase,
            String searchFilter, SearchControls controls)
            throws UserRepositoryException {
        List<Map<String, String>> results;
        try {
            myLogger.finer("Searching " + searchBase + " with filter '"
                    + searchFilter + "'.");
            results = searchReturnResults(searchBase, searchFilter, controls);
        } catch (InvalidNameException e) {
            throw new ObjectNotFoundException("Invalid search base given: "
                    + searchBase, e);
        } catch (InvalidSearchFilterException e) {
            throw new alma.userrepository.errors.InvalidSearchFilterException(
                    "Invalid search filter given: " + searchFilter, e);
        } catch (NamingException e) {
            throw new UserRepositoryException(e);
        }

        return results;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.addressbook.AddressBookSession#searchGetDn(java.lang
     * .String)
     */
    public List<String> searchGetLocation(String searchFilter)
            throws UserRepositoryException {
        SearchControls controls = new SearchControls();
        controls.setSearchScope(SearchControls.SUBTREE_SCOPE);

        myLogger
                .finer("searchGetDn(): searching subtree scope using the filter '"
                        + searchFilter + "'.");
        return searchGetDn(searchFilter, controls);
    }

    /**
     * Return the DN of objects matching the given search filter.
     * 
     * @param searchFilter
     *            the LDAP search filter to match
     * @param controls
     *            the SearchControls specifying scope
     * @return the DN list of matching objects
     * @throws UserRepositoryException
     */
    private List<String> searchGetDn(String searchFilter,
            SearchControls controls) throws UserRepositoryException {
        String[] searchAttributes = { "" };
        controls.setReturningAttributes(searchAttributes);

        List<String> results = new ArrayList<String>();
        try {
            myLogger.finer("searchGetDn(): Searching using the filter '"
                    + searchFilter + "'.");
            for (NamingEnumeration<SearchResult> ne = ctx.search(
                    DnDirectory.PEOPLE_BASE, searchFilter, controls); ne
                    .hasMoreElements();) {
                SearchResult result = ne.nextElement();
                myLogger.fine("Adding '" + result.getNameInNamespace()
                        + "' to search results.");
                results.add(result.getNameInNamespace());
            }
        } catch (InvalidSearchFilterException e) {
            throw new alma.userrepository.errors.InvalidSearchFilterException(
                    "Invalid search filter given: " + searchFilter, e);
        } catch (NamingException e) {
            throw new UserRepositoryException(e);
        }

        return results;
    }

    private List<Map<String, String>> searchReturnResults(String searchBase,
            String searchFilter, SearchControls controls)
            throws NamingException {
        // search for objects satisfying the filter
        List<Map<String, String>> results = new ArrayList<Map<String, String>>();

        myLogger.finer("Searching " + searchBase + " using the filter '"
                + searchFilter + "'.");
        for (NamingEnumeration<SearchResult> ne = ctx.search(searchBase,
                searchFilter, controls); ne.hasMoreElements();) {
            SearchResult result = ne.next();
            Attributes resultAttrs = result.getAttributes();
            Map<String, String> dict = new HashMap<String, String>();
            for (NamingEnumeration<? extends Attribute> resultAttr = resultAttrs
                    .getAll(); resultAttr.hasMoreElements();) {
                Attribute attr = resultAttr.nextElement();
                String id = attr.getID();
                for (NamingEnumeration<?> vals = attr.getAll(); vals.hasMore();) {
                    String s = (String) vals.next();
                    myLogger.finer("Found match; adding attibute value '" + s
                            + "' to the results.");
                    dict.put(id, s);
                }
            }
            results.add(dict);
        }
        return results;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.addressbook.AddressBookSession#setPassword(java.lang
     * .String, java.lang.String)
     */
    public void setPassword(String uid, String password)
            throws UserRepositoryException {
        ModificationItem[] mods = new ModificationItem[1];
        mods[0] = new ModificationItem(DirContext.REPLACE_ATTRIBUTE,
                new BasicAttribute("userPassword", password));

        try {
            myLogger.fine("Changing password for user '" + uid + "'.");
            ctx.modifyAttributes(DnDirectory.getUserDn(uid), mods);
        } catch (NameNotFoundException e) {
            throw new ObjectNotFoundException("Entry not found: ", e);
        } catch (NoPermissionException e) {
            throw new PermissionException("No permission to change password: ",
                    e);
        } catch (NamingException e) {
            throw new UserRepositoryException(e);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.userrepository.addressbook.AddressBookSession#setPreferences(java
     * .lang.String, alma.userrepository.domainmodel.Preferences)
     */
    public void setPreferences(String uid, Preferences preferences)
            throws UserRepositoryException {
        myLogger.finer("Retrieving existing user with uid '" + uid + "'.");
        Preferences oldPreferences = getPreferences(uid);

        // calculate difference between new and old entries
        LdapPreferences oldPrefs = new LdapPreferences(oldPreferences);
        LdapPreferences newPrefs = new LdapPreferences(preferences);

        Attributes oldAttrs = oldPrefs.toAttributes();
        Attributes newAttrs = newPrefs.toAttributes();
        ModificationItem[] mods = AttributeUtilities.getDifference(newAttrs,
                oldAttrs);

        // write user modifications to the directory
        try {
            myLogger.fine("Modifying attributes for preferences: " + mods);
            String userDn = DnDirectory.getUserDn(uid);
            ctx.modifyAttributes(userDn, mods);
        } catch (NoPermissionException e) {
            throw new PermissionException(e);
        } catch (AuthenticationNotSupportedException e) {
            throw new PermissionException(e);
        } catch (NamingException e) {
            throw new UserRepositoryException(e);
        }
    }

    private void updateAddresses(AddressBookEntry newEntry)
            throws UserRepositoryException {
        // user entry always contains an address
        if (newEntry.getAddresses().size() == 0) {
            newEntry.getAddresses().add(new LdapAddress());
        }

        // set uid of first address to that of the user
        if (newEntry.getAddresses().size() > 0) {
            newEntry.getAddresses().get(0).setUid(newEntry.getUser().getUid());
        }

        createNewAddresses(newEntry);

        // retrieve the old entry, which should now include the new addresses
        String userUid = newEntry.getUser().getUid();
        myLogger.finer("Retrieving old entry with uid=" + userUid + ".");
        AddressBookEntry oldEntry = getEntry(userUid);

        deleteOldAddresses(newEntry, oldEntry);
        updateAddresses(newEntry, oldEntry);
    }

    private void updateAddresses(AddressBookEntry newEntry,
            AddressBookEntry oldEntry) throws UserRepositoryException {
        // addresses that should be updated will have the same UUID in both
        Map<String, Address> oldMap = new HashMap<String, Address>();
        for (Address address : oldEntry.getAddresses()) {
            oldMap.put(address.getUid(), address);
        }

        // for each Address in the new entry
        for (Address address : newEntry.getAddresses()) {
            // if it already exists in the old entry or is the address attached
            // to the user entry
            if (oldMap.containsKey(address.getUid())
                    || newEntry.getAddresses().indexOf(address) == 0) {
                LdapAddress newAddress = new LdapAddress(address);
                newAddress.syncName(newEntry.getUser());
                LdapAddress oldAddress = new LdapAddress(oldMap.get(address
                        .getUid()));
                oldAddress.syncName(oldEntry.getUser());

                // check the data is still current
                if ((newAddress instanceof LdapAddress)
                        && (oldAddress instanceof LdapAddress)) {
                    LdapAddress newLdapAddress = (LdapAddress) newAddress;
                    LdapAddress oldLdapAddress = (LdapAddress) oldAddress;

                    // the timestamp will be null if the address is new
                    // so test 'if the address existed previously and the
                    // modification dates are different'
                    if ((newLdapAddress.getModifiedTimestamp() != null)
                            && (!newLdapAddress.getModifiedTimestamp().equals(
                                    oldLdapAddress.getModifiedTimestamp()))) {
                        throw new DataExpiredException(
                                "Address '"
                                        + newAddress.getUid()
                                        + "' has been modified since the object was retrieved");
                    }
                } else {
                    // so the DAOs were not LDAP beans?
                    throw new InvalidAttributeException(
                            "LDAP back-end cannot handle non-LDAP data objects");
                }

                // get the attributes for new and old Addresses
                Attributes newAttrs = newAddress.toAttributes();
                Attributes oldAttrs = oldAddress.toAttributes();

                // find the difference
                ModificationItem[] addrMods = AttributeUtilities.getDifference(
                        newAttrs, oldAttrs);

                // commit the modificationItems to the context

                // first address is written to user entry
                String addressDn = null;
                if (newEntry.getAddresses().indexOf(address) == 0) {
                    addressDn = DnDirectory.getUserDn(newEntry);
                } else {
                    addressDn = DnDirectory.getAddressDn(address, newEntry);
                }

                try {
                    Name name = new LdapName(addressDn);
                    ctx.modifyAttributes(name, addrMods);
                } catch (InvalidNameException e) {
                    throw new UserRepositoryException("Cannot update details ("
                            + addressDn + " is not a valid LDAP dn)", e);
                } catch (NamingException e) {
                    throw new UserRepositoryException(e);
                }
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @seealma.userrepository.addressbook.AddressBookSession#commitUser(
     * AddressBookEntry)
     */
    public void commitEntry(AddressBookEntry newEntry)
            throws UserRepositoryException {
        String userUid = newEntry.getUser().getUid();

        if (exists(userUid)) {
            // check for duplicate users
            if (newEntry.getUser() instanceof LdapUser) {
                LdapUser newUser = (LdapUser) newEntry.getUser();
                if (newUser.getModifiedTimestamp() == null) {
                    myLogger.finer("User with uid '" + userUid
                            + "' already exists.");
                    throw new DuplicateObjectException("User with uid '"
                            + userUid + "' already exists");
                }
            }

            myLogger.finer("Updating user with uid '" + userUid + "'.");
            updateUser(newEntry);
        } else {
            myLogger.finer("User with uid '" + userUid
                    + "' does not exist. Creating new user..");
            createNewUser(newEntry);
        }

        setPreferences(userUid, newEntry.getPreferences());
        updateAddresses(newEntry);
    }

    private void updateUser(AddressBookEntry newEntry)
            throws UserRepositoryException {
        String userDn = DnDirectory.getUserDn(newEntry);

        String userUid = newEntry.getUser().getUid();
        myLogger.finer("Retrieving existing user with uid '" + userUid + "'.");
        AddressBookEntry oldEntry = getEntry(userUid);

        if ((newEntry.getUser() instanceof LdapUser)
                && (oldEntry.getUser() instanceof LdapUser)) {
            LdapUser newUser = (LdapUser) newEntry.getUser();
            LdapUser oldUser = (LdapUser) oldEntry.getUser();

            if (!newUser.getModifiedTimestamp().equals(
                    oldUser.getModifiedTimestamp())) {
                throw new DataExpiredException("User '" + newUser.getUid()
                        + "' has been modified since the object was retrieved");
            }
        } else {
            // so the DAOs were not LDAP beans?
            throw new InvalidAttributeException(
                    "LDAP back-end cannot handle non-LDAP data objects");
        }

        // calculate difference between new and old entries
        LdapUser newUser = (LdapUser) newEntry.getUser();
        LdapUser oldUser = (LdapUser) oldEntry.getUser();

        Attributes newAttrs = newUser.toAttributes();
        Attributes oldAttrs = oldUser.toAttributes();
        ModificationItem[] mods = AttributeUtilities.getDifference(newAttrs,
                oldAttrs);

        // write user modifications to the directory
        try {
            ctx.modifyAttributes(userDn, mods);
        } catch (NoPermissionException e) {
            throw new PermissionException(e);
        } catch (NamingException e) {
            throw new UserRepositoryException(e);
        }
    }

}
