/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 */
package alma.userrepository.addressbook.ldap;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;

import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.BasicAttribute;
import javax.naming.directory.DirContext;
import javax.naming.directory.ModificationItem;

import alma.userrepository.errors.UserRepositoryException;

public class AttributeUtilities {
	protected static UserInfoProperties ldapMapping = UserInfoProperties
			.getInstance();

	protected static UserAccountProperties userProperties = UserAccountProperties
			.getInstance();

	private AttributeUtilities() {
	}

	public static final String MODIFY_ATTR = "modifyTimestamp";

	public static final String[] MODIFY_ARRAY = { MODIFY_ATTR };

	static ModificationItem[] getDifference(Attributes newEntry,
			Attributes oldEntry) {
		List<ModificationItem> modifications = new ArrayList<ModificationItem>();

		// // TODO can we replace these tests with .REPLACE_ATTRIBUTE?
		//
		// // for each item in new that is not in old
		// // make a 'create' modification item
		// for (NamingEnumeration<? extends Attribute> ne = newEntry.getAll();
		// ne
		// .hasMoreElements();) {
		// Attribute attr = ne.nextElement();
		// if (oldEntry.get(attr.getID()) == null) {
		// modifications.add(new ModificationItem(
		// DirContext.ADD_ATTRIBUTE, attr));
		// }
		// }
		//

		// for each item in new that is defined differently to that in old
		// make a 'change' modification item
		for (NamingEnumeration<? extends Attribute> ne = newEntry.getAll(); ne
				.hasMoreElements();) {
			Attribute attr = ne.nextElement();

			// don't want to change object class of entries
			if (attr.getID().equals("objectclass"))
				continue;

			// don't change UIDs
			if (attr.getID().equals("uid"))
				continue;

			modifications.add(new ModificationItem(
					DirContext.REPLACE_ATTRIBUTE, attr));
		}

		// it turns out that using DirContext.REPLACE_ATTRIBUTE with an empty
		// value does not delete the attribute from the definition (as the
		// documentation suggests it should), so we need to explicitly use
		// DirContext.REMOVE_ATTRIBUTE

		// for each item in old that is not in new
		// make a 'delete' modification item
		for (NamingEnumeration<? extends Attribute> ne = oldEntry.getAll(); ne
				.hasMoreElements();) {
			Attribute attr = ne.nextElement();
			if ((newEntry.get(attr.getID()) == null)
					|| (newEntry.get(attr.getID()).equals(""))) {
				// fax has no matching rule in RFC, so we can't delete a
				// specific value; instead we have to replace or delete all
				if (attr.getID().equals(ldapMapping.getFax())) {
					Attribute fax = new BasicAttribute(ldapMapping.getFax());
					ModificationItem mod = new ModificationItem(
							DirContext.REMOVE_ATTRIBUTE, fax);
					modifications.add(mod);
				} else {
					modifications.add(new ModificationItem(
							DirContext.REMOVE_ATTRIBUTE, attr));
				}
			}
		}

		return (ModificationItem[]) modifications
				.toArray(new ModificationItem[modifications.size()]);
	}

	public static void addPropertyToAttributes(String mappingId, String value,
			Attributes attrs) {
		// if this property is defined..
		if ((value != null) && (!value.equals(""))) {
			// .. then for each LDAP attribute that this property maps to..
			for (String attrId : UserInfoProperties
					.splitPropertyToList(mappingId)) {
				// .. create a new LDAP attribute, first setting the attribute
				// key..
				BasicAttribute attr = new BasicAttribute(attrId);
				// .. then setting the attribute value
				attr.add(value);
				// then add it to the collected attributes.
				attrs.put(attr);
			}
		}
	}

	public static String getAttributeDefinition(Attributes attrs,
			String propertyValue) throws UserRepositoryException {
		List<String> results = new ArrayList<String>();
		String result = null;

		// for each LDAP attribute that this object property maps to..
		for (String attrID : UserInfoProperties
				.splitPropertyToList(propertyValue)) {
			// .. retrieve attribute if it exists on the LDAP entry
			Attribute attr = attrs.get(attrID);

			// if this attribute was defined..
			if (attr != null) {
				// .. get all the attribute's values..
				Enumeration<?> e;
                try {
                    e = attr.getAll();
                } catch (NamingException ne) {
                    throw new UserRepositoryException("Unexpected problem reading LDAP attributes.", ne);
                }

				// .. and create a List<String> from those values.
				List<String> attrValues = new ArrayList<String>();
				while (e.hasMoreElements()) {
					attrValues.add((String) e.nextElement());
				}

				// Throw an exception if multiple definitions are present, as
				// multiple definitions breaks the Castor interface
				if (attrValues.size() > 1) {
					throw new UserRepositoryException("Multiple definitions found for LDAP attribute " + attrID);
				}

				// .. else we add the definition to the results set.
				if (!attrValues.isEmpty()) {
					String firstValue = attrValues.get(0);

					if (!results.contains(firstValue)) {
						results.add(firstValue);
					}
				}
			}
		}

		// set final result to the value of the first definition
		if (!results.isEmpty()) {
			result = results.get(0);
		}

		return result;
	}

}
