/*
 * 	  Created on 24-Oct-2003
 * 
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 */
package alma.archive.loader;

import java.io.File;
import java.io.FileFilter;
import java.io.FileReader;
import java.io.IOException;
import java.io.StringReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.Namespace;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

import alma.JavaContainerError.wrappers.AcsJContainerServicesEx;
import alma.acs.component.client.ComponentClient;
import alma.acs.container.archive.Range;
import alma.acs.logging.ClientLogManager;
import alma.archive.database.interfaces.IdentifierManager;
import alma.archive.database.interfaces.InternalIF;
import alma.archive.database.interfaces.InternalIFFactory;
import alma.archive.database.interfaces.SchemaManager;
import alma.archive.exceptions.ArchiveException;
import alma.archive.exceptions.ModuleCriticalException;
import alma.archive.exceptions.general.DatabaseException;
import alma.archive.exceptions.general.NamespaceDefinedException;
import alma.archive.exceptions.general.UnknownSchemaException;
import alma.archive.exceptions.syntax.MalformedURIException;
import alma.archive.wrappers.Permissions;
import alma.xmlstore.Administrative;
import alma.xmlstore.ArchiveConnection;
import alma.xmlstore.ArchiveConnectionHelper;
import alma.xmlstore.ArchiveInternalError;

/**
 * @author simon
 */
public class SchemaLoader {

    // the following two variables are used to provide a summary of the errors occurred
    // they are modified in method load() and addSchema only.
		static String errorString = new String();
		static boolean allFine=true;
	
	private class SuffixFilter implements FileFilter {
		private String suffix;

		public SuffixFilter(String suffix) {
			this.suffix = suffix;
		}

		public boolean accept(File pathname) {
			String name = pathname.getName();
			if (name.endsWith(suffix))
				return true;
			else
				return false;
		}
	}

	/**
	 * used for informing Archive/ACS components that new schemas arrived.
	 * @author hmeuss
	 *
	 */
	private class AcsClient extends ComponentClient {

		private String connName = "ARCHIVE_CONNECTION";

		/**
		 * @param logger
		 * @param managerLoc
		 * @param clientName
		 * @throws Exception
		 * @throws Exception
		 */
		public AcsClient(Logger logger, String managerLoc, String clientName) throws Exception {
			super(logger, managerLoc, clientName);
		}

		/**
		 * if ACS is running, call reinit on administrative component, so that
		 * Archive components are informed that new schemas are there
		 */
		public boolean informACScomponents() {
			
			// get reference to Archive Connection:
			ArchiveConnection conn = null;
			try {
				conn = ArchiveConnectionHelper.narrow(getContainerServices().getComponent(connName));
			} catch (AcsJContainerServicesEx e) {
				m_logger.log(Level.WARNING, "Failed to connect to archive components. If any are running, they will not know about new schemas, better restart them.");
				return false;
			}
			Administrative admin;
			try {
				admin = conn.getAdministrative("ArchiveSchemaLoader");
			} catch (Exception e) {
				m_logger.log(Level.SEVERE, "Failed to connect to 'administrative' archive component. Restart ACS to propagate the schema change.");
				return false;
			}
			try {
				admin.reinit("almabtrieb"); 
			} catch (ArchiveInternalError e1) {
				m_logger.log(Level.SEVERE, "Failed to reinitialize the 'administrative' archive component.");
				return false;
			}
			return true;
		}
	}

	//private String inpath = "/home/alma/ACS-3.0/ACSSW/idl/";
	private SchemaManager smanager;

	private InternalIF internal;

	private IdentifierManager imanager;

	private URI identifier;
	
	private static Logger m_logger;
	
	private Range range;

	private SAXBuilder builder = new SAXBuilder();
	
	public SchemaLoader() {

		try {
			m_logger = ClientLogManager.getAcsLogManager().getLoggerForApplication("SchemaLoader", false);
		
			
			internal = InternalIFFactory.getInternalIF(m_logger);
			smanager = internal.getSchemaManager("user");
			imanager = InternalIFFactory.getIdentifierManager(m_logger);

			// initialize Archive:
			internal.init();

				range = new Range(imanager.getNewRange());
			identifier = range.next();
		} catch (ModuleCriticalException e) {
			System.out.println("Critical exception:\n"
					+ e.getCause().toString());
		} catch (Throwable e) {
		System.out.println("Critical Exception:\n"+e.toString());
		}
	}

	private HashMap checkNamespaces(List namespaces) throws DatabaseException,
			ModuleCriticalException, ArchiveException {
		HashMap<String, String> errors = new HashMap<String, String>();
		HashMap<String, String> registered = smanager.namespaces();
		Iterator iter = namespaces.iterator();
		while (iter.hasNext()) {
			Namespace ns = (Namespace) iter.next();
			String prefix = ns.getPrefix();
			String uri = ns.getURI();
			if (registered.containsKey(prefix)) {
				String space = registered.get(prefix);
				if (!space.equalsIgnoreCase(uri)) {
					errors.put(prefix, space);
				}
			}
		}
		return errors;
	}

	// public, since we need it in the test of the internal IF
	public void registerNamespaces(List namespaces) throws DatabaseException,
			MalformedURIException, ModuleCriticalException, ArchiveException,
			URISyntaxException {
		ListIterator iter = namespaces.listIterator();
		while (iter.hasNext()) {
			Namespace ns = (Namespace) iter.next();
			try {
				smanager
						.registerNamespace(ns.getPrefix(), new URI(ns.getURI()));
			} catch (NamespaceDefinedException e) {
				throw new DatabaseException(e);
			}
		}
	}

	// public, since we need it in the test of the internal IF
	public void associateNamespaces(List namespaces, URI schemaUri)
			throws DatabaseException, UnknownSchemaException, ModuleCriticalException, ArchiveException {
		ListIterator iter = namespaces.listIterator();
		while (iter.hasNext()) {
			Namespace ns = (Namespace) iter.next();
			String nsname = ns.getPrefix();
			smanager.assignNamespace(nsname, schemaUri);
		}
	}

	/**
	 * @param namespaces
	 * @param targetUri
	 */
	// public, since we need it in the test of the internal IF
	public void disassociateNamespaces(List namespaces, URI schemaUri) throws ModuleCriticalException, DatabaseException, ArchiveException {
		ListIterator iter = namespaces.listIterator();
		while (iter.hasNext()) {
			Namespace ns = (Namespace) iter.next();
			String nsname = ns.getPrefix();
			try {
			smanager.withdrawNamespace(nsname, schemaUri);
			} catch (Exception e) {
				// nothing bad. MicroArchive throws this exception if namespace was not defined
				// TODO fix this! Only NullPointerException was caught (what I removed now), but if we see an exception here,
				// we should assume that something is wrong in the code!
				m_logger.warning(e.toString());
			}
		}
	}

	private String addTab(String output, String text, int tab, boolean good) {
		int left = tab - output.length();
		for (int j = 0; j < left; j++) {
			output += " ";
		}
		if (good) {
			output += (char) 27 + "[33;32m" + text + (char) 27 + "[m";
		} else {
			output += (char) 27 + "[33;31m" + text + (char) 27 + "[m";
		}
		return output;
	}

    /** returns true, if things went fine, in error case false **/
	private boolean addSchema(File schema, boolean update) {
	    boolean retValue=true;
		String filename = schema.getName();
		String output = "Processing: " + filename + " ";

		// Get the name to be used for the schema
		String name = filename.substring(0, filename.indexOf("."));

		boolean stored = false; // if the schema is really stored, becomes true (in case of errors or no updates, stays false)
		
		try {
			FileReader freader = new FileReader(schema);
			
			if (!freader.ready()) {
				throw new IOException("ERROR: Can not read from file "+schema);
			}
			
			//FileInputStream input = new FileInputStream(schema);
			//Reader freader = new InputStreamReader(input);
//			while (!freader.ready()) {
//				System.out.print(freader.ready()+"-"+input.available());
//				try {
//					Thread.sleep(1000);
//				} catch (InterruptedException e) {
//					// TODO Auto-generated catch block
//					e.printStackTrace();
//				}
//			}
			
			Document doc = builder.build(freader);
			//input.close();
			freader.close();
			freader=null;
			//input=null;
			Element root = doc.getRootElement();
			List namespaces = root.getAdditionalNamespaces();

			HashMap errors = checkNamespaces(namespaces);
			if (errors.isEmpty()) {
				XMLOutputter out = new XMLOutputter(Format.getRawFormat());
				String xml = out.outputString(doc);

				URI targetUri = null;
				URI newURI = identifier;
				try {
					identifier = range.next();
					targetUri = smanager.getSchemaURI(name);
					if (update) {
						output = output + "Updating: "+newURI;
						//smanager.removeSchema(name);
						smanager.updateSchema(name, xml, "", targetUri, newURI, "schemaLoader",
								new Permissions());
						stored = true;
						identifier = range.next();
					} else {
						output = output + "No Update ";
					}
				} catch (UnknownSchemaException e) {
					// this means we have to add the schema as a new one
					try {
						smanager.addSchema(name, xml, "", newURI, "schemaLoader",
								new Permissions());
						stored = true;
					} catch (ModuleCriticalException e1) {
						System.out.println("Critical exception:\n"
								+ e1.getCause().toString());
						output = addTab(output, " Failed", 70, false);
						retValue=false;
						stored = false;
					}
					//System.out.print(" Storing ");
					output = output + "Storing: "+newURI;
				} catch (ModuleCriticalException e) {
					System.out.println("Critical exception:\n"
							+ e.getCause().toString());
					output = addTab(output, " Failed", 70, false);
						retValue=false;
					stored = false;
				} catch (Throwable e) {
					System.out.println("Critical Exception:\n"+e.toString());
					output = addTab(output, " Failed", 70, false);
						retValue=false;
					stored = false;					
				}

				try {
					// TODO remove namespaces from old schema. But be carefull! If namespace is also registered for another schema, it must stay!
					// Probably rethink all this namespace storage issue. 
					registerNamespaces(namespaces);
					//System.out.println("Namespaces:");
					//for (Iterator it=namespaces.iterator(); it.hasNext(); ) {
					//		System.out.println(it.next());
					//}
					
					// get namespaces from targetUri and remove them
					if (stored && targetUri!=null) {
						String oldSchema;
						oldSchema=internal.get(targetUri, "schemaLoader");
						List oldNamespaces = builder.build(new StringReader(oldSchema)).getRootElement().getAdditionalNamespaces();
						//System.out.println("Old namespaces:");
						//for (Iterator it=oldNamespaces.iterator(); it.hasNext(); ) {
						//		System.out.println(it.next());
						//}
						//System.out.println("... disassociating from "+targetUri);
						disassociateNamespaces(oldNamespaces, targetUri);			
					}
					if (stored) {associateNamespaces(namespaces, newURI);}
				} catch (ModuleCriticalException e1) {
					System.out.println("Critical exception:\n"
							+ e1.getCause().toString());
					output = addTab(output, " Failed", 70, false);
						retValue=false;
				}

				output = addTab(output, " Done", 70, true);
				System.out.println(output);
			} else {
				output = addTab(output, " Failed", 70, false);
						retValue=false;
				System.out.println(output);
				Iterator iter = errors.keySet().iterator();
				while (iter.hasNext()) {
					String tag = (String) iter.next();
					String space = (String) errors.get(tag);
					System.out.println("ERROR: " + tag
							+ " already registered as " + space);
				}
			}
			doc = null;
		} catch (JDOMException e) {
			output = addTab(output, " Failed", 70, false);
						retValue=false;
			System.out.println(output);
			e.printStackTrace();
		} catch (IOException e) {
			output = addTab(output, " Failed", 70, false);
						retValue=false;
			System.out.println(output);
			e.printStackTrace();
		} catch (ArchiveException e) {
			output = addTab(output, " Failed", 70, false);
						retValue=false;
			System.out.println(output);
			e.printStackTrace();
		} catch (URISyntaxException e) {
			output = addTab(output, " Failed", 70, false);
						retValue=false;
			System.out.println(output);
			e.printStackTrace();
		} catch (ModuleCriticalException e) {
			output = addTab(output, " Failed", 70, false);
						retValue=false;
			System.out.println(output);
			System.out.println("Critical exception:\n"
					+ e.getCause().toString());
		}
		if (!retValue) {
		    // error case, add to errorString:
		    errorString+="\n"+output;
		}
		return retValue;
	}

	public void load(String inpaths, boolean update) {
		String[] paths = inpaths.split(":");
		for (int i = 0; i < paths.length; i++) {
			File path = new File(paths[i]);
			if (path.exists()) {
				if (path.isDirectory()) {
					System.out.println("Loading .xsd files from: " + paths[i]);

					SuffixFilter filter = new SuffixFilter("xsd");
					File[] files = path.listFiles(filter);
					for (int x = 0; x < files.length; x++) {
						allFine = addSchema(files[x], update) && allFine;
					}
				} else {
					System.out.println("Loading: " + paths[i]);
					addSchema(path, update);
				}
			} else {
				System.out
						.println("ERROR: " + paths[i] + " could not be found");
			}
		}
	}

	public void loadIntList(String inpaths, boolean update) {
		String[] paths = inpaths.split(":");
		for (int i = 0; i < paths.length; i++) {
			String path = paths[i] + "/idl";
			load(path, update);
		}
	}

	public void clear() throws DatabaseException, MalformedURIException,
			ModuleCriticalException, ArchiveException {
		smanager.removeAll();
	}

	public void close() throws DatabaseException, ModuleCriticalException {
		imanager.close();
		smanager.close();
		internal.close();
	}

	public static void main(String[] args) throws Exception {
	    allFine=true;
	    errorString=new String();
		if (args.length < 1) {
			System.out.println("Usage: archiveLoadSchema -[cul] paths");
			System.out.println("                           c Clear all schema");
			System.out.println("                           u Update");
			System.out
					.println("                           l Load from an INTLIST");
		} else {
			
			SchemaLoader sl = new SchemaLoader();
			if (args[0].startsWith("-")) {
				// Clear the schema
				if (args[0].indexOf("c") != -1) {
					sl.clear();
				}
				// Set the update flag
				boolean update = false;
				if (args[0].indexOf("u") != -1) {
					update = true;
				}
				// Load from an intlist
				if (args[0].indexOf("l") != -1) {
					for (int x = 1; x < args.length; x++) {
						sl.loadIntList(args[x], update);
					}
				}
				// Load from a normal file list
				else {
					for (int x = 1; x < args.length; x++) {
						sl.load(args[x], update);
					}
				}
			} else // no arguments
			{
				for (int x = 0; x < args.length; x++) {
					sl.load(args[x], false);
				}
			}
//			 check whether ACS is running. If not, call constructor with null
			// arguments, else call it with real managerLoc argument

			// get location of manager
			if (System.getProperty("ACS.manager") != null) {
				String managerLoc = System.getProperty("ACS.manager").trim();
				System.out
						.println("Archive SchemaLoader: Trying to inform ACS components using manager location: "+managerLoc);
			
				// PrintStream origStdout = System.out;
				try {
					// PrintStream nullPrintStream = new PrintStream(new ByteArrayOutputStream(0));
					// System.setOut(nullPrintStream);
					AcsClient client = sl.new AcsClient(m_logger, managerLoc, "ArchiveSchemaLoader");
					// System.setOut(origStdout);
					if (client.informACScomponents()) {
						System.out.println("Archive/ACS component informed.");
					}
					else {
						System.out.println("No Archive components informed about new schemas.");
					}
				} catch (Exception e) {
					// System.setOut(origStdout);
					System.out.println("ACS not running or not available. No Archive components informed about new schemas.");
				}
			} else {
				System.out
						.println("Archive SchemaLoader: Running in standalone mode.");
			}
			// provide summary of errors
			if (!allFine) {
			    System.out.println("WARNING! The following errors occurred (summary of errors mentioned above): ");
			    System.out.println(errorString);
			} else {
				System.out.println("SUMMARY: All schemas loaded without errors.");
			}
			sl.close();
		}
	}
}
