/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 */
package alma.archive.loader;

import java.net.InetAddress;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.util.StringTokenizer;
import java.util.logging.Logger;

import org.omg.CORBA.ORB;

import com.cosylab.CDB.DAL;
import com.cosylab.CDB.DALHelper;

import alma.acs.component.client.AdvancedComponentClient;
import alma.acs.container.AcsManagerProxy;
import alma.acs.container.ContainerServices;
import alma.archive.database.interfaces.InternalIF;
import alma.archive.database.interfaces.InternalIFFactory;
import alma.archive.database.interfaces.SchemaManager;
import alma.archive.exceptions.ModuleCriticalException;
import alma.archive.exceptions.general.DatabaseException;

/**
 * Loads the CDB into the archive.
 * 
 * @author hsommer Jun 4, 2003 10:37:24 AM
 */
public class Loader 
{
	private ContainerServices m_services;

	private ORB m_orb;

	private DAL m_dal;
	private Logger m_logger;
	
	private InternalIF internal;
	private SchemaManager smanager;

	/**
	 * 
	 */
	public Loader(ContainerServices services, ORB orb) throws Exception
	{
		m_services = services;
		m_orb = orb;
		m_logger = services.getLogger();
		internal = InternalIFFactory.getInternalIF(m_logger);
		smanager = internal.getSchemaManager("cdb");
	}
	
	/**
	 * Disconnects from the archive internal interface
	 *
	 */
	private void disConnect() throws ModuleCriticalException
	{
		try
		{
			internal.close();
		}
		catch (DatabaseException e)
		{
			m_logger.severe("ERROR: Could not disconnect from the Archive Internal Interface");
		}
	}
	
	/**
	 * Stores a the xml string in the specified location (curl)
	 * @param curl
	 * @param xml
	 */
	private void storeCurl(String curl, String xml)
	{
		try
		{
			/*
			 * This uses a non versioning type of store removing problems with 
			 * updating the entry when it changes. When the CDB manager is used
			 * it will connect to a different interface and know whether an entry
			 * is being created or updated.
			 */
			URI uri = new URI(curl); 
			//internal.store(uri,xml);
		}
		//catch (ArchiveException e)
		//{
		//	m_logger.severe("ERROR: Could not store the curl: " + curl);
		//}
		catch (URISyntaxException e)
		{
			m_logger.severe("ERROR: Illegal curl: " + curl);
		}
	}


	/**
	 * 
	 */
	public void run() throws Exception
	{
		getDAL();
		
		traverseNodes("");
		
		disConnect();
	}


	private void getDAL() throws UnknownHostException
	{
		m_logger.info("trying to get DAL reference...");

		String dalLoc = System.getProperty("ACS.DAL");
			
		if (dalLoc == null)
		{
			System.err.println("Property ACS.DAL not defined. Will try localhost, port 5000...");
			try
			{
				dalLoc = "corbaloc::" + InetAddress.getLocalHost().getHostName() + ":5000/DAL";
			}
			catch (UnknownHostException e)
			{
				System.err.println("Failed to build managerloc for localhost.");
				throw e;
			}
		}
			
		m_dal = DALHelper.narrow(m_orb.string_to_object(dalLoc));
		
		m_logger.info("got DAL reference.");
	}


	private void traverseNodes(String parentNode)
	{
//		m_logger.info("will call DAL#list_nodes for node: " + parentNode);

		String nodes = m_dal.list_nodes(parentNode);
		
		StringTokenizer tok = new StringTokenizer(nodes, " ");
		
		while (tok.hasMoreElements())
		{
			String node = (String) tok.nextElement();
			// as long as we get all files back...
			if (node.indexOf(".xml") < 0)
			{
				String nodePath = parentNode + "/" + node;
				getDalXml(nodePath);			
				traverseNodes(nodePath);
			}
		}
	}


	/**
	 * Gets the schema-expanded XML for the specified node.
	 * 
	 * @param nodePath
	 */
	private void getDalXml(String nodePath)
	{
		m_logger.fine("will get DAL info for node: " + nodePath);
		
		try
		{
			String xml = m_dal.get_DAO(nodePath);
			
			m_logger.info(xml);
			
			storeXml(nodePath, xml);
			
		}
		catch (Exception e)
		{
			m_logger.fine("failed, hopefully because it was an illegal node.");
		}
		finally
		{
			m_logger.fine("--------------------");
		}
	}


	/**
	 * Stores a CDB xml to the operational archive
	 * @param nodePath
	 * @param xml
	 */
	private void storeXml(String nodePath, String xml)
	{
		// TODO store xml in operational archive
//		m_opArchive.???
	}




	public static void main(String[] args)
	{
		AdvancedComponentClient compClient = null;
		try
		{
			String managerLoc = System.getProperty("ACS.manager");
			
			if (managerLoc == null)
			{
				System.err.println("Property ACS.manager not defined. Will try localhost...");
				managerLoc = AcsManagerProxy.getLocalManagerCorbaloc();
			}
			
			compClient = new AdvancedComponentClient(null, managerLoc, "CDBDatabaseLoader");
			System.out.println("componentClient object created.");
			
			ContainerServices services = compClient.getContainerServices();
			ORB orb = compClient.getAcsCorba().getORB();
			
			Loader loader = new Loader(services, orb);
			
			loader.run();
			
		}
		catch (Exception e)
		{
			System.err.println("exception in main method of " + Loader.class.getName() );
			e.printStackTrace(System.err);
		}
		finally
		{
			if (compClient != null)
			{
				try
				{
					compClient.tearDown();
				}
				catch (Exception e)
				{
					e.printStackTrace(System.err);
				}
			}
		}
	}

}
