/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 */

package alma.archive.components;

import java.io.IOException;
import java.io.StringReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.ListIterator;
import java.util.Vector;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.jdom.Document;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;

import alma.ACS.OffShootOperations;
import alma.JavaContainerError.wrappers.AcsJContainerServicesEx;
import alma.acs.container.ContainerServices;
import alma.acs.exceptions.AcsJException;
import alma.acs.nc.SimpleSupplier;
import alma.archive.database.helpers.DBConfiguration;
import alma.archive.database.interfaces.DBCursor;
import alma.archive.database.interfaces.InternalIF;
import alma.archive.database.interfaces.InternalIFFactory;
import alma.archive.database.interfaces.SchemaManager;
import alma.archive.database.oracle.DatabaseConnectionPool;
import alma.archive.exceptions.ArchiveException;
import alma.archive.exceptions.ModuleCriticalException;
import alma.archive.exceptions.access.EntityDirtyException;
import alma.archive.exceptions.access.PermissionDeniedException;
import alma.archive.exceptions.general.ArchiveCommunicationException;
import alma.archive.exceptions.general.DatabaseException;
import alma.archive.exceptions.general.EntityDoesNotExistException;
import alma.archive.exceptions.general.UnknownSchemaException;
import alma.archive.exceptions.syntax.MalformedURIException;
import alma.archive.exceptions.syntax.MalformedXMLException;
import alma.archive.exceptions.user.UserDoesNotExistException;
import alma.archive.helpers.InternalCommunicationHelper;
import alma.archive.wrappers.ArchiveTimeStamp;
import alma.archive.wrappers.DocumentData;
import alma.archive.wrappers.Permissions;
import alma.archive.wrappers.ResultStruct;
import alma.xmlentity.XmlEntityStruct;
import alma.xmlstore.ArchiveInternalError;
import alma.xmlstore.Cursor;
import alma.xmlstore.CursorHelper;
import alma.xmlstore.OperationalOperations;
import alma.xmlstore.XmlStoreNotificationEvent;
import alma.xmlstore.OperationalPackage.DirtyEntity;
import alma.xmlstore.OperationalPackage.IllegalEntity;
import alma.xmlstore.OperationalPackage.MalformedURI;
import alma.xmlstore.OperationalPackage.NotFound;
import alma.xmlstore.OperationalPackage.NotYetThere;
import alma.xmlstore.OperationalPackage.StatusStruct;

/**
 * @author simon
 */
public class OperationalImpl implements OffShootOperations,
		OperationalOperations {
	private String user;

	private ContainerServices containerServices;

	private Logger logger;

	private InternalIF internal = null;

	private SchemaManager smanager = null;

	/** name of this component as appearing the CDB */
	protected String myName;

	protected SimpleSupplier xmlNotifyChannel;

	// for Archive-only methods, a password must be provided (Simple protection)
	private String m_password = "almabtrieb";

	/**
	 * Used to group the connection to the internal interface in one place.
	 */
	private void connect() {
		if (internal == null) {
			try {
				internal = InternalIFFactory.getInternalIF(logger);
				logger.log(Level.INFO,
						"ARCHIVE: Connected to the Internal Interface");

				smanager = internal.getSchemaManager(user);
				logger.log(Level.INFO,
						"ARCHIVE: Connected to the Schema Manager");

				// listenerManager =
				// ListenerManager.instance(containerServices);
				// logger.log(Level.INFO,"Connected to the Liatener Manager");
				// pathmanager = PathManager.instance(logger);
			} catch (DatabaseException e) {
				logger.log(Level.SEVERE, e.getMessage());
			} catch (PermissionDeniedException e) {
				logger.log(Level.SEVERE, e.getMessage());
			} catch (UserDoesNotExistException e) {
				logger.log(Level.SEVERE, e.getMessage());
			} catch (ArchiveException e) {
				logger.log(Level.SEVERE, e.getMessage());
			} catch (ModuleCriticalException e) {
				logger.log(Level.SEVERE, e.getCause().getMessage());
				try {
					InternalCommunicationHelper.notifyMaster(e,
							containerServices, myName, logger);
				} catch (ArchiveCommunicationException e1) {
					// ignore
					logger
							.warning("ARCHIVE: Cannot connect to Archive subsystem master");
				}
			}
		}
	}

	/**
	 * Build the Operational Interface and call the connect method.
	 * 
	 * @param user
	 *            username
	 * @param logger
	 *            a logger from the container
	 * @param containerServices
	 *            a link to the container services
	 */
	public OperationalImpl(String user, Logger logger,
			ContainerServices containerServices) {
		myName = containerServices.getName();
		this.user = user;
		this.logger = logger;
		this.containerServices = containerServices;
		connect();

		if (xmlNotifyChannel == null) {
			try {
				xmlNotifyChannel = new SimpleSupplier(
						alma.xmlstore.CHANNELNAME.value, containerServices);
			} catch (AcsJException e) {
				logger
						.severe("ARCHIVE: Could not initialize Notification channel "
								+ alma.xmlstore.CHANNELNAME.value);
			}
		}
	}

	/**
	 * Closes the interface and disconnects from the internal interface.
	 */
	public void close(String password) throws ArchiveInternalError {
		if (!password.equals(m_password)) {
			throw new ArchiveInternalError(
					"Uncorrect password provided to call this method.");
		}
		if (xmlNotifyChannel != null) {
			xmlNotifyChannel.disconnect();
			xmlNotifyChannel = null;
		}

		if (internal != null) {
			try {
				smanager.close();
				internal.close();
				logger.log(Level.INFO,
						"ARCHIVE: Disconnected from the internal interface");
			} catch (DatabaseException e) {
				logger
						.log(Level.INFO,
								"ARCHIVE: Unable to disconnect from the internal interface");
				throw new ArchiveInternalError(e.toString());
			} catch (ModuleCriticalException e) {
				try {
					InternalCommunicationHelper.notifyMaster(e,
							containerServices, myName, logger);
				} catch (ArchiveCommunicationException e1) {
					// ignore
					logger
							.warning("ARCHIVE: Cannot connect to Archive subsystem master");
				}
				logger
						.log(Level.INFO,
								"ARCHIVE: Unable to disconnect from the internal interface");
				throw new ArchiveInternalError(e.toString());
			}
		}
	}

	/**
	 * Checks out the Entity Struct and throws exceptions as necessary, uses a
	 * very simeple way of checking that the XML is valid amy need to be made a
	 * little faster.
	 * 
	 * @param entity
	 * @throws IllegalEntity
	 */
	private void checkEntityValid(XmlEntityStruct entity) throws IllegalEntity {
		if ((entity.entityId == null) || (entity.entityId == "")) {
			logger.severe("ARCHIVE: Entity did not have a valid ID");
			throw new IllegalEntity("Entity ID must have a value");
		}
		SAXBuilder builder = new SAXBuilder();
		builder.setIgnoringElementContentWhitespace(true);
		try {
			// TODO: Use sax to parse this against a schema
			Document doc = builder.build(new StringReader(entity.xmlString));
		} catch (JDOMException e) {
			logger.severe("ARCHIVE: Entity URI: " + entity.entityId
					+ " Does not contain vaild XML");
			throw new IllegalEntity(e.getMessage());
		} catch (IOException e) {
			logger.severe("ARCHIVE: Entity URI: " + entity.entityId
					+ " Does not contain vaild XML");
			throw new IllegalEntity(e.getMessage());
		}
	}

	/**
	 * Stores a new XmlEntityStruct in the Archive, will attempt to check the
	 * validity of the struct before storing it.
	 */
	public void store(XmlEntityStruct entity) throws IllegalEntity,
			ArchiveInternalError {
		logger.log(Level.FINEST, "ARCHIVE: Storing URI: " + entity.entityId
				+ " Type: " + entity.entityTypeName);

		checkEntityValid(entity);

		try {
			// pathmanager.addPath(uri);
			URI uri = new URI(entity.entityId);
			Permissions permissions = new Permissions();
			URI schema = smanager.getSchemaURI(entity.entityTypeName);

			logger.log(Level.FINEST, "ARCHIVE: Schema Location for Type "
					+ entity.entityTypeName + " is " + schema.toASCIIString());

			internal.store(uri, entity.xmlString, schema,
					entity.entityTypeName, user, // the owner is the user
					permissions, user, true);

			logger.log(Level.FINEST, "ARCHIVE: Stored: " + entity.entityId
					+ " successfully");
			XmlStoreNotificationEvent successEvent = new XmlStoreNotificationEvent(
					uri.toString(), alma.xmlstore.operationType.STORED_XML);

			try {
				xmlNotifyChannel.publishEvent(successEvent);
			} catch (AcsJException e1) {
				logger.warning("ARCHIVE: Could not publish on NC "
						+ alma.xmlstore.CHANNELNAME.value + ".");
			}

		} catch (URISyntaxException e) {
			throw new IllegalEntity(e.getMessage());
		} catch (UnknownSchemaException e) {
			throw new IllegalEntity(e.getMessage());
		} catch (ArchiveException e) {
			logger.severe("Could not store entity " + entity.entityId);
			throw new ArchiveInternalError(e.getMessage());
		} catch (ModuleCriticalException e) {
			try {
				logger.severe("Could not store entity " + entity.entityId);
				InternalCommunicationHelper.notifyMaster(e, containerServices,
						myName, logger);
			} catch (ArchiveCommunicationException e1) {
				// ignore
				logger
						.warning("ARCHIVE: Cannot connect to Archive subsystem master");
			}
			throw new ArchiveInternalError(e.getCause().toString());
		}
		// listenerManager.updatedEntity(entity.entityId);
	}

	/**
	 * Updates an existing entity, the timestamp for the entity must be the same
	 * as the most recent or the update will cause an exception.
	 */
	public void update(XmlEntityStruct entity) throws IllegalEntity,
			ArchiveInternalError {
		update(entity, false);
	}

	/**
	 * Forces the update regardless of the timestamp
	 */
	public void forceUpdate(XmlEntityStruct entity) throws IllegalEntity,
			ArchiveInternalError {
		update(entity, true);
	}

	private void update(XmlEntityStruct entity, boolean force)
			throws IllegalEntity, ArchiveInternalError {
		checkEntityValid(entity);

		logger.log(Level.FINEST, "ARCHIVE: Updating URI: " + entity.entityId
				+ " Type: " + entity.entityTypeName + " Schema Version: "
				+ entity.schemaVersion + " TimeStamp: " + entity.timeStamp);

		try {
			URI uri = new URI(entity.entityId);
			URI schema = smanager.getSchemaURI(entity.entityTypeName);
			ArchiveTimeStamp timeStamp = new ArchiveTimeStamp(entity.timeStamp);

			logger.log(Level.FINEST, "ARCHIVE: Schema Location for Type "
					+ entity.entityTypeName + " is " + schema.toASCIIString());

			internal.clean(uri, user);
			internal.update(uri, timeStamp, entity.xmlString, schema, force,
					user);

			logger.log(Level.FINEST, "ARCHIVE: Updated: " + entity.entityId
					+ " successfully");

			XmlStoreNotificationEvent successEvent = new XmlStoreNotificationEvent(
					uri.toString(), alma.xmlstore.operationType.UPDATED_XML);

			try {
				xmlNotifyChannel.publishEvent(successEvent);
			} catch (AcsJException e1) {
				logger.warning("ARCHIVE: Could not publish on NC "
						+ alma.xmlstore.CHANNELNAME.value + ".");
			}

		} catch (URISyntaxException e) {
			throw new IllegalEntity(e.getMessage());
		} catch (UnknownSchemaException e) {
			throw new IllegalEntity(e.getMessage());
		} catch (ArchiveException e) {
			throw new ArchiveInternalError(e.toString());
		} catch (ModuleCriticalException e) {
			try {
				InternalCommunicationHelper.notifyMaster(e, containerServices,
						myName, logger);
			} catch (ArchiveCommunicationException e1) {
				// ignore
				logger
						.warning("ARCHIVE: Cannot connect to Archive subsystem master");
			}
			throw new ArchiveInternalError(e.getCause().toString());
		}
		// listenerManager.updatedEntity(entity.entityId);
	}

	/**
	 * performd an incremental update of the already stored XML document uid
	 * (belonging to schema schmea): The XML string newChild is appended as the
	 * last child of the root element.
	 * 
	 * @param uid
	 * @param schema
	 * @param newChild
	 * @throws ArchiveInternalError
	 * @throws IllegalEntity
	 *             in case of schema problems
	 * @throws NotYetThere
	 *             if entity UID was not yet stored in Archive
	 * @throws MalformedURI
	 *             if UID is not wellformed
	 */
	public void updateXML(String uid, String schema, String newChild)
			throws ArchiveInternalError, IllegalEntity, NotYetThere,
			MalformedURI {
		try {
			internal.updateXML(new URI(uid), schema, newChild);
		} catch (DatabaseException e) {
			throw new ArchiveInternalError(e.toString());
		} catch (MalformedXMLException e) {
			throw new IllegalEntity(e.toString());
		} catch (ModuleCriticalException e) {
			try {
				InternalCommunicationHelper.notifyMaster(e, containerServices,
						myName, logger);
			} catch (ArchiveCommunicationException e1) {
				// ignore
				logger
						.warning("ARCHIVE: Cannot connect to Archive subsystem master");
			}
			throw new ArchiveInternalError(e.getCause().toString());
		} catch (EntityDoesNotExistException e) {
			throw new NotYetThere(uid);
		} catch (URISyntaxException e) {
			throw new MalformedURI(uid);
		}
	}

	private XmlEntityStruct retrieve(String uid, boolean update)
			throws MalformedURI, ArchiveInternalError, NotFound, DirtyEntity {
		logger.log(Level.FINEST, "ARCHIVE: Retrieve URI: " + uid);
		try {
			URI uri = new URI(uid);
			DocumentData dd = internal.status(uri, user);

			XmlEntityStruct result = new XmlEntityStruct();

			result.xmlString = internal.get(uri, user);
			result.entityId = uid;
			result.entityTypeName = smanager.getSchemaName(dd.getSchema());
			result.schemaVersion = Integer.toString(smanager
					.getSchemaVersion(dd.getSchema()));
			result.timeStamp = dd.getTimestamp().toISOString();

			if (update) {
				internal.dirty(uri, user);
			}

			return result;
		} catch (EntityDirtyException e) {
			throw new DirtyEntity(e.getMessage());
		} catch (URISyntaxException e) {
			throw new MalformedURI(e.getMessage());
		} catch (EntityDoesNotExistException e) {
			throw new NotFound("Document " + uid + " not found in Archive.");
		} catch (MalformedURIException e) {
			throw new MalformedURI(e.getMessage());
		} catch (ArchiveException e) {
			throw new ArchiveInternalError(e.toString());
		} catch (ModuleCriticalException e) {
			try {
				InternalCommunicationHelper.notifyMaster(e, containerServices,
						myName, logger);
			} catch (ArchiveCommunicationException e1) {
				// ignore
				logger
						.warning("ARCHIVE: Cannot connect to Archive subsystem master");
			}
			throw new ArchiveInternalError(e.getCause().toString());
		}
	}

	/**
	 * Retrieve the Entity given by the uri.
	 */
	public XmlEntityStruct retrieve(String uri) throws MalformedURI,
			ArchiveInternalError, NotFound, DirtyEntity {
		return retrieve(uri, false);
	}

	/**
	 * @see alma.xmlstore.OperationalOperations#exists(java.lang.String)
	 */
	public boolean exists(String uid) throws DirtyEntity, MalformedURI,
			NotFound, ArchiveInternalError {
		try {
			retrieve(uid, false);
		} catch (DirtyEntity e) {
			// the entity exists but is dirty, therefore we return true
			return true;
		} catch (NotFound e) {
			return false;
		}
		// if we get to this point, no exception is thrown, ie the entity exists
		return true;
	}

	/**
	 * Retrieve the entity given by the uri and mark it as dirty.
	 */
	public XmlEntityStruct updateRetrieve(String uri) throws MalformedURI,
			ArchiveInternalError, NotFound, DirtyEntity {
		return retrieve(uri, true);
	}

	/**
	 * Retrieve the entity given by the uri, ignore any dirty flag.
	 */
	public XmlEntityStruct retrieveDirty(String uid) throws MalformedURI,
			ArchiveInternalError, NotFound {
		logger.log(Level.FINEST, "ARCHIVE: Retrieve URI: " + uid);
		try {
			URI uri = new URI(uid);
			DocumentData dd = internal.status(uri, user);

			XmlEntityStruct result = new XmlEntityStruct();

			result.xmlString = internal.get(uri, user);
			result.entityId = uid;
			result.entityTypeName = smanager.getSchemaName(dd.getSchema());
			result.schemaVersion = Integer.toString(smanager
					.getSchemaVersion(dd.getSchema()));
			result.timeStamp = dd.getTimestamp().toISOString();

			return result;
		} catch (URISyntaxException e) {
			throw new MalformedURI(e.getMessage());
		} catch (EntityDoesNotExistException e) {
			throw new NotFound(e.getMessage());
		} catch (MalformedURIException e) {
			throw new MalformedURI(e.getMessage());
		} catch (ArchiveException e) {
			throw new ArchiveInternalError(e.toString());
		} catch (ModuleCriticalException e) {
			try {
				InternalCommunicationHelper.notifyMaster(e, containerServices,
						myName, logger);
			} catch (ArchiveCommunicationException e1) {
				// ignore
				logger
						.warning("ARCHIVE: Cannot connect to Archive subsystem master");
			}
			throw new ArchiveInternalError(e.getCause().toString());
		}
	}

	/**
	 * Retrieve a frangment of the entity given by the uri, the id selects the
	 * fragment
	 */
	public String[] retrieveFragment(String uri, String id)
			throws MalformedURI, ArchiveInternalError, NotFound, DirtyEntity {
		logger.log(Level.FINEST, "ARCHIVE: Getting URI: " + uri);
		try {
			// the normal XPath id mechanism does not work for eXist (presumably
			// because the schemas are not attached during storage)
			// String xpath = "id(" + id + ")";
			String xpath = "//*[@id=\"" + id + "\"]";
			HashMap namespaces = smanager.getSchemaNamespaces(new URI(uri));
			return internal.get(new URI(uri), xpath, namespaces, user);
		} catch (EntityDirtyException e) {
			throw new DirtyEntity(e.getMessage());
		} catch (URISyntaxException e) {
			throw new MalformedURI(e.getMessage());
		} catch (EntityDoesNotExistException e) {
			throw new NotFound(e.getMessage());
		} catch (MalformedURIException e) {
			throw new MalformedURI(e.getMessage());
		} catch (ArchiveException e) {
			throw new ArchiveInternalError(e.toString());
		} catch (ModuleCriticalException e) {
			try {
				InternalCommunicationHelper.notifyMaster(e, containerServices,
						myName, logger);
			} catch (ArchiveCommunicationException e1) {
				// ignore
				logger
						.warning("ARCHIVE: Cannot connect to Archive subsystem master");
			}
			throw new ArchiveInternalError(e.getCause().toString());
		}
	}

	/**
	 * @see alma.xmlstore.OperationalOperations#delete(java.lang.String)
	 * 
	 * Marks a document as deleted, as such it will no longer show up in
	 * searches
	 */
	public void delete(String uri) throws MalformedURI, NotFound,
			ArchiveInternalError {
		logger.log(Level.FINEST, "ARCHIVE: Deleting URI: " + uri);
		try {
			internal.delete(new URI(uri), user);
			// listenerManager.updatedEntity(uri);

			XmlStoreNotificationEvent successEvent = new XmlStoreNotificationEvent(
					uri, alma.xmlstore.operationType.DELETED_XML);

			try {
				xmlNotifyChannel.publishEvent(successEvent);
			} catch (AcsJException e1) {
				logger.warning("ARCHIVE: Could not publish on NC "
						+ alma.xmlstore.CHANNELNAME.value + ".");
			}

		} catch (URISyntaxException e) {
			throw new MalformedURI(e.getMessage());
		} catch (EntityDoesNotExistException e) {
			throw new NotFound(e.getMessage());
		} catch (ArchiveException e) {
			throw new ArchiveInternalError(e.toString());
		} catch (ModuleCriticalException e) {
			try {
				InternalCommunicationHelper.notifyMaster(e, containerServices,
						myName, logger);
			} catch (ArchiveCommunicationException e1) {
				// ignore
				logger
						.warning("ARCHIVE: Cannot connect to Archive subsystem master");
			}
			throw new ArchiveInternalError(e.getCause().toString());
		}
	}

	/**
	 * @see alma.xmlstore.OperationalOperations#undelete(java.lang.String)
	 * 
	 * Removes a deleted flag from a document.
	 */
	public void undelete(String uri) throws ArchiveInternalError, NotFound,
			MalformedURI {
		logger.log(Level.FINEST, "ARCHIVE: Un-deleting URI: " + uri);

		try {
			internal.undelete(new URI(uri), user);
			// listenerManager.updatedEntity(uri);
		} catch (URISyntaxException e) {
			throw new MalformedURI(e.getMessage());
		} catch (ArchiveException e) {
			throw new ArchiveInternalError(e.toString());
		}
	}

	/**
	 * Return a status struct for a particular entity
	 */
	public StatusStruct status(String uri) throws MalformedURI, NotFound,
			ArchiveInternalError {
		logger.log(Level.FINEST, "ARCHIVE: Status URI: " + uri);
		try {
			DocumentData dd = internal.status(new URI(uri), user);

			StatusStruct result = new StatusStruct();
			result.deleted = dd.getDeleted();
			result.dirty = dd.getDirty();
			result.locks = dd.getAdmin();
			result.owner = dd.getOwner();
			result.schema = dd.getSchema().toASCIIString();
			result.hidden = dd.getHidden();
			return result;
		} catch (URISyntaxException e) {
			throw new MalformedURI(e.getMessage());
		} catch (ArchiveException e) {
			throw new ArchiveInternalError(e.toString());
		} catch (ModuleCriticalException e) {
			try {
				InternalCommunicationHelper.notifyMaster(e, containerServices,
						myName, logger);
			} catch (ArchiveCommunicationException e1) {
				// ignore
				logger
						.warning("ARCHIVE: Cannot connect to Archive subsystem master");
			}
			throw new ArchiveInternalError(e.getCause().toString());
		}
	}

	/**
	 * @see alma.xmlstore.OperationalOperations#queryRecent(java.lang.String,
	 *      java.lang.String)
	 */
	public String[] queryRecent(String schema, String timestamp)
			throws ArchiveInternalError {
		logger.log(Level.FINE,
				"ARCHIVE: Querying recent documents from schema " + schema
						+ " later than " + timestamp);
		try {
			URI[] result = null;
			try {
				result = internal.queryRecent(new ArchiveTimeStamp(timestamp),
						schema, user);
			} catch (NullPointerException e) {
				throw new ArchiveInternalError("Unknown schema: " + schema);
			}
			if (result == null) {
				throw new ArchiveInternalError("Unknown schema: " + schema);
			}
			String[] out = new String[result.length];
			for (int i = 0; i < result.length; i++) {
				out[i] = result[i].toString();
			}
			return out;
		} catch (ArchiveException e) {
			throw new ArchiveInternalError(e.toString());
		} catch (ModuleCriticalException e) {
			try {
				InternalCommunicationHelper.notifyMaster(e, containerServices,
						myName, logger);
			} catch (ArchiveCommunicationException e1) {
				// ignore
				logger
						.warning("ARCHIVE: Cannot connect to Archive subsystem master");
			}
			throw new ArchiveInternalError(e.getCause().toString());
		}
	}

	/**
	 * @see alma.xmlstore.OperationalOperations#query(java.lang.String,java.lang.String)
	 *      Query a particular set of documents with an XPath, only applies to
	 *      the the given schema type. Only the latest versions of documents are
	 *      queried. Also and documents marked as dirty or deleted will be
	 *      excluded. Works not correctly for queries returning content (and not
	 *      XML), eg.: a/@b. In this case, use queryContent().
	 */
	public Cursor query(String query, String schema)
			throws ArchiveInternalError {
		logger.log(Level.FINE, "ARCHIVE: Querying XPath: " + query
				+ " Schema: " + schema);
		try {
			URI schemaURI = smanager.getSchemaURI(schema);
			HashMap namespaces = smanager.getSchemaNamespaces(schemaURI);

			logger.log(Level.FINEST, "ARCHIVE: Schema location for " + schema
					+ " is " + schemaURI.toASCIIString());

			DBCursor internalCursor = internal.query(query, schema, namespaces,
					false, user);
			CursorImpl externalCursor = new CursorImpl(internalCursor,
					containerServices);

			logger.log(Level.FINE, "ARCHIVE: Returning cursor to user");

			return CursorHelper.narrow(containerServices
					.activateOffShoot(externalCursor));
		} catch (ArchiveException e) {
			// TODO: revisit log level, and use ErrorTrace-aware exception
			logger.log(Level.INFO, "Query failed.", e);
			throw new ArchiveInternalError(e.toString());
		} catch (AcsJContainerServicesEx e) {
			// TODO: revisit log level, and use ErrorTrace-aware exception
			logger.log(Level.INFO, "Query failed.", e);
			throw new ArchiveInternalError(e.getContextInfo());
		} catch (ModuleCriticalException e) {
			try {
				InternalCommunicationHelper.notifyMaster(e, containerServices,
						myName, logger);
			} catch (ArchiveCommunicationException e1) {
				// ignore
				logger
						.warning("ARCHIVE: Cannot connect to Archive subsystem master");
			}
			throw new ArchiveInternalError(e.getCause().toString());
		}
	}

	/**
	 * A query method that handles content queries correctly. Do only use for
	 * content queries, eg. a/b/text()
	 * 
	 * @param query
	 * @param schema
	 * @return
	 * @throws ArchiveInternalError
	 */
	public Cursor queryContent(String query, String schema)
			throws ArchiveInternalError {
		logger.log(Level.FINE, "ARCHIVE: Querying XPath: " + query
				+ " Schema: " + schema);
		try {
			URI schemaURI = smanager.getSchemaURI(schema);
			HashMap namespaces = smanager.getSchemaNamespaces(schemaURI);

			logger.log(Level.FINEST, "ARCHIVE: Schema location for " + schema
					+ " is " + schemaURI.toASCIIString());

			DBCursor internalCursor = internal.queryContent(query, schema,
					namespaces, false, user);
			CursorImpl externalCursor = new CursorImpl(internalCursor,
					containerServices);

			logger.log(Level.FINE, "ARCHIVE: Returning cursor to user");

			return CursorHelper.narrow(containerServices
					.activateOffShoot(externalCursor));
		} catch (ArchiveException e) {
			// TODO: revisit log level, and use ErrorTrace-aware exception
			logger.log(Level.INFO, "Query failed.", e);
			throw new ArchiveInternalError(e.toString());
		} catch (AcsJContainerServicesEx e) {
			// TODO: revisit log level, and use ErrorTrace-aware exception
			logger.log(Level.INFO, "Query failed.", e);
			throw new ArchiveInternalError(e.getContextInfo());
		} catch (ModuleCriticalException e) {
			try {
				InternalCommunicationHelper.notifyMaster(e, containerServices,
						myName, logger);
			} catch (ArchiveCommunicationException e1) {
				// ignore
				logger
						.warning("ARCHIVE: Cannot connect to Archive subsystem master");
			}
			throw new ArchiveInternalError(e.getCause().toString());
		}
	}

	/**
	 * The same as query except that documents marked as Dirty will be returned.
	 */
	public Cursor queryDirty(String query, String schema)
			throws ArchiveInternalError {
		logger.log(Level.FINE, "ARCHIVE: Querying XPath: " + query
				+ " Schema: " + schema);
		try {
			URI schemaURI = smanager.getSchemaURI(schema);
			HashMap namespaces = smanager.getSchemaNamespaces(schemaURI);

			logger.log(Level.FINEST, "ARCHIVE: Schema location for " + schema
					+ " is " + schemaURI.toASCIIString());

			DBCursor internalCursor = internal.query(query, schema, namespaces,
					true, user);
			CursorImpl externalCursor = new CursorImpl(internalCursor,
					containerServices);

			logger.log(Level.FINE, "ARCHIVE: Returning cursor to user");

			return CursorHelper.narrow(containerServices
					.activateOffShoot(externalCursor));
		} catch (ArchiveException e) {
			// TODO: revisit log level, and use ErrorTrace-aware exception
			logger.log(Level.INFO, "Query failed.", e);
			throw new ArchiveInternalError(e.toString());
		} catch (AcsJContainerServicesEx e) {
			// TODO: revisit log level, and use ErrorTrace-aware exception
			logger.log(Level.INFO, "Query failed.", e);
			throw new ArchiveInternalError(e.getContextInfo());
		} catch (ModuleCriticalException e) {
			try {
				InternalCommunicationHelper.notifyMaster(e, containerServices,
						myName, logger);
			} catch (ArchiveCommunicationException e1) {
				// ignore
				logger
						.warning("ARCHIVE: Cannot connect to Archive subsystem master");
			}
			throw new ArchiveInternalError(e.getCause().toString());
		}
	}

	/**
	 * Returns an array if UIDs that match the given XPath query, multiple
	 * occurences are removed.
	 */
	public String[] queryUIDs(String query, String schema)
			throws ArchiveInternalError {
		logger.log(Level.FINE, "ARCHIVE: Querying: " + query);

		try {
			URI schemaURI = smanager.getSchemaURI(schema);
			HashMap namespaces = smanager.getSchemaNamespaces(schemaURI);
			DBCursor internalCursor = internal.query(query, schema, namespaces,
					false, user);

			Vector<String> vector = new Vector<String>();
			while (internalCursor.hasNext()) {
				ResultStruct res = internalCursor.next();
				vector.add(res.getUri().toASCIIString());
			}
			ListIterator iter = vector.listIterator();
			String[] result = new String[vector.size()];
			while (iter.hasNext()) {
				result[iter.nextIndex()] = (String) iter.next();
			}
			return result;
		} catch (ArchiveException e) {
			throw new ArchiveInternalError(e.toString());
		} catch (ModuleCriticalException e) {
			try {
				InternalCommunicationHelper.notifyMaster(e, containerServices,
						myName, logger);
			} catch (ArchiveCommunicationException e1) {
				// ignore
				logger
						.warning("ARCHIVE: Cannot connect to Archive subsystem master");
			}
			throw new ArchiveInternalError(e.getCause().toString());
		}
	}

	/**
	 * The same as above except that dirty entities are returned
	 */
	public String[] queryUIDsDirty(String query, String schema)
			throws ArchiveInternalError {
		logger.log(Level.FINE, "ARCHIVE: Querying: " + query);

		try {
			URI schemaURI = smanager.getSchemaURI(schema);
			HashMap namespaces = smanager.getSchemaNamespaces(schemaURI);
			DBCursor internalCursor = internal.query(query, schema, namespaces,
					true, user);

			Vector<String> vector = new Vector<String>();
			while (internalCursor.hasNext()) {
				ResultStruct res = internalCursor.next();
				vector.add(res.getUri().toASCIIString());
			}
			ListIterator iter = vector.listIterator();
			String[] result = new String[vector.size()];
			while (iter.hasNext()) {
				result[iter.nextIndex()] = (String) iter.next();
			}
			return result;
		} catch (ArchiveException e) {
			throw new ArchiveInternalError(e.toString());
		} catch (ModuleCriticalException e) {
			try {
				InternalCommunicationHelper.notifyMaster(e, containerServices,
						myName, logger);
			} catch (ArchiveCommunicationException e1) {
				// ignore
				logger
						.warning("ARCHIVE: Cannot connect to Archive subsystem master");
			}
			throw new ArchiveInternalError(e.getCause().toString());
		}
	}

	/**
	 * Adds xmlElement as last child of the (first) element found by xPath in
	 * document uid (belonging to schema schema)
	 */
	public void addElement(String uid, String schema, String xPath,
			String xmlElement) throws ArchiveInternalError, IllegalEntity,
			MalformedURI, NotYetThere {
		try {
		internal.addElement(new URI(uid), schema, xPath, xmlElement, user);
		} catch (DatabaseException e) {
			throw new ArchiveInternalError(e.toString());	
		} catch (URISyntaxException e) {
				throw new IllegalEntity(uid);
		} catch (EntityDoesNotExistException e) {
			throw new NotYetThere(uid);
		} catch (ModuleCriticalException e) {
			try {
				InternalCommunicationHelper.notifyMaster(e, containerServices,
						myName, logger);
			} catch (ArchiveCommunicationException e1) {
				// ignore
				logger
						.warning("ARCHIVE: Cannot connect to Archive subsystem master");
			}
			throw new ArchiveInternalError(e.getCause().toString());
		} catch (PermissionDeniedException e) {
			throw new ArchiveInternalError("Permission denied to modify document "+uid);
		}
	}

	
	
	/**
	 * The (first) element pointed at by xPath in document uid (belonging to
	 * schema schema) will be replaced by element xmlElement.
	 */
	public void updateElement(String uid, String schema, String xPath,
			String xmlElement) throws ArchiveInternalError, IllegalEntity,
			MalformedURI, NotYetThere {
		try {
			internal.updateElement(new URI(uid), schema, xPath,
					xmlElement, user);
		} catch (DatabaseException e) {
			throw new ArchiveInternalError(e.toString());
		} catch (URISyntaxException e) {
			throw new IllegalEntity(uid);
		} catch (EntityDoesNotExistException e) {
			throw new NotYetThere(uid);
		} catch (ModuleCriticalException e) {
			try {
				InternalCommunicationHelper.notifyMaster(e, containerServices,
						myName, logger);
			} catch (ArchiveCommunicationException e1) {
				// ignore
				logger
						.warning("ARCHIVE: Cannot connect to Archive subsystem master");
			}
			throw new ArchiveInternalError(e.getCause().toString());
		} catch (PermissionDeniedException e) {
			throw new ArchiveInternalError("Permission denied to modify document "+uid);
		}
	}

	/**
	 * The (first) element pointed at by xPath in document uid (belonging to
	 * schema schema) will be deleted.
	 */
	public void deleteElement(String uid, String schema, String xPath)
			throws ArchiveInternalError, IllegalEntity, MalformedURI,
			NotYetThere {
		try {
			internal.deleteElement(new URI(uid), schema, xPath, user);
		} catch (DatabaseException e) {
			throw new ArchiveInternalError(e.toString());
		} catch (URISyntaxException e) {
			throw new IllegalEntity(uid);
		} catch (EntityDoesNotExistException e) {
			throw new NotYetThere(uid);
		} catch (ModuleCriticalException e) {
			try {
				InternalCommunicationHelper.notifyMaster(e, containerServices,
						myName, logger);
			} catch (ArchiveCommunicationException e1) {
				// ignore
				logger
						.warning("ARCHIVE: Cannot connect to Archive subsystem master");
			}
			throw new ArchiveInternalError(e.getCause().toString());
		} catch (PermissionDeniedException e) {
			throw new ArchiveInternalError("Permission denied to modify document "+uid);
		}
	}

	/**
	 * 
	 * @return number of open connection to Oracle
	 */

	public int getConnectionNumber() throws ArchiveInternalError {
		try {
			if (!DBConfiguration.instance(logger).dbBackend
					.equalsIgnoreCase("oracle")) {
				return 0;
			} else {
				return DatabaseConnectionPool.instance(logger)
						.getConnectionNumber();
			}
		} catch (SQLException e) {
			throw new ArchiveInternalError(e.toString());
		} catch (DatabaseException e) {
			throw new ArchiveInternalError(e.toString());
		}
	}

}
