/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 */

package alma.archive.components;

import java.io.StringReader;
import java.io.StringWriter;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.logging.Level;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.exolab.castor.xml.MarshalException;
import org.exolab.castor.xml.ValidationException;

import alma.acs.component.ComponentImplBase;
import alma.acs.component.ComponentLifecycleException;
import alma.acs.container.ContainerServices;
import alma.acs.container.archive.Range;
import alma.archive.database.interfaces.IdentifierManager;
import alma.archive.database.interfaces.InternalIF;
import alma.archive.database.interfaces.InternalIFFactory;
import alma.archive.database.interfaces.SchemaManager;
import alma.archive.exceptions.ArchiveException;
import alma.archive.exceptions.ModuleCriticalException;
import alma.archive.exceptions.general.ArchiveCommunicationException;
import alma.archive.exceptions.general.DatabaseException;
import alma.archive.helpers.InternalCommunicationHelper;
import alma.archive.range.IdentifierRange;
import alma.archive.wrappers.Permissions;
import alma.xmlstore.IdentifierJ;
import alma.xmlstore.IdentifierPackage.NotAvailable;
import alma.xmlstore.IdentifierPackage.NotFound;

/**
 * @author simon
 * 
 * This class allows access to the internal interface's Identifier Manager,
 * returning globally uniqiue identifiers.
 * 
 * It assumes that only one instantiation of this class runs in the system. 
 * Otherwise  the local parts of UIDs will be spoiled, i.e. remain unused,
 * in many cases.)
 * 
 */
public class IdentifierImpl
	extends ComponentImplBase
	implements IdentifierJ {
	
	private IdentifierManager imanager = null;

	// the following expression works for new UIDs only, but is not tested yet!!!
	//Pattern p = Pattern.compile("^[uU][iI][dD]://[xX][0-9a-fA-F]{2,}(/[xX][0-9a-fA-F]+){2}(#\\w{1,}){0,}$");
	//Pattern p = Pattern.compile("^[uU][iI][dD]://[xX][0-9a-fA-F]{2,}(/[xX][0-9a-fA-F]+){2}(#\\w{1,}){0,}$");
	// and now the first part (archiveId) can be any string consisting of characters and numbers:
	private static Pattern p = Pattern.compile("^[uU][iI][dD]://[0-9a-zA-Z]+(/[xX][0-9a-fA-F]+){2}(#\\w{1,}){0,}$");

	/** name of this component as appearing the CDB */
	protected String myName;
	
	/**
	 * Connect to the internal interface.
	 */
	private void connect() throws DatabaseException {
		if (imanager == null) {
			m_logger.log(Level.INFO, "Connecting to the identifier manager");

			try {
				imanager = InternalIFFactory.getIdentifierManager(m_logger);
			} catch (ModuleCriticalException e) {
				m_logger.log(Level.SEVERE,
					"Could not connect to the Identifer manager");
				try {
					InternalCommunicationHelper.notifyMaster(
						e, m_containerServices, myName, m_logger);
				} catch (ArchiveCommunicationException e1) {
					// ignore
					m_logger.warning(
						"Can nont connect to Archive subsystem master");
				}
			}

			m_logger.log(Level.INFO, "Connected to the identifier manager");
		}
	}

	/**
	 * Disconnect from the internal interface. 
	 */
	private void close() {
		if (imanager != null) {
			try {
				imanager.close();
				m_logger.log(
					Level.INFO,
					"Disconnected from the identifier manager");
			} catch (DatabaseException e) {
				m_logger.log(
					Level.INFO,
					"Unable to disconnect from the identifier manager");
			}
		}
	}

	/**
	 * @see alma.xmlstore.IdentifierOperations#getUIDs(short)
	 * 
	 * It uses actGlobal and actLocal, which is a counter for the local part. 
	 * 
	 */
	public String[] getUIDs(short number) 
		throws 
			NotAvailable 
	{
		return syncGetUIDs(number);
	}

	/**
	 * @see alma.xmlstore.IdentifierOperations#checkUIDsyntax(URI)
	 * 
	 * It uses actGlobal and actLocal, which is a counter for the local part. 
	 * 
	 */
	public boolean checkUIDsyntax(String identifier) 
	{
		 Matcher m = p.matcher(identifier);
		 return m.matches();
	}

	/**
	 * For synchronizing the method getUIDs()
	 * 
	 */
	private synchronized String[] syncGetUIDs(short number) 
		throws 
			NotAvailable 
	{
		// Output array:
		String[] out = new String[number];


		for (int count = 0; count < number; count++) {
			out[count]=getIdNamespace();
			}

		return out;
	}

	/**
	 * Return a globally unique identifier, connect to the internal interface
	 * if necessary.
	 */
	public String getIdNamespace() throws NotAvailable {
		try {
			if (imanager == null) {
				m_logger.log(
					Level.INFO,
					"Not connected to the identifier manager, connecting");
				connect();
			}
			return imanager.getIdNamespace().toASCIIString();
		} catch (DatabaseException e) {
			throw new NotAvailable(e.getMessage());
		} catch (ArchiveException e) {
			throw new NotAvailable(e.getMessage());
		} catch (ModuleCriticalException e) {
			try {
				InternalCommunicationHelper.notifyMaster(
					e, m_containerServices, myName, m_logger);
			} catch (ArchiveCommunicationException e1) {
				// ignore
				m_logger.warning("Can nont connect to Archive subsystem master");
			}
			throw new NotAvailable(e.getMessage());
		}
	}
	
	/**
	 * Fetch a Unique IdentifierRange entity from the archvie, to be passed to
	 * the client. 
	 */
	public IdentifierRange getNewRange() throws NotAvailable
	{
		try{
			return imanager.getNewRange();
		}
		catch (ArchiveException e){
			throw new NotAvailable(e.getLocalizedMessage());
		}
		catch (ModuleCriticalException e){
			throw new NotAvailable(e.getLocalizedMessage());
		}
	}

	/**
	 * @see alma.xmlstore.IdentifierJ#getNewRestrictedRange(int)
	 * 
	 * Fetch a Unique IdentierRange that is restricted (number).
	 */
	public IdentifierRange getNewRestrictedRange(int number, String user) throws NotAvailable {
		try
		{
			IdentifierRange r = imanager.getNewRange(number);  
			Range range = new Range(r);
			URI uri = range.rangeId(); 
			
			StringWriter writer = new StringWriter();
			r.setIsLocked(true);
			r.marshal(writer);
			
			InternalIF internal = InternalIFFactory.getInternalIF(m_logger); 
			SchemaManager smanager = internal.getSchemaManager(user);
			
			// TODO this should be defined somewhere else. Just fixing here in order to get things running now.
			String schemaName = "IdentifierRange";
			
			URI schemaUri = smanager.getSchemaURI(schemaName);
			
			internal.store(uri,writer.toString(),schemaUri,schemaName,
				user,new Permissions(),user,true);
			
			r.setIsLocked(false);
			return r;
		}
		catch (Throwable thr){
			throw new NotAvailable(thr.toString());
		}
	}
	
	/**
	 * Try to return an existing range from the archive. 
	 */
	public IdentifierRange getExistingRange(String uri, String user) 
		throws 
			NotFound
	{
		try
		{
			InternalIF internal = InternalIFFactory.getInternalIF(m_logger);
			URI uid = new URI(uri);
			String rangexml = internal.get(uid,user);
			IdentifierRange range = IdentifierRange.unmarshalIdentifierRange(
				new StringReader(rangexml)); 
			return range;
		}
		catch (DatabaseException e){
			throw new NotFound(e.getLocalizedMessage());
		}
		catch (ModuleCriticalException e){
			throw new NotFound(e.getLocalizedMessage());
		}
		catch (URISyntaxException e){
			throw new NotFound(e.getLocalizedMessage());
		}
		catch (ArchiveException e){
			throw new NotFound(e.getLocalizedMessage());
		}
		catch (MarshalException e){
			throw new NotFound(e.getLocalizedMessage());
		}
		catch (ValidationException e){
			throw new NotFound(e.getLocalizedMessage());
		}
	}
	
	/**
	 * Initializes the interface, connects to the internal interface.
	 */
	public void initialize(ContainerServices containerServices)
		throws ComponentLifecycleException {
		super.initialize(containerServices);
		myName = containerServices.getName();
		try {
			connect();
		} catch (DatabaseException e) {
			throw new ComponentLifecycleException(e.getMessage());
		}
	}

	/**
	 * Closes the connection to the internal interface.
	 */
	public void cleanUp() {
		close();
	}

}
