/*
 * 	  Created on 10-May-2003
 * 
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 */
package alma.archive.components;

import java.util.logging.Level;
import java.util.logging.Logger;

import alma.acs.container.ContainerServices;
import alma.archive.database.interfaces.DBCursor;
import alma.archive.exceptions.ArchiveException;
import alma.archive.wrappers.ResultStruct;
import alma.xmlstore.ArchiveInternalError;
import alma.xmlstore.CursorPOA;
import alma.xmlstore.CursorPackage.QueryResult;



/**
 * @author simon
 * 
 * the cursor class is used to access results returned by the internal
 * interface. Care must be taken to close instances of the interface when
 * they are done with.
 */

public class CursorImpl extends CursorPOA 
{
	private final DBCursor cursor;
	private final ContainerServices containerServices;	
	private final Logger logger;	

	/**
	 * Create an instance of the cursor, attatch to the internal interface
	 */
	public CursorImpl(DBCursor cursor, ContainerServices cs)
	{
		if (cursor == null || cs == null) {
			throw new IllegalArgumentException("DBCursor and ContainerServices must not be null.");
		}
		this.cursor = cursor;
		this.containerServices = cs;
		this.logger = containerServices.getLogger();
	}

	/**
	 * Closes the the internal cursor and deactivates this Offshoot Corba object.
	 */
	public void close()
	{
		try
		{
			cursor.close();
		}
		catch (Throwable thr){
			logger.log(Level.WARNING, "Failed to close the query cursor.", thr);
		}
		finally {
			try {
				containerServices.deactivateOffShoot(this);
			} catch (Throwable thr) {
				logger.log(Level.WARNING, "Failed to deactivate the CORBA cursor object.", thr);				
			}			
		}
	}

	
	/**
	 * Returns true if there are more results to be returned.
	 * @see alma.xmlstore.CursorOperations#hasNext()
	 */
	public boolean hasNext() throws ArchiveInternalError
	{
		try
		{
			return cursor.hasNext();
		}
		catch (ArchiveException e){
			throw new ArchiveInternalError();
		}
	}

	/**
	 * Returns the next result if there is one, if not an exception will
	 * be thrown.
	 * @see alma.xmlstore.CursorOperations#next()
	 */
	public QueryResult next() throws ArchiveInternalError
	{
		try
		{
			ResultStruct res = cursor.next();
			QueryResult result = new QueryResult();
			result.xml = res.getXml();
			result.identifier = res.getUri().toASCIIString();
			return result; 
		}
		catch (ArchiveException e){
			throw new ArchiveInternalError(e.getMessage());
		}
	}
	
	/**
	 * Returns a specific sized block of results, if there are not enough
	 * results then the entries will contain null.
	 */
	public QueryResult[] nextBlock(short size) throws ArchiveInternalError
	{
		try
		{
			// Fetch a block from the internal cursor
			ResultStruct[] res = cursor.nextBlock(size);
			QueryResult result[] = new QueryResult[size];
			//Create the result structures
			for (int x = 0; x < res.length; x++)
			{
				if (res[x] != null)
				{
					result[x].xml = res[x].getXml();
					result[x].identifier = res[x].getUri().toASCIIString();
				}
				else result[x] = null;
			}
			return result;
		}
		catch (ArchiveException e) {
			throw new ArchiveInternalError();
		}
	}

}
