/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 *
 */
package alma.archive.idregistry.server.hibernate;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.hibernate.cfg.Configuration;

import alma.archive.idregistry.exceptions.DatabaseException;
import alma.archive.idregistry.exceptions.DocumentDoesNotExistException;
import alma.archive.idregistry.exceptions.PermissionDeniedException;
import alma.archive.idregistry.server.ArchiveIdRegistry;
import alma.archive.idregistry.shared.ArchiveDescriptor;
import alma.userrepository.errors.PermissionException;
import alma.userrepository.errors.UserRepositoryException;
import alma.userrepository.roledirectory.Role;
import alma.userrepository.roledirectory.RoleDirectory;
import alma.userrepository.roledirectory.RoleDirectorySession;
import alma.userrepository.shared.DirectorySessionFactory;

public class HibernateIdRegistry implements ArchiveIdRegistry {

    // ------------------------------------------------------ Instance variables

    /**
     * Logging output for this user database instance
     */
    private Log log = LogFactory.getLog(this.getClass());

    /**
     * Hibernate query to get user with matching id
     */
    static private final String HQL_MATCH_ID = "select d from HibernateDescriptor as d where d.id = :IDARG";

    /**
     * Hibernate query to get user with matching MAC address
     */
    static private final String HQL_MATCH_MAC = "select d from HibernateDescriptor as d where d.macAddress = :MACARG";

    /**
     * Hibernate session factory
     */
    private static SessionFactory sessionFactory = null;

    private static final List<Role> REGISTRY_PRIVILEGED_ROLES;

    static {
        sessionFactory = new Configuration().configure().buildSessionFactory();
        REGISTRY_PRIVILEGED_ROLES = new ArrayList<Role>();
        REGISTRY_PRIVILEGED_ROLES.add(new Role("ADMINISTRATOR", "MASTER"));
    }

    public ArchiveDescriptor getArchiveDescriptor(String archiveId,
            String username, String password) throws PermissionDeniedException,
            UserRepositoryException, DatabaseException,
            DocumentDoesNotExistException {
        checkPermission(username, password);

        Long l = fromHex(archiveId);
        if (log.isDebugEnabled())
            log.debug("Retrieving descriptor with id '" + l + "'");

        Session session = sessionFactory.openSession();
        Transaction tx = session.beginTransaction();

        HibernateDescriptor result = null;
        // uid defined as unique in db schema , so don't strictly need to check
        // for >1 match exception
        try {
            result = (HibernateDescriptor) session.createQuery(HQL_MATCH_ID)
                    .setLong("IDARG", l).uniqueResult();
            tx.commit();
        } finally {
            session.close();
        }

        if (result == null) {
            throw new DocumentDoesNotExistException(
                    "ArchiveDescriptor with id '" + archiveId
                            + "' does not exist");
        } else {
            return result;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * alma.archive.registry.server.ArchiveIdRegistry#getArchiveId(alma.archive
     * .registry.server.ArchiveDescriptor, java.lang.String)
     */
    public String getArchiveId(ArchiveDescriptor archiveDescriptor,
            String username, String password) throws UserRepositoryException,
            DatabaseException, PermissionDeniedException {
        checkPermission(username, password);

        // all further permission checks are performed by db layer

        // user is authenticated and has the required role, so update the
        // ArchiveDescriptor document with the username so we have a record of
        // who committed the document
        HibernateDescriptor hd = new HibernateDescriptor(archiveDescriptor);
        hd.setUsername(username);

        if (log.isTraceEnabled()) {
            log.trace("Saving ArchiveDescriptor: " + hd);
        }

        Session session = sessionFactory.openSession();
        Transaction tx = session.beginTransaction();

        if (log.isTraceEnabled()) {
            log.trace("Checking existing records for MAC address '"
                    + hd.getMacAddress() + "'");
        }
        // MAC address defined as unique in db schema, so don't need to check
        // for >1 match exception
        HibernateDescriptor existing = (HibernateDescriptor) session
                .createQuery(HQL_MATCH_MAC).setString("MACARG",
                        hd.getMacAddress()).uniqueResult();
        tx.commit();

        // so if the machine is already registered, give the existing archive ID
        if (existing != null) {
            session.close();
            return existing.getArchiveId();
        }

        // otherwise this is a new descriptor, so save it to the database.
        tx = session.beginTransaction();
        try {
            // we may receive constraint violation exceptions etc, so clean up
            // in finally
            session.saveOrUpdate(hd);
            tx.commit();
        } catch (HibernateException he) {
            throw new DatabaseException(
                    "Operation generated Hibernate exception: ", he);
        } finally {
            session.close();
        }

        return hd.getArchiveId();
    }

    private void checkPermission(String username, String password)
            throws PermissionDeniedException, UserRepositoryException {
        // does this user have the necessary privileges?
        RoleDirectory userRoles;
        try {
            if (log.isTraceEnabled())
                log.trace("Connecting to role directory with uid=" + username
                        + ".");
            RoleDirectorySession roleSession = DirectorySessionFactory
                    .getInstance(null).newRoleDirectorySession(username, password);
            userRoles = roleSession.getAllUserRoles(username);
        } catch (PermissionException e) {
            throw new PermissionDeniedException(e);
        }

        if (userRoles == null
                || !userRoles.hasAnyRole(REGISTRY_PRIVILEGED_ROLES)) {
            throw new PermissionDeniedException(
                    "User '"
                            + username
                            + "' does not have sufficient privileges to retrieve an archive ID");
        }
    }

    private long fromHex(String s) {
        long l = Long.parseLong(s, 16);
        return l;
    }
}
