/*
 * Copyright (C) 2002 Associated Universities Inc
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Correspondence concerning ALMA Software should be addressed as follows:
 *   Internet email: alma-sw-admin@nrao.edu
 */

/************************************************************************
* "@(#) $Id: rtlogImpl.cpp,v 1.43 2009/10/07 09:30:43 bjeram Exp $"
*
* who       when        what
* --------  ----------  ----------------------------------------------
* ramestic  2003-06-20  created
*/

/************************************************************************
*   NAME
*	rtlogImpl - a real-time logger interface implementation. 
*
*   SYNOPSIS
*
*   PARENT CLASS
*
*   DESCRIPTION
*	based on an RTAI fifo this component implementation receives messages
*	from real-time applications and publish them into the ACS logging system.
*	This module implements two major mechanisms:
*		1. an ACS component which spawns a baci thread that will be
*		   normally blocking on a real-time fifo for messages coming
*		   from real-time applications. Those could be kernel or user space
*		   entities, the usage of a fifo makes that detail transparent
*		   at this level. Every time the thread gets a message it 
*		   extraxts the data and logs it into the ACS logging system.
*		2. a macro available for user space and kernel space real-time
*		   applications that feeds into the rtlog real-time fifo the
*		   relevant source data (file, line num, time) and the application
*		   logging text.
*
*	The RTLOG macro receives as one of its parameters a priority, which can have
*	three possible values that are translated into ACS logging priorities in the
*	following way:
*		RTLOG_NORMAL -> LM_INFO
*		RTLOG_ERROR  -> LM_ERROR
*		RTLOG_DEBUG  -> LM_DEBUG
*
*	If the rtlog component is not running then the real-time fifo is not
*	available, the macro detects that condition and defaults to rt_printk calls.
*	In this the priority is reflected as part of the output string (normal, error
*	and debug). In this case also the macro assumes that in the code scope in 
*	which was deployed there is a debug integer variable, which controls
*	whether debug logs are printed or not via rt_printk (0=no output).
*
*   BUGS
*
*------------------------------------------------------------------------
*/


#include "rtlogImpl.h"
#include <cdbErrType.h>

using namespace std;
using namespace rtlogErrType;
using namespace ACS_RT;

//--------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------
/// Implementation of thread class that send logg message to the logging system
RTLogThread::RTLogThread(const ACE_CString &_name, rtlogImpl &_rtlogImpl) :
    ACS::Thread(_name), mp_rtlogImpl(&_rtlogImpl)
{
    ACS_TRACE("RTLogThread::RTLogThread");
}

void RTLogThread::run()
{
    int recvn;
    
    while (check())
	{
//TBD: add wait for semaphore
	if ( (recvn = rtf_read_timed(mp_rtlogImpl->m_fd,
				     &m_logRecord,
				     sizeof(m_logRecord),
				     1000))
	     == EINVAL )
	    {
	    ACS_LOG(LM_SOURCE_INFO, "RTLogThread::run",
		    (LM_ERROR, "can not read the FIFO"));  //TBD: replace with AES ?
	    return;
	    }
	else if ( recvn == sizeof(m_logRecord) )
	    {

	    m_timeData = mp_rtlogImpl->getTimeData();

/*	    printf("==> %lld, %d, %d, %lld \n",
		   m_timeData.cpu_on_last_te,
		   m_timeData.cpu_hz,
		   m_timeData.ticks,
		   m_timeData.t0
		);
*/
	    if (m_timeData.ticks != 0) 
		{
		m_timeStamp = ((m_logRecord.time - m_timeData.cpu_on_last_te) *
			       10000000ULL /
			       (ACS::Time)m_timeData.cpu_hz +
			       (ACS::Time)m_timeData.ticks *
			       (ACS::Time)480000 +
			       (ACS::Time)m_timeData.t0); 
		}
	    else
		{
		m_timeStamp = ::getTimeStamp();
		}

//	  ACS_CHECK_LOGGER;
	    LoggingProxy::Flags(LM_SOURCE_INFO | LM_RUNTIME_CONTEXT);
	    LOG_RECORD(Logging::ace2acsPriority(ACE_Log_Priority(1 << (m_logRecord.priority -1))),
		       m_logRecord.text,
		       m_logRecord.file, 
		       m_logRecord.line, 
		       getName().c_str(),
		       m_timeStamp,
		       /*mp_rtlogImpl->name()*/ m_logRecord.lkm);
	    }
	else if ( recvn != 0 )
	    {
	    ACS_LOG(LM_SOURCE_INFO, "RTLogThread::run",
		    (LM_ERROR, "received incomplete record: recvn = %d rec. size = %d", 
		     recvn, sizeof(m_logRecord)));
	    }//if-else
	}//while
}//run



//--------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------
/// implementation of rtlog component
rtlogImpl::rtlogImpl( const ACE_CString &name, maci::ContainerServices* containerServices) 
    : CharacteristicComponentImpl(name, containerServices),
      m_FIFOSize(RTLOG_FIFO_SIZE),
      mp_rtLogThread(0),
      mp_timeDataThread(0)
{
    ACS_TRACE("::LoggerImpl::LoggerImpl");
}//rtlogImpl

//--------------------------------------------------------------------------------------

rtlogImpl::~rtlogImpl()
{
    ACS_TRACE("::LoggerImpl::~LoggerImpl");
}//~rtlogImpl

void rtlogImpl::initialize() 
{  
    ACE_Time_Value unixTime;
    CDB::DAL_var cdb;
    CDB::DAO_var dao;
    ACS_RT::RTLogLevels initLogLevel = ACS_RT::RTLogInfo; /*=4*/

    ACS_TRACE("::LoggerImpl::initialize");
  

    ACE_hrtime_t nstime= ACE_OS::gethrtime();
    unixTime = ACE_OS::gettimeofday();
    m_timeOffset = (ACE_UINT64_LITERAL(0x2D8539C80) + ACE_static_cast(CORBA::ULongLong, unixTime.sec())) *ACE_static_cast(ACE_UINT32, 10000000) + ACE_static_cast(CORBA::ULongLong, unixTime.usec() * 10);
    
    m_timeOffset -= (nstime/100);  // time at computer startup

    //try to get data from CDB: FIFO sieze and init log level
    try
	{
	ACE_CString fullCDBPath = "alma/";
	cdb = getContainerServices()->getCDB();  
	fullCDBPath += name();
	dao = cdb->get_DAO_Servant(fullCDBPath.c_str());
	}
    catch(cdbErrType::CDBXMLErrorEx &_ex)
	{
	GettingConfigurationDataProblemExImpl ex(_ex, __FILE__, __LINE__, "LoggerImpl::initialize");
	ex.log(LM_DEBUG);
	}
    catch(cdbErrType::CDBRecordDoesNotExistEx &_ex)
	{
	GettingConfigurationDataProblemExImpl ex(_ex, __FILE__, __LINE__, "LoggerImpl::initialize");
	ex.log(LM_DEBUG);
	}
    catch(CORBA::SystemException &_ex)
	{
	ACSErrTypeCommon::CORBAProblemExImpl corbaProblemEx(__FILE__, __LINE__, "LoggerImpl::initialize");
	corbaProblemEx.setMinor(_ex.minor());
	corbaProblemEx.setCompletionStatus(_ex.completed());
	corbaProblemEx.setInfo(_ex._info().c_str());
	GettingConfigurationDataProblemExImpl ex(corbaProblemEx, __FILE__, __LINE__, "LoggerImpl::initialize");
	ex.log(LM_DEBUG);
	}//try-catch

    try
	{
	initLogLevel = (ACS_RT::RTLogLevels)(dao->get_long("initLogLevel")-2);  // conversion from unsigned int to IDL enum !!
	}
    catch(cdbErrType::WrongCDBDataTypeEx &_ex)
	{
	GettingConfigurationDataProblemExImpl ex(_ex, __FILE__, __LINE__, "LoggerImpl::initialize");
	ex.setConfigData("initLogLevel");
	ex.log(LM_DEBUG);
	}
    catch(cdbErrType::CDBFieldDoesNotExistEx &_ex)
	{
	GettingConfigurationDataProblemExImpl ex(_ex, __FILE__, __LINE__, "LoggerImpl::initialize");
	ex.setConfigData("initLogLevel");
	ex.log(LM_DEBUG);
	}
    catch(CORBA::SystemException &_ex)
	{
	ACSErrTypeCommon::CORBAProblemExImpl corbaProblemEx(__FILE__, __LINE__, "LoggerImpl::initialize");
	corbaProblemEx.setMinor(_ex.minor());
	corbaProblemEx.setCompletionStatus(_ex.completed());
	corbaProblemEx.setInfo(_ex._info().c_str());
	GettingConfigurationDataProblemExImpl ex(corbaProblemEx, __FILE__, __LINE__, "LoggerImpl::initialize");
	ex.setConfigData("initLogLevel");
	ex.log(LM_DEBUG);
	}//try-catch

    try
	{
	m_FIFOSize = dao->get_long("FIFOsize");
	}
    catch(cdbErrType::WrongCDBDataTypeEx &_ex)
	{
	GettingConfigurationDataProblemExImpl ex(_ex, __FILE__, __LINE__, "LoggerImpl::initialize");
	ex.setConfigData("FIFOsize");
	ex.log(LM_DEBUG);
	}
    catch(cdbErrType::CDBFieldDoesNotExistEx &_ex)
	{
	GettingConfigurationDataProblemExImpl ex(_ex, __FILE__, __LINE__, "LoggerImpl::initialize");
	ex.setConfigData("FIFOsize");
	ex.log(LM_DEBUG);
	}
    catch(CORBA::SystemException &_ex)
	{
	ACSErrTypeCommon::CORBAProblemExImpl corbaProblemEx(__FILE__, __LINE__, "LoggerImpl::initialize");
	corbaProblemEx.setMinor(_ex.minor());
	corbaProblemEx.setCompletionStatus(_ex.completed());
	corbaProblemEx.setInfo(_ex._info().c_str());
	GettingConfigurationDataProblemExImpl ex(corbaProblemEx, __FILE__, __LINE__, "LoggerImpl::initialize");
	ex.setConfigData("FIFOsize");
	ex.log(LM_DEBUG);
	}//try-catch

    if ( (m_fd = rtf_open_sized(RTLOG_FIFO_NAME, O_CREAT, m_FIFOSize * sizeof(rtlogRecord_t))) < 0 )
	{
	throw CanNotCreateLogMsgFIFOExImpl(__FILE__, __LINE__, "rtlogImpl::initialize");
	}//if

    ACS_LOG(LM_SOURCE_INFO, "LoggerImpl::initialize", (LM_INFO, "RT FIFO with size %d has been created.",m_FIFOSize));

    // we lock the memory befor going to do set/get level (using LXRT)
    rt_allow_nonroot_hrt();
    /*    if ( mlockall(MCL_CURRENT ) != 0)
	{
	throw CanNotGetLevelExImpl(__FILE__, __LINE__, "rtlogImpl::initialize").getCanNotGetLevelEx();
	}
    */
    // we lock again the memory in TimeDataThread for the whole rtlog component. 

    try
	{
	setLevel(initLogLevel);
	ACS_LOG(LM_SOURCE_INFO, "LoggerImpl::initialize", (LM_INFO, "Set initial log level to: %d.", initLogLevel+2));
	}
    catch (CORBA::SystemException &ex)
	{
	ACSErrTypeCommon::CORBAProblemExImpl corbaProblemEx(__FILE__, __LINE__, "rtlogImpl::initialize");
	corbaProblemEx.setMinor(ex.minor());
	corbaProblemEx.setCompletionStatus(ex.completed());
	corbaProblemEx.setInfo(ex._info().c_str());

	throw CanNotSetInitialLogLevelExImpl(corbaProblemEx, __FILE__, __LINE__, "rtlogImpl::initialize");
	}
    catch(rtlogErrType::CanNotSetLevelEx &_ex)
	{
	throw CanNotSetInitialLogLevelExImpl(_ex, __FILE__, __LINE__, "rtlogImpl::initialize");
	}//try-catch

    try
	{
	rtlogImpl *self = this;
	
	// spawn the time data thread
	mp_timeDataThread = getContainerServices()->getThreadManager()->create<TimeDataThread>("rtlogTimeDataThread");
	mp_timeDataThread->resume();

	ACS_LOG(LM_SOURCE_INFO, "LoggerImpl::initialize", (LM_INFO, "Time Data thread created and resumed."));

	// spawn the logging thread
	mp_rtLogThread = getContainerServices()->getThreadManager()->create<RTLogThread, rtlogImpl>("loggerThread", *self);
	mp_rtLogThread->resume();
	ACS_LOG(LM_SOURCE_INFO, "LoggerImpl::initialize", (LM_INFO,"Logger thread created and resumed."));
	}
    catch(ACSErr::ACSbaseExImpl& ex)
	{ 
	close(m_fd);  
	if (mp_timeDataThread) getContainerServices()->getThreadManager()->destroy(mp_timeDataThread);

	throw CanNotSpawnLoggingThreadExImpl(ex, __FILE__, __LINE__, "rtlogImpl::initialize");
	}
    catch(...) // this in principal can not happend but ....
	{
	close(m_fd); 
	ACSErrTypeCommon::UnexpectedExceptionExImpl uex(__FILE__, __LINE__,
							"rtlogImpl::initialize");

	throw CanNotSpawnLoggingThreadExImpl(uex, __FILE__, __LINE__, "rtlogImpl::initialize");
	}
    
}//initialize
//--------------------------------------------------------------------------------------
void rtlogImpl::cleanUp()
{
    ACS_TRACE("::LoggerImpl::cleanUp");

    // stop and delete the threads
    mp_rtLogThread->stop();
    getContainerServices()->getThreadManager()->destroy(mp_rtLogThread);
    mp_timeDataThread->stop();
    getContainerServices()->getThreadManager()->destroy(mp_timeDataThread);

    close(m_fd); 
    /*
    if (munlockall()!=0)
      ACS_LOG(LM_SOURCE_INFO, "rtlogImpl::cleanUp", (LM_ERROR, "could not unlock the memory"));
    */
}//cleanUp

//--------------------------------------------------------------------------------------
ACS_RT::RTLogLevels rtlogImpl::getLevel()
{
    ACE_TRACE("rtlogImpl::getLevel");
    get_rtlog_level = 0;

    /* implementation using lxrt */
    rt_allow_nonroot_hrt();
    if (!(get_level_task = rt_task_init_schmod(nam2num("getLevel"), 10, 0, 0, SCHED_FIFO, 0xFF)))
	{
	throw CanNotGetLevelExImpl(__FILE__, __LINE__, "rtlogImpl::getLevel").getCanNotGetLevelEx();
	};//if

    get_rtlog_level = rtlogGetLevel(); //lxrt call

    rt_task_delete(get_level_task);

    return (ACS_RT::RTLogLevels)(get_rtlog_level-RTLOG_TRACE_LEVEL);
}//getLevel

//--------------------------------------------------------------------------------------

void rtlogImpl::setLevel(ACS_RT::RTLogLevels level) 
{
    ACE_TRACE("rtlogImpl::setLevel");
    set_rtlog_level=(RTLogLevelType)level;
   // this works only if IDL enum at first position is mapped to number 0
    set_rtlog_level += RTLOG_TRACE_LEVEL; // RTLOG_TRACE_LEVEL is offset of value 2

/* implementation using lxrt */
    rt_allow_nonroot_hrt();
    if (!(set_level_task = rt_task_init_schmod(nam2num("setLevel"), 10, 0, 0, SCHED_FIFO, 0xFF)))
	{
	throw CanNotSetLevelExImpl(__FILE__, __LINE__, "rtlogImpl::setLevel").getCanNotSetLevelEx();
	};//if

    rtlogSetLevel(set_rtlog_level); //lxrt call

    rt_task_delete(set_level_task);

    ACS_SHORT_LOG((LM_INFO, "rtlog_level has been set to %d", set_rtlog_level));
}//setRTLogLevel


//--------------------------------------------------------------------------------------

void rtlogImpl::setFIFOSize (CORBA::ULong size )
{
    int st;
    
    st = rtf_resize(m_fd, size * sizeof(rtlogRecord_t));
    
    m_FIFOSize = size;
}//setFIFOSize

//--------------------------------------------------------------------------------------

CORBA::ULong rtlogImpl::getFIFOSize () 
{
    return m_FIFOSize;
}//getFIFOSize

/////////////////////////////////////////////////
// MACI DLL support functions
/////////////////////////////////////////////////

#include <maciACSComponentDefines.h>
MACI_DLL_SUPPORT_FUNCTIONS(rtlogImpl)
