#ifndef _rtLog_H_
#define _rtLog_H_
/*******************************************************************************
* E.S.O. - VLT project
*
* "@(#) $Id: rtLog.h,v 1.9 2009/10/07 09:27:08 bjeram Exp $"
*
* who       when        what
* --------  ----------  ----------------------------------------------
* bjeram  yyyy-mm-dd  created
*/

#ifndef RTAI_HOME
#warning "Compiling for NO RTAI"
#define NO_RTAI_HOME
#endif

#ifndef NO_RTAI_HOME
#include <rtai_sched.h>
#include <rtai_fifos.h>
#else
#include <logging.h>
#include <acsutilTimeStamp.h>
#endif

/**
 * log message structure
 */
typedef struct rtlogRecord
{
	long long time;
	long priority;
	char file[32];
	int  line;
	char text[256];
        char lkm[20];
} rtlogRecord_t;


/*
 if we are not in RT world we do not need a ot of things
 */
#ifndef NO_RTAI_HOME

#define RTLOG_FIFO_NAME		"/dev/rtf0"
#define RTLOG_FIFO_SIZE		5000		/* records */
#define RTLOG_TASK_PRIO		10

/*
 * public API (LXRT)
 */
#define RTLOG_SET_LOG_LEVEL	0
#define RTLOG_GET_LOG_LEVEL	1
#define RTLOG_GET_TIME_DATA     2

#define RTLOG_IDX               12 /* could be that is already used by someone else */

/* log level type */
typedef unsigned int RTLogLevelType;

/*
 * structure that contains data that are used for calculating the time stamp 
 * in rtlog component.
 *
*/
typedef struct rtLogTimeData
{
    RTIME cpu_on_last_te;    /* TSC at last TE */
    unsigned int cpu_hz;    /* estimated cpu frequency */
    unsigned int ticks;      /*  number of ticks (time events) handled, so far */
    RTIME t0;                /* abosulte time at tick zero (ACS)	*/
} rtLogTimeData_t;


#ifdef __KERNEL__

/*
 * function to set rtlog_time_data. This function can be used just from another module 9teHandler).
 */
void setRtLogTimeData(rtLogTimeData_t td);

/* in kerenl space we want to access rtlog_level directlly 
 instead using getRTLogLevel for the performance reason
*/
extern atomic_t rtlog_level;
#define CHECK_LOG_LEVEL(LOG_MSG_L) if ( LOG_MSG_L >= atomic_read(&rtlog_level) )
#else
static inline RTLogLevelType  rtlogGetLevel(void)
{
	struct {int n; } arg = { 0 }; /** mandatory one (unused) element in your struct. */
	return (RTLogLevelType)rtai_lxrt(RTLOG_IDX,
					 SIZARG,
					 RTLOG_GET_LOG_LEVEL,
					 &arg).rt;
}

static inline int rtlogSetLevel(RTLogLevelType level)
{
	struct {RTLogLevelType level; } arg = { level };
	return rtai_lxrt(RTLOG_IDX,
			 SIZARG,
			 RTLOG_SET_LOG_LEVEL,
			 &arg).i[LOW];
}

static inline int rtlogGetTimeData (rtLogTimeData_t *td)
{

	struct {rtLogTimeData_t* td; int sz; } arg = { td, sizeof(rtLogTimeData_t) };
	return rtai_lxrt(RTLOG_IDX,
			 SIZARG,
			 RTLOG_GET_TIME_DATA,
			 &arg).i[LOW];

}/* rtlogGetTimeData */

#define CHECK_LOG_LEVEL(LOG_MSG_L) if ( LOG_MSG_L >= rtlogGetLevel() )
#endif

#endif /*NO_RTAI_HOME*/

/* log levels which coresponds to ACE/ACS levels
see: $ACE_ROOT/Log_Priority.h
*/
#define RTLOG_TRACE_LEVEL       2
#define RTLOG_DEBUG_LEVEL	3
#define RTLOG_INFO_LEVEL        4
#define RTLOG_NOTICE_LEVEL      5
#define RTLOG_WARNING_LEVEL     6
#define RTLOG_ERROR_LEVEL	8
#define RTLOG_CRITICAL_LEVEL    9
#define RTLOG_ALERT_LEVEL      10
#define RTLOG_EMERGENCY_LEVEL   11

#define RTLOG_TRACE_MSG         "trace"
#define RTLOG_DEBUG_MSG		"debug"
#define RTLOG_INFO_MSG          "info-normal"
#define RTLOG_NOTICE_MSG        "notice"
#define RTLOG_WARNING_MSG       "warning"
#define RTLOG_ERROR_MSG         "error"
#define RTLOG_CRITICAL_MSG      "critical"
#define RTLOG_ALERT_MSG         "alert"
#define RTLOG_EMERGENCY_MSG     "emergency"


#define RTLOG(mod, log_macro, ...) log_macro(mod, __VA_ARGS__)

/*
 * Set the RTLOG_LEVEL to a default value, if it is not defined elsewhere.
 */
#ifndef RTLOG_LEVEL
#define RTLOG_LEVEL RTLOG_TRACE_LEVEL
#endif

#if RTLOG_LEVEL <= RTLOG_TRACE_LEVEL
#define RTLOG_TRACE( mod, ...) _RTLOG_(RTLOG_TRACE_MSG, mod, RTLOG_TRACE_LEVEL, __VA_ARGS__)
#else
#define RTLOG_TRACE( mod, ...)
#endif

#if RTLOG_LEVEL <= RTLOG_DEBUG_LEVEL
#define RTLOG_DEBUG( mod, ...) _RTLOG_(RTLOG_DEBUG_MSG, mod, RTLOG_DEBUG_LEVEL, __VA_ARGS__)
#else
#define RTLOG_DEBUG( mod, ...)
#endif

#if RTLOG_LEVEL <= RTLOG_INFO_LEVEL
#define RTLOG_INFO( mod,  ...) _RTLOG_(RTLOG_INFO_MSG, mod, RTLOG_INFO_LEVEL,  __VA_ARGS__)
#else
#define RTLOG_INFO( mod,  ...)
#endif

#if RTLOG_LEVEL <= RTLOG_NOTICE_LEVEL
#define RTLOG_NOTICE( mod,  ...) _RTLOG_(RTLOG_NOTICE_MSG, mod, RTLOG_NOTICE_LEVEL,  __VA_ARGS__)
#else
#define RTLOG_NOTICE( mod,  ...)
#endif

#if RTLOG_LEVEL <= RTLOG_WARNING_LEVEL
#define RTLOG_WARNING( mod,  ...) _RTLOG_(RTLOG_WARNING_MSG, mod, RTLOG_WARNING_LEVEL,  __VA_ARGS__)
#else
#define RTLOG_WARNING( mod,  ...)
#endif

#if RTLOG_LEVEL <= RTLOG_ERROR_LEVEL
#define RTLOG_ERROR( mod,  ...) _RTLOG_(RTLOG_ERROR_MSG, mod, RTLOG_ERROR_LEVEL,  __VA_ARGS__)
#else
#define RTLOG_ERROR( mod,  ...)
#endif

#if RTLOG_LEVEL <= RTLOG_CRITICAL_LEVEL
#define RTLOG_CRITICAL( mod,  ...) _RTLOG_(RTLOG_CRITICAL_MSG, mod, RTLOG_CRITICAL_LEVEL,  __VA_ARGS__)
#else
#define RTLOG_CRITICAL( mod,  ...)
#endif

#if RTLOG_LEVEL <= RTLOG_ALERT_LEVEL
#define RTLOG_ALERT( mod,  ...) _RTLOG_(RTLOG_ALERT_MSG, mod, RTLOG_ALERT_LEVEL,  __VA_ARGS__)
#else
#define RTLOG_ALERT( mod,  ...)
#endif

#if RTLOG_LEVEL <= RTLOG_EMERGENCY_LEVEL
#define RTLOG_EMERGENCY( mod,  ...) _RTLOG_(RTLOG_EMERGENCY_MSG, mod, RTLOG_EMERGENCY_LEVEL,  __VA_ARGS__)
#else
#define RTLOG_EMERGENCY( mod,  ...)
#endif

/* If we are in non-RT enviroment 
   we send log message directly to the logging system using ACS macros
   otherwise we send it through the FIFO
 */  
#ifndef NO_RTAI_HOME
/*
 __FILE__ macro is expended is something else if we build the LKM
 */
#ifdef __KERNEL__

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 19) /* Kernel version >= 2.6.19 */
#define SHORT__FILE__ KBUILD_BASENAME
#define __FILE_EXT__ ".c"
#else /* Kernel version < 2.6.19 */ 
#define _STR2(B) #B
#define _STR1(A) _STR2(A.c)
#define SHORT__FILE__ _STR1(KBUILD_BASENAME)
#define __FILE_EXT__ ""
#endif 

#else
#define SHORT__FILE__ __FILE__          
#define __FILE_EXT__ ""
#endif 

#define _RTLOG_(logtype, mod, prio, format, ...)                                \
do {                                                                            \
CHECK_LOG_LEVEL( prio )  {							\
logRecord.time = rt_get_time(); 						\
logRecord.priority = prio;							\
logRecord.file[31] = 0;								\
snprintf(logRecord.file, 31, "%s%s", SHORT__FILE__, __FILE_EXT__);		\
logRecord.lkm[19] = 0;								\
snprintf(logRecord.lkm, 19, "%s", mod);                				\
logRecord.line = __LINE__;							\
logRecord.text[255] = 0;							\
snprintf(logRecord.text, 255, format, ##__VA_ARGS__);				\
if ( rtf_put(0, &logRecord, sizeof(logRecord)) != sizeof(logRecord) )		\
{                                                                               \
rt_printk(mod " (" logtype "): %s%s(%d) - " format "\n", SHORT__FILE__, __FILE_EXT__, __LINE__, ## __VA_ARGS__); \
}                                                                               \
}}while(0);
#else
#define _RTLOG_(logtype, mod, prio, format, ...)                                \
do {                                                                            \
   ACS_CHECK_LOGGER;                                                            \
   logRecord.text[255] = 0;							\
   snprintf(logRecord.text, 255, format, ##__VA_ARGS__);			\
   LoggingProxy::Flags(LM_SOURCE_INFO | LM_RUNTIME_CONTEXT);                    \
   LOG_RECORD(Logging::ace2acsPriority(ACE_Log_Priority(1 << (prio -1))),       \
	      logRecord.text,                                                   \
              __FILE__,                                                         \
	      __LINE__,                                                         \
	      "_RTLOG_",                                                        \
             getTimeStamp(),                                                    \
	      mod);                                                             \
}while(0);
#endif /* !NO_RTAI_HOME */

#endif /*!_H*/
