/*******************************************************************************
* ALMA - Atacama Large Millimiter Array
* (c) European Southern Observatory, 2007 
* 
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2.1 of the License, or (at your option) any later version.
* 
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*
* "@(#) $Id: rtAlarmImpl.cpp,v 1.11 2009/10/13 11:02:51 bjeram Exp $"
*
* who       when      what
* --------  --------  ----------------------------------------------
* bjeram  2007-06-26  created 
*/

#include "vltPort.h"

static char *rcsId="@(#) $Id: rtAlarmImpl.cpp,v 1.11 2009/10/13 11:02:51 bjeram Exp $"; 
static void *use_rcsId = ((void)&use_rcsId,(void *) &rcsId);

#include "rtAlarmImpl.h"
#include <faultStateConstants.h>
#include <ACSAlarmSystemInterfaceFactory.h>

#include <ACSErrTypeCommon.h>
#include <acsErrTypeLifeCycle.h>

/// if we send this descriptor signals reading thread to exit. It is used just internally!!!
#define ALARM_EXIT_SIG  255  

using namespace rtAlarm;

// --------------------------------------------------------------------------------------------------------------
// class rtAlarmThread
rtAlarmThread::rtAlarmThread(const ACE_CString &_name, rtAlarmSourceData &_asd)
    : ACS::Thread(_name, 1000*1000*10/*respons time =1s*/),
      asd(_asd)
{
    ACS_TRACE("rtAlarmThread::rtAlarmThread");
    ACE_CString taskName="T";
    taskName += asd.name;

    // here we check if RT task with _name already exist
    if (rt_get_adr(nam2num(taskName.c_str())))
	{
	throw ACSErrTypeCommon::CouldntCreateThreadExImpl(__FILE__, __LINE__, "rtAlarmThread::rtAlarmThread");
	}

    try
	{
	alarmSource_mp = ACSAlarmSystemInterfaceFactory::createSource(asd.name.c_str());
	//@tbd should we setHostName or is it done automatically?
	faultState_map  = ACSAlarmSystemInterfaceFactory::createFaultState(asd.faultFamily.c_str(), asd.faultMember.c_str(), 0); // should we put into run?

	} 
    catch (acsErrTypeAlarmSourceFactory::ACSASFactoryNotInitedExImpl &ex) 
	{
	throw ACSErrTypeCommon::CouldntCreateThreadExImpl(ex, __FILE__, __LINE__, "rtAlarmThread::rtAlarmThread");
	}//try-catch

    // could be that creating MBX in user space (not LXRT) can cause a p[roblem, but so far it works and because of 
    // error handling it is better that is done here than in onStop
    alarmMBX_p = rt_named_mbx_init(asd.name.c_str(), sizeof(rtAlarm_t)*RTALARM_MBX_SIZE);
    if (alarmMBX_p==NULL)
	{
	throw ACSErrTypeCommon::CouldntCreateThreadExImpl( __FILE__, __LINE__, "rtAlarmThread::rtAlarmThread");
	}//if

}//rtAlarmThread

rtAlarmThread::~rtAlarmThread()
{
    ACS_TRACE("rtAlarmThread::~rtAlarmThread");
    rtAlarm_t exitSignal = {ALARM_EXIT_SIG, "",0 , (rtAlarmState_t)ALARM_EXIT_SIG}; // signal for exit
    int ret;
    RT_TASK *t;
// Strategy to exit the thread is to send so called EXIT alarm, or if this doe not work wait that
// rt_mbx_receive timeout and than the thread exit

    ThreadBase::exit();  // we signal that we want to exit the loop ..

    rt_allow_nonroot_hrt();
    if ( mlockall(MCL_CURRENT) != 0)
	{
	// here we can just log error message
	ACS_LOG(LM_FULL_INFO, "rtAlarmThread::~rtAlarmThread", (LM_ERROR, "Problem locking memory"));
	}

    ACE_CString taskName="E";
    taskName += asd.name;
    // we have to create RT task to sen a message to exit
    if (!(t = rt_task_init(nam2num(taskName.c_str()), 10, 0, 0)))
	{
	ACS_LOG(LM_FULL_INFO, "rtAlarmThread::~rtAlarmThread",
		(LM_WARNING, "Problem creating RT Task %s.", taskName.c_str()));
	sleep(1); // here we sleep for 1sec(=timeout of rt_mbx_receive) and thread should exit itself
	return; // there is no seens to continue
	};//if 

    // we send exit signal that exit thread exits
    if (ret=rt_mbx_send(alarmMBX_p, &exitSignal, sizeof(rtAlarm_t))!=0)  
	{
	ACS_LOG(LM_FULL_INFO, "rtAlarmThread::~rtAlarmThread",
		(LM_WARNING, "Problem sending exit event to MBX. Reason %d.", ret));
	sleep(1); // here we sleep for 1sec(=timeout of rt_mbx_receive) and thread should exit itself
	}//if

    rt_task_delete(t);
}//~rtAlarmThread


void rtAlarmThread::onStart()
{
    ACS_TRACE("rtAlarmThread::onStart");
    rt_allow_nonroot_hrt();
    if ( mlockall(MCL_CURRENT) != 0)
	{
	// here we can just log error message
	ACS_LOG(LM_FULL_INFO, "rtAlarmThread::onStart", (LM_ERROR, "Problem locking memory"));
	}
    
    ACE_CString taskName="T";
    taskName += asd.name;

    if (!(readingAlarmsTask = rt_task_init(nam2num(taskName.c_str()), 10, 0, 0)))
	{
	// here we can just log error message
	ACS_LOG(LM_FULL_INFO, "rtAlarmThread::onStart",
		(LM_ERROR, "Problem creating RT Task with name %s.", taskName.c_str()));
	};//if 

}//onStart

void rtAlarmThread::run()
{
    rtAlarm_t rtAlarm;
    rtLogTimeData_t timeData; // local time data
    ACS::Time timeStamp;
    long secTS, usecTS;
    int ret;

    if (readingAlarmsTask==0 || alarmMBX_p==0)
	{
		ACS_LOG(LM_FULL_INFO, "rtAlarmThread::run",
		(LM_ERROR, "RT Task T%s or MBX %s is NULL! Can not prceed. Exiting the thread", asd.name.c_str(), asd.name.c_str()));
		return;
	}//if

    while (check())
	{

	if( isSuspended() )//hendling suspended!!!
	    {
	    sleep();
	    continue; //go to next loop to check if we have to exit
	    }//if

//	printf ("++++++++++++++++++++++++++++++++++++++ going to receive \n");
	// we read the MBX with timeout of 1sec
	ret = rt_mbx_receive_timed(alarmMBX_p, &rtAlarm, sizeof(rtAlarm_t), nano2count(1000000000)); 
//	printf ("++++++++++++++++++++++++++++++++++++++ received %d\n", ret);
	
	if (ret!=0)  // in case of an error (timeout) we go to next loop
	    {
	    if (ret==272) sleep(); // in case if we have problem with timer
	    continue;
	    }
	
	switch (rtAlarm.descriptor)
	    {
	    case rtAlarm_ACTIVE:
		faultState_map->setDescriptor(faultState::ACTIVE_STRING);
		break;
	    case rtAlarm_TERMINATE:
		faultState_map->setDescriptor(faultState::TERMINATE_STRING);
		break;
	    case ALARM_EXIT_SIG:
		if (rtAlarm.alarmCode!=ALARM_EXIT_SIG) 	ACS_LOG(LM_FULL_INFO, "rtAlarmThread::run",
								(LM_WARNING, "Exit signal should come with exit code!"));
		continue;
		break;
	    default:
		faultState_map->setDescriptor("UNKNOWN state");
	    }//switch
	
	faultState_map->setCode(rtAlarm.alarmCode);  // set to the code that was read from the mailbox

	// create a Timestamp and use it to configure the FaultState
	timeData = asd.timeDataSource_mp->getTimeData();
	if (timeData.ticks != 0) 
	    {
	    timeStamp = ((rtAlarm.timeStamp - timeData.cpu_on_last_te) *
			   10000000ULL /
			   (ACS::Time)timeData.cpu_hz +
			   (ACS::Time)timeData.ticks *
			   (ACS::Time)480000 +
			   (ACS::Time)timeData.t0); 
	    
	    secTS =  ACE_static_cast(CORBA::ULongLong, timeStamp) / ACE_static_cast(ACE_UINT32, 10000000u) - ACE_UINT64_LITERAL(0x2D8539C80); 
	    usecTS = (timeStamp % 10000000u) / 10; 

	    auto_ptr<acsalarm::Timestamp> tstampAutoPtr(new acsalarm::Timestamp(secTS, usecTS)); //@TBD should be optimized
	    faultState_map->setUserTimestamp(tstampAutoPtr); 
	    }
	else
	    {
	    auto_ptr<acsalarm::Timestamp> tstampAutoPtr(new acsalarm::Timestamp());  //@TBD should be optimized
	    faultState_map->setUserTimestamp(tstampAutoPtr); 
	    }//if-else


	if ( strlen(rtAlarm.property)>0 )
	    {
//	    printf("set property: %s\n",rtAlarm.property );
	    auto_ptr<acsalarm::Properties> props(new acsalarm::Properties()); //@tbd for some reason setUserProperties deletes contents of props so it has to be recreated evry time
	    props->setProperty("rtAlarm.property", rtAlarm.property);
	    faultState_map->setUserProperties(props);
	    }//if

	//here should be error handling, but push does not return any error information
	alarmSource_mp->push(*faultState_map); // send the alarm
    }//while

// we exit the loop and we have to delete RT stuff
    if(ret=rt_named_mbx_delete(alarmMBX_p)<0)
	{
	ACS_LOG(LM_FULL_INFO, "rtAlarmThread::run",
		(LM_WARNING, "Problem deleting MBX %s. Reason: %d.", asd.name.c_str(), ret));
	}//if
    if(ret=rt_task_delete(readingAlarmsTask)!=0)
	{
	ACS_LOG(LM_FULL_INFO, "rtAlarmThread::run",
		(LM_WARNING, "Problem deleting RT Task T%s. Reason: %d.", asd.name.c_str(), ret));
	}//if
}//run

// --------------------------------------------------------------------------------------------------------------
// --------------------------------------------------------------------------------------------------------------
// --------------------------------------------------------------------------------------------------------------
// class rtAlarmImpl

rtAlarmImpl::rtAlarmImpl(const ACE_CString &name, maci::ContainerServices* containerServices) :
    baci::CharacteristicComponentImpl(name, containerServices)
{
    ACS_TRACE("rtAlarmImpl::rtAlarmImpl");
}//rtAlarmImpl


rtAlarmImpl::~rtAlarmImpl()
{
    ACS_TRACE("rtAlarmImpl::~rtAlarmImpl");
}//~rtAlarmImpl

void rtAlarmImpl::initialize() 
{
    rtAlarmThread *thr;
    ACE_CString fullPath = "alma/";

    CDB::DAL_var cdb = getContainerServices()->getCDB();
  
    fullPath += name();
    char *xml = cdb->get_DAO(fullPath.c_str());
    rtAlarmCDBParser parser;
    if (parser.parse(xml, &rtAlarmSources))
	{
	ACSErrTypeCommon::ParsingXMLProblemExImpl ex(__FILE__, __LINE__, "rtAlarmImpl::initialize");
	ex.setXML(xml);
	throw ex;
	}//if

    try
	{
	rt_allow_nonroot_hrt();
	if ( mlockall(MCL_CURRENT) != 0)
	{
	// here we can just log error message
	ACS_LOG(LM_FULL_INFO, "rtAlarmImpl::initialize", (LM_ERROR, "Problem locking memory"));
	}

	// spawn the time data thread
	timeDataThread_mp = getContainerServices()->getThreadManager()->create<TimeDataThread>("TDrtAlarm");
	timeDataThread_mp->resume();


	for (std::vector<rtAlarmSourceData>::iterator asd = rtAlarmSources.begin();  asd!=rtAlarmSources.end();  ++asd)
	    {
//	printf("Source: %s, %s, %s\n", asd->name.c_str(), asd->faultFamily.c_str(), asd->faultMember.c_str());
	    asd->timeDataSource_mp = timeDataThread_mp;
	    thr = getContainerServices()->getThreadManager()->
		create<rtAlarmThread, rtAlarmSourceData>(asd->name+"_rtAlarm_thread", *asd);
	    thr->resume();
	    rtAlarmThreads.push_back(thr); 
	    }//for
	}
    catch(ACSErr::ACSbaseExImpl &_ex)
	{
	throw acsErrTypeLifeCycle::StartingThreadsFailureExImpl(_ex, __FILE__, __LINE__, "rtAlarmImpl::initialize");
	}//try-catch
}//initialize

void rtAlarmImpl::cleanUp()
{
    ACS_TRACE("rtAlarmImpl::cleanUp");
    
    for (std::vector<rtAlarmThread*>::iterator thr = rtAlarmThreads.begin();  thr!=rtAlarmThreads.end();  ++thr)
	{
	getContainerServices()->getThreadManager()->destroy(*thr);
	}
    rtAlarmThreads.clear();

    getContainerServices()->getThreadManager()->destroy(timeDataThread_mp);

    if (munlockall()!=0)
      ACS_LOG(LM_SOURCE_INFO, "rtAlarmImpl::cleanUp", (LM_ERROR, "could not unlock the memory"));
}//cleanUp

//
//--------------------------------------------------------------------------------------
//  parser
//

int rtAlarmImpl::rtAlarmCDBParser::parse(char *xml, std::vector<rtAlarmSourceData>* asd)
{
	XML_Parser p = XML_ParserCreate(NULL);
	if (! p)	
	    {
	    ACS_STATIC_LOG(LM_FULL_INFO, "rtAlarmImpl::rtAlarmCDBParser::parse",
			   (LM_WARNING, "Creating parser error")); 
	    return 1;
	    }
	
	  //Connect to the parser the handler for the end and the start of a tag
	XML_SetElementHandler(p, startElm, endElm);

	XML_SetUserData(p, asd);

	// We have all the xml in the string so we parse all the document
	// with just one call
	if (XML_Parse(p, xml, strlen(xml), 1)==0)
        {
	ACS_STATIC_LOG(LM_FULL_INFO, "rtAlarmImpl::rtAlarmCDBParser::parse",
		       (LM_WARNING, "Erro parsing CDB entry: %s", xml));
		return 1;
        }
        // Release the memory used by the parser
	XML_ParserFree(p);

	return 0;
}
	
void rtAlarmImpl::rtAlarmCDBParser::startElm(void *data, const XML_Char *el, const XML_Char **attr)
{
    int i=0;
   
    if (strcmp(el, "rtAlarmSource")==0)
	{
//	printf("Element %s\n", el);
   
	while(attr[i]!=0)
	    {
//	    printf("atrribute: %s of value %s\n", attr[i], attr[i+1]);
	    if (strcmp(attr[i], "name")==0)
		{
		asd.name = attr[++i];
		i++;
		continue;
		}

	    if (strcmp(attr[i], "faultFamily")==0)
		{
		asd.faultFamily = attr[++i];
		i++;
		continue;
		}

	    if (strcmp(attr[i], "faultMember")==0)
		{
		asd.faultMember = attr[++i];
		i++;
		continue;
		}
	    
	    ACS_STATIC_LOG(LM_FULL_INFO, "rtAlarmImpl::rtAlarmCDBParser::startElm",
			   (LM_WARNING, "Strange attribute found: %s %s\n", attr[i++], attr[i++]));
	    }//while

	}//if
}//startElm

void rtAlarmImpl::rtAlarmCDBParser::endElm(void *data, const XML_Char *el)
{
    if (strcmp(el, "rtAlarmSource")==0)
	{
//	printf("End-Element %s\n", el);
	// we add data to the vector
	((std::vector<rtAlarmSourceData>*)data)->push_back(asd);
	}//if
}//endElm

//static members
rtAlarmSourceData rtAlarmImpl::rtAlarmCDBParser::asd;


/////////////////////////////////////////////////
// MACI DLL support functions
/////////////////////////////////////////////////

#include <maciACSComponentDefines.h>
MACI_DLL_SUPPORT_FUNCTIONS(rtAlarmImpl)

/*___oOo___*/
