/*
* Copyright (C) The Community OpenORB Project. All rights reserved.
*
* This software is published under the terms of The OpenORB Community Software
* License version 1.0, a copy of which has been included with this distribution
* in the LICENSE.txt file.
*/

package org.openorb.io;

import java.util.List;
import java.util.Vector;
import org.omg.CORBA.portable.InputStream;
import org.openorb.util.NumberCache;

/**
 * This class is a very simple OutputStream that implements the basic functionalities
 * for transfering arguments between a server and a client in the same JVM. It is useful
 * for stubs that do not handle local calls, like those generated by the JDK 1.4 idlj.
 * This class uses a {@link java.util.List} for storing the arguments.
 * @author <a href="erik.putrycz@int-evry.fr">Erik Putrycz</a>
 * @version $Revision: 1.3 $ $Date: 2002/08/01 13:26:59 $ 
 */

public class LocalOutputStream extends org.omg.CORBA_2_3.portable.OutputStream
{

    private List m_arglist;

    /** Creates a new instance of LocalOutputStream */
    public LocalOutputStream()
    {
        // Note that LocalOutputStream accesses m_arglist using index operations,
        // so using a LinkedList is probably not a good idea, although
        // the implementation of this class might suggest that.

        // TODO: if synchronization is unnecessary we should use ArrayList here
        m_arglist = new Vector();
    }

    public InputStream create_input_stream()
    {
        return new LocalInputStream( m_arglist );
    }

    public void write_Object( org.omg.CORBA.Object value )
    {
        m_arglist.add( value );
    }

    public void write_TypeCode( org.omg.CORBA.TypeCode value )
    {
        m_arglist.add( value );
    }

    public void write_any( org.omg.CORBA.Any value )
    {
        m_arglist.add( value );
    }

    public void write_boolean( boolean value )
    {
        m_arglist.add( value ? Boolean.TRUE : Boolean.FALSE );
    }

    public void write_boolean_array( boolean[] value, int offset, int length )
    {
        if ( ( offset == 0 ) && ( length == value.length ) )
            m_arglist.add( value );
        else
        {
            boolean[] new_array = new boolean[ length ];
            System.arraycopy( value, offset, new_array, 0, length );
            m_arglist.add( new_array );
        }
    }

    public void write_char( char value )
    {
        m_arglist.add( new Character( value ) );
    }

    public void write_char_array( char[] value, int offset, int length )
    {
        if ( ( offset == 0 ) && ( length == value.length ) )
            m_arglist.add( value );
        else
        {
            char[] new_array = new char[ length ];
            System.arraycopy( value, offset, new_array, 0, length );
            m_arglist.add( new_array );
        }
    }

    public void write_double( double value )
    {
        m_arglist.add( NumberCache.getDouble( value ) );
    }

    public void write_double_array( double[] value, int offset, int length )
    {
        if ( ( offset == 0 ) && ( length == value.length ) )
            m_arglist.add( value );
        else
        {
            double[] new_array = new double[ length ];
            System.arraycopy( value, offset, new_array, 0, length );
            m_arglist.add( new_array );
        }
    }

    public void write_float( float value )
    {
        m_arglist.add( NumberCache.getFloat( value ) );
    }

    public void write_float_array( float[] value, int offset, int length )
    {
        if ( ( offset == 0 ) && ( length == value.length ) )
            m_arglist.add( value );
        else
        {
            float[] new_array = new float[ length ];
            System.arraycopy( value, offset, new_array, 0, length );
            m_arglist.add( new_array );
        }
    }

    public void write_long( int value )
    {
        m_arglist.add( NumberCache.getInteger( value ) );
    }

    public void write_long_array( int[] value, int offset, int length )
    {
        if ( ( offset == 0 ) && ( length == value.length ) )
            m_arglist.add( value );
        else
        {
            int[] new_array = new int[ length ];
            System.arraycopy( value, offset, new_array, 0, length );
            m_arglist.add( new_array );
        }
    }

    public void write_longlong( long value )
    {
        m_arglist.add( NumberCache.getLong( value ) );
    }

    public void write_longlong_array( long[] value, int offset, int length )
    {
        if ( ( offset == 0 ) && ( length == value.length ) )
            m_arglist.add( value );
        else
        {
            long[] new_array = new long[ length ];
            System.arraycopy( value, offset, new_array, 0, length );
            m_arglist.add( new_array );
        }
    }

    public void write_octet( byte value )
    {
        m_arglist.add( NumberCache.getByte( value ) );
    }

    public void write_octet_array( byte[] value, int offset, int length )
    {
        if ( ( offset == 0 ) && ( length == value.length ) )
            m_arglist.add( value );
        else
        {
            byte[] new_array = new byte[ length ];
            System.arraycopy( value, offset, new_array, 0, length );
            m_arglist.add( new_array );
        }
    }

    public void write_short( short value )
    {
        m_arglist.add( NumberCache.getShort( value ) );
    }

    public void write_short_array( short[] value, int offset, int length )
    {
        if ( ( offset == 0 ) && ( length == value.length ) )
            m_arglist.add( value );
        else
        {
            short[] new_array = new short[ length ];
            System.arraycopy( value, offset, new_array, 0, length );
            m_arglist.add( new_array );
        }
    }

    public void write_string( String value )
    {
        m_arglist.add( value );
    }

    public void write_ulong( int value )
    {
        m_arglist.add( NumberCache.getInteger( value ) );
    }

    public void write_ulong_array( int[] value, int offset, int length )
    {
        write_long_array( value, offset, length );
    }

    public void write_ulonglong( long value )
    {
        write_longlong( value );
    }

    public void write_ulonglong_array( long[] value, int offset, int length )
    {
        write_longlong_array( value, offset, length );
    }

    public void write_ushort( short value )
    {
        write_short( value );
    }

    public void write_ushort_array( short[] value, int offset, int length )
    {
        write_short_array( value, offset, length );
    }

    public void write_wchar( char value )
    {
        write_char( value );
    }

    public void write_wchar_array( char[] value, int offset, int length )
    {
        write_char_array( value, offset, length );
    }

    public void write_wstring( String value )
    {
        write_string( value );
    }

}
