/*
* Copyright (C) The Community OpenORB Project. All rights reserved.
*
* This software is published under the terms of The OpenORB Community Software
* License version 1.0, a copy of which has been included with this distribution
* in the LICENSE.txt file.
*/
package org.openorb.iiop;

import java.io.IOException;
import java.io.InterruptedIOException;

import java.net.InetAddress;
import java.net.Socket;
import java.net.ServerSocket;
import java.net.SocketException;

import org.apache.avalon.framework.logger.Logger;

import org.omg.IIOP.ListenPoint;

import org.omg.PortableInterceptor.ORBInitInfo;

import org.openorb.net.Transport;
import org.openorb.net.TransportServerInitializer;

import org.openorb.PI.FeatureInitInfo;

import org.openorb.util.ExceptionTool;

import org.omg.CORBA.COMM_FAILURE;

/**
 * Interface for creating sockets.
 *
 * @author Chris Wood
 * @version $Revision: 1.15 $ $Date: 2002/09/15 21:48:46 $
 */
public class IIOPTransportServerInitializer implements TransportServerInitializer
{
    private ListenPoint m_primary_endpoint = new ListenPoint();
    private ListenPoint [] m_bidir_endpoints = new ListenPoint[] { m_primary_endpoint };

    private InetAddress m_host;

    private ServerSocket m_svr_socket;

    private Logger m_logger;
    private volatile boolean m_closed = true;

    private final boolean m_socketNoDelay;
    private final boolean m_boostRecvPriority;
    private final int m_sendBufferSize;
    private final int m_receiveBufferSize;
    private final int m_bufferedOutputStreamSize;
    
    private final int m_backlogQueueLength;

    private final int m_maxSocketTimeout;
    private final int m_minSocketTimeout;
    private final int m_overrideSocketTimeout;
    
    private int m_currentSocketTimeout;

    /**
     * Any replacement classes must implement an identical constructor.
     */
    public IIOPTransportServerInitializer( ORBInitInfo orbinfo, FeatureInitInfo featureinfo )
    {
        org.openorb.CORBA.kernel.ORBLoader loader = featureinfo.getLoader();

        m_logger = ((org.openorb.CORBA.ORBSingleton) featureinfo.orb()).getLogger();

        m_socketNoDelay = 
                loader.getBooleanProperty( "iiop.serverNoDelay", true );
        
        m_boostRecvPriority = 
                loader.getBooleanProperty( "iiop.boostReceivePriority", false );
        
        m_sendBufferSize = 
                loader.getIntProperty( "iiop.sendBufferSize", 0 );

        m_receiveBufferSize = 
                loader.getIntProperty( "iiop.receiveBufferSize", 0 );

        m_bufferedOutputStreamSize = 
                loader.getIntProperty( "iiop.bufferedOutputStreamSize", 0 );

        m_backlogQueueLength = 
                loader.getIntProperty( "iiop.serverBacklogQueueLength", 50 );

        m_maxSocketTimeout = 
                loader.getIntProperty( "iiop.serverMaxSocketAcceptTimeout", 250 );
        
        // if this class has been extended then ensure that min time is 0
        m_minSocketTimeout = IIOPTransportServerInitializer.class.equals(getClass()) 
                ? loader.getIntProperty( "iiop.serverMinSocketAcceptTimeout", 0 ) 
                : 0;
        
        m_overrideSocketTimeout = 
                loader.getIntProperty( "iiop.serverOverrideSocketTimeout", 250 );

        m_primary_endpoint.host = loader.getStringProperty( "iiop.hostname", "" );

        if ( m_primary_endpoint.host.length() == 0 )
        {
            try
            {
                String publishIP = loader.getStringProperty( "iiop.publishIP", "auto" );

                if ( publishIP.equalsIgnoreCase( "auto" ) )
                {
                    m_primary_endpoint.host = InetAddress.getByName(
                            InetAddress.getLocalHost().getHostAddress() ).getHostName();

                    if ( m_primary_endpoint.host.indexOf( '.' ) < 0 )
                        m_primary_endpoint.host = InetAddress.getLocalHost().getHostAddress();
                }
                else if ( publishIP.equalsIgnoreCase( "true" ) )
                    m_primary_endpoint.host = InetAddress.getLocalHost().getHostAddress();
                else
                    m_primary_endpoint.host = InetAddress.getByName(
                            InetAddress.getLocalHost().getHostAddress() ).getHostName();
            }
            catch ( final java.net.UnknownHostException ex )
            {
                // TODO: give this a number
                getLogger().error( "Unable to find hostname for local host.", ex );

                throw ExceptionTool.initCause( new org.omg.CORBA.INITIALIZE(
                        "Unable to find hostname for local host (" + ex + ")",
                        0, org.omg.CORBA.CompletionStatus.COMPLETED_NO), ex );
            }
        }

        int port = loader.getIntProperty( "iiop.port", 0 );

        if ( port > 0xFFFF || port < 0 )
            throw new org.omg.CORBA.INITIALIZE(
                    "Value for server port " + port + " is out of range" );

        m_primary_endpoint.port = ( short ) port;

        try
        {
            // 0.0.0.0 means that no specific interface on a multi-homed host
            // has been specified, i.e. the primary should be used...
            m_host = InetAddress.getByName( loader.getStringProperty(
                    "iiop.listenAddress", "0.0.0.0" ) );
        }
        catch (final java.net.UnknownHostException ex)
        {
            // TODO: give this a number
            getLogger().error( "Unable to find address for local listen port.", ex );

            throw ExceptionTool.initCause( new org.omg.CORBA.INITIALIZE(
                    "Unable to find address for local listen port (" + ex + ")",
                    0, org.omg.CORBA.CompletionStatus.COMPLETED_NO), ex );
        }

    }

    /**
     * Start listening for incoming connections. Idempotent.
     *
     * @throws org.omg.CORBA.COMM_FAILURE If unable to listen. This will result
     *  in server shutdown.
     * @throws org.omg.CORBA.TRANSIENT If unable to listen, and try again later.
     */
    public void open()
    {
        if ( m_svr_socket != null )
            return;

        open_port();
    }

    private void open_port()
    {
        try
        {
            int port = ( m_primary_endpoint.port & 0xFFFF );
            m_svr_socket = new ServerSocket( port, m_backlogQueueLength, m_host );
        }
        catch ( final IOException ex )
        {
            getLogger().error( "Unable to listen on " + svrString() + " (" + ex + ").", ex );

            throw ExceptionTool.initCause( new org.omg.CORBA.COMM_FAILURE(
                    "Unable to listen on " + svrString() + " (" + ex + ")" ), ex );
        }
        catch ( final SecurityException ex )
        {
            getLogger().error( "Access denied for " + svrString() + ".", ex );

            throw ExceptionTool.initCause( new org.omg.CORBA.NO_PERMISSION(
                    "Access denied for " + svrString() + " (" + ex + ")" ), ex );
        }

        m_closed = false;

        if ( m_primary_endpoint.port == 0 )
            m_primary_endpoint.port = ( short ) m_svr_socket.getLocalPort();
    }

    /**
     * Get the primary endpoint published in the IIOP profile.
     */
    public ListenPoint getPrimaryEndpoint()
    {
        return m_primary_endpoint;
    }

    /**
     * Get the list of endpoints allowed for bidirectional use. These will
     * be transmitted in BI_DIR_IIOP service contexts. If empty or null then
     * bidirectional IIOP will be disabled.
     */
    public ListenPoint [] getBiDirEndpoints()
    {
        return m_bidir_endpoints;
    }


    /**
     * Get the listen host.
     */
    protected InetAddress getListenHost()
    {
        return m_host;
    }

    /**
     * Stop listening for a connection. Idempotent.
     */
    public void close()
    {
        if ( m_closed || m_svr_socket == null )
            return;

        m_closed = true;

        try
        {
            m_svr_socket.close();
        }
        catch ( IOException ex )
        {
            // TODO: ???
        }

        m_svr_socket = null;
    }

    /**
     * Is is the transport open?
     */
    public boolean isOpen()
    {
        return !m_closed;
    }

    /**
     * Listen for an incoming connection.
     *
     * @return transport for new connection, or null if no connection received.
     * @throws org.omg.CORBA.COMM_FAILURE If some permanent comms problem occours
     *  this will result in server shutdown.
     */
    public Transport accept(final int timeout) 
    {
        try 
        {
            if ( m_closed ) 
            {
                return null;
            }

            setSocketTimeout( timeout );

            try 
            {
                final Socket sock = m_svr_socket.accept();
    
                return new IIOPTransport( sock, m_primary_endpoint.port & 0xFFFF,
                        getLogger(), m_socketNoDelay, m_boostRecvPriority,
                        m_sendBufferSize, m_receiveBufferSize, 
                        m_bufferedOutputStreamSize );
    
            } 
            catch ( final InterruptedIOException ex ) 
            {
                // Solaris workaround
                if ( "operation interrupted".equals( ex.getMessage() ) ) 
                {
                    Thread.currentThread().interrupt();
                }
                
                return null;
    
            } 
            catch ( final SocketException ex ) 
            {
                // On AIX a SocketException is thrown when the accept() thread is interrupted
                return null;
            }
        } 
        catch ( final IOException ex ) 
        {
            if ( m_closed ) 
            {
                return null;
            }
            
            getLogger().error( "IOException during accept.", ex );
    
            throw ExceptionTool.initCause( new COMM_FAILURE(), ex );
        }
    }

    private void setSocketTimeout( int timeout ) throws SocketException
    {
        synchronized ( m_svr_socket ) 
        {
            timeout = ( ( m_minSocketTimeout < timeout ) 
                    && ( timeout < m_maxSocketTimeout ) ) ? timeout 
                    : m_overrideSocketTimeout;
                    
            if ( timeout == m_currentSocketTimeout ) 
            {
                // no change
                return;
            }
            
            m_svr_socket.setSoTimeout( timeout );
            m_currentSocketTimeout = timeout;
        }
    }

    public String toString()
    {
        return "(iiop) " + svrString();
    }

    protected String svrString()
    {
        return "" + m_host + ":" + ( m_primary_endpoint.port & 0xFFFF );
    }

    private Logger getLogger()
    {
        return m_logger;
    }
}

