// acs hack BEGIN
package org.jacorb.orb.acs;

import org.jacorb.poa.RequestProcessor;


/**
 * ACS callback interface hacked into JacORB code, to allow advanced profiling and error reporting.
 * An implementing object must be injected via the ORB's {@link AcsProfilingORB} interface.
 * <p>
 * The purpose of using this interface is to abstract the ACS changes in JacORB code,
 * so that in the future we could implement similar changes in a different Java ORB.
 * However we do not yet follow this idea completely, as it would require us to define this interface
 * outside of JacORB code, and in a package like "acs.corba" to be neutral of JacORB.
 * This is too cumbersome to implement in the ACS/ExtProd/ structure so that we accept 
 * the small porting effort to move this file and change package names when we use a different Java ORB for profiling.
 */
public interface AcsORBProfiler
{
	/**
	 * Notification when a request could not be delivered to the request queue.
	 * 
	 * @param messageSize  size of the undelivered GIOP message
	 * @param poaName  Simple name of the target POA (that typically owns the request queue) 
	 *           if the POA could be identified; 
	 *           otherwise null
	 * @param operation  name of the target operation, or null if not known (may happen very early in the message processing)
	 * @param causedByQueueFull  true means that the failure was caused by a full request queue; 
	 *           otherwise it was caused by some other failure (e.g. received a request with a non-jacorb object key) 
	 *           which does then not exclude that the queue was also full.
	 */
	public void undeliveredRequest(int messageSize, String poaName, String operation, boolean causedByQueueFull);
	
	/**
	 * Notification about an incoming request being added to or removed from a request queue.
	 * More than this event in itself, the interesting aspect is how full the queue gets.
	 * As the queue is constantly being drained by the request handler threads from the thread pool,
	 * there will always be some fluctuations in the queue length, but considerable queue sizes 
	 * are only expected in the case where all worker threads are busy.
	 * <p>
	 * This design of getting notified about request queue changes, as opposed to getting a reference to the queue, 
	 * helps in avoiding memory leaks as the profiler cannot hold on to objects associated with a POA when the POA should be unloaded.
	 * The disadvantage is that the profiler would need to store a map of POA names and their associated request queue lengths
	 * in order to serve an external API that offers inquiries at any time.
	 * 
	 * @param requestId  The ORB's request ID
	 * @param poaName  Simple name of the POA that owns the queue, 
	 *                 or {@code null} if an ORB-wide queue is used or the POA name is not available.
	 * @param queueSize  current size of the request queue
	 * @param maxQueueLength  configured maximum request queue length, to be used for computing percentages
	 */
	public void requestQueueSizeChanged(int requestId, String poaName, int queueSize, int maxQueueLength);
	
	/**
	 * Notification about a thread being added to or removed from the pool of request processor threads.
	 * More than this event in itself, the interesting aspect is how full the pool gets.
	 * <p>
	 * This design of getting notified about thread pool changes, as opposed to getting a reference to the thread pools, 
	 * helps in avoiding memory leaks as the profiler cannot hold on to objects associated with a POA when the POA should be unloaded.
	 * The disadvantage is that the profiler would need to store a map of POA names and their associated thread pool information
	 * in order to serve an external API that offers inquiries at any time.
	 * 
	 * @param poaName  Simple name of the POA that owns the thread pool, 
	 *                 or {@code null} if an ORB-wide thread pool is used or if the poa name is not available (which is currently the case with JacORB)
	 * @param idleThreads  number of idle threads, ready to serve requests
	 * @param totalThreads  number of threads (idle or busy)
	 * @param maxThreads  configured maximum thread size, to be used for computing percentages
	 * @see org.jacorb.poa.RPPoolManagerListener#processorAddedToPool(RequestProcessor, int, int)
	 * @see org.jacorb.poa.RPPoolManagerListener#processorRemovedFromPool(RequestProcessor, int, int)
	 */
	public void threadPoolSizeChanged(String poaName, int idleThreads, int totalThreads, int maxThreads);
	
	/**
	 * Called when a Corba request is about to be run.
	 * @param requestId  The ORB's request ID
	 * @param poaName  Simple name of the POA, or {@code null} if not known.
	 * @param operation
	 * @see #requestFinished(int, String, String)
	 */
	public void requestStarted(int requestId, String poaName, String operation);
	
	/**
	 * This method is called when a request has finished (success or error).
	 * It will be called only from the same thread as {@link #requestStarted(int, String, String)}.
	 * 
	 * @param requestId  The ORB's request ID
	 * @param poaName  Simple name of the POA, or {@code null} if not known.
	 * @param operation
	 */
	public void requestFinished(int requestId, String poaName, String operation);

}
//acs hack END
