/*
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 */

package alma.archive.wrappers;

import java.net.URI;
import java.net.URISyntaxException;

import alma.archive.exceptions.identifier.GlobalRangeExceededException;
import alma.archive.exceptions.identifier.LocalRangeExceededException;
import alma.archive.exceptions.identifier.UIDSyntaxException;

/**
 * @author simon
 * 
 * Uid.java Created by simon on 07-Nov-2002
 * 
 */
public class UniqueIdentifier 
{
	//TODO change this to have range checking
	private long globalIdentifier = 0;
	private long localIdentifier = 0;
	
	private final String protocol = "uid";
	private final String seperator = "://" ;
	
	private final int globalIdentifierLength = 16;
	private final int localIdentifierLength = 8;
	private final long localIdentifierMAX = 
		(long)Math.pow(16,localIdentifierLength);
		
	/**
	 * Constructor for Uid.
	 */
	public UniqueIdentifier(){}
	
	public UniqueIdentifier(long globalIdentifier, long localIdentifier)
	{
		this.globalIdentifier = globalIdentifier;
		this.localIdentifier = localIdentifier;
	}
	
	public UniqueIdentifier(String id) throws UIDSyntaxException
	{
		if (!parse(id,false)){
			throw new UIDSyntaxException("Invalid UID: " + id);
		}
	}
	
	public UniqueIdentifier(URI uri) throws UIDSyntaxException
	{
		String id = uri.toASCIIString();
		if (!parse(id,false)){
			throw new UIDSyntaxException("Invalid UID: " + id);	
		}
	}
	
	/**
	 * Increments the global section of the Identifier
	 * WARNING: this is only to be used by the Archive
	 * there is NO reason that it should be used by any
	 * other subsystem
	 * @throws GlobalRangeExceededException
	 */
	public void incrementGlobal() throws GlobalRangeExceededException
	{
		globalIdentifier++;
		if (globalIdentifier >= Long.MAX_VALUE){
			throw new GlobalRangeExceededException(
				"The maximum value for a local identifier is " + 
				"2^64-1 Long.MAX_VALUE");
		}
	}
	
	/**
	 * Increments the local section of the Identifier
	 * WARNING: this should be used with care new identifiers
	 * should be obtained from the Container or Activator
	 * @throws LocalRangeExceededException
	 */
	public void incrementLocal() throws LocalRangeExceededException
	{
		localIdentifier++;
		if (localIdentifier >= localIdentifierMAX){
			throw new LocalRangeExceededException(
				"The maximum value for a local identifier is 16^8-1");
		}
	}
	
	/**
	 * Parses a String version of the @param uid and returns
	 * @return a boolean relating to it's validity. Will be updated
	 * in the future to check against schemas.
	 */
	public boolean parse(String id, boolean plain)
	{
		String idstring = id;
		if (!plain)
		{
			int divider = id.indexOf(seperator);
			if (divider == -1) return false;
			if (id.indexOf(protocol) == -1) return false;
			idstring = id.substring(divider+seperator.length(),id.length());
			
			if (idstring.indexOf("/") == -1) return false;
		}
		
		// find the two X's
		int firstX = id.indexOf("X");
		if (firstX == -1) return false;
		int secondX = id.indexOf("X",firstX+1);
		if (secondX == -1) return false;
		
		String global = id.substring(firstX+1,firstX+1+globalIdentifierLength);
		String local = id.substring(secondX+1,id.length());

		globalIdentifier = Long.parseLong(global, 16);
		localIdentifier = Long.parseLong(local,16);
		
		return true;
	}
	
	/** Returns and XML representation of the UID for use with XML DB
	 */
	public String toXML()
	{	
		return "<"+ protocol + ">" + toString() + "</" + protocol + ">";
	}
	
	/** Extracts values from XML string 
	 * @throws UIDSyntaxException*/
	// need to extend this to use jdom
	public void fromXML(String xmlid) throws UIDSyntaxException
	{
		if (!parse(extractXML(protocol,xmlid),false)){
			throw new UIDSyntaxException("Invalid UID: " + xmlid);
		}
	}
	
	private String extractXML(String tag, String xmlid) 
		throws 
			UIDSyntaxException
	{
	   String starttag = "<" + tag + ">";
	   String endtag = "</" + tag + ">";
	   String id = "";
	   int start = xmlid.indexOf(starttag);
	   int end = xmlid.indexOf(endtag);
	   if ((start != -1) && (end != -1)){
	   		return xmlid.substring(start+starttag.length(),end);
	   }
	   else{
	   		throw new UIDSyntaxException("Invalid UID: " + id);
	   }
	}
	
	/**
	 * Returns a string that does not contain a "/" character
	 * Xindice does not tolerate "/"'s in keys useful for maps
	 */
	public String toStringPlain()
	{
		String globalString = padString(Long.toString(globalIdentifier, 16),16);
		String localString = padString(Long.toString(localIdentifier, 16),8);
		String plain = "X" + globalString + "X" + localString;
		return plain;
	}
	
	/** Returns a string version of the UID */
	public String toString()
	{
		String globalString = padString(Long.toString(globalIdentifier, 16),16);
		String localString = padString(Long.toString(localIdentifier, 16),8);
		String uid = protocol + seperator + "X" + globalString + "/X" + localString;
		return uid;
	}
	
	public URI toURI()
	{
		try{
			return new URI(toString());
		}
		catch (URISyntaxException e){
			return null;
			//MPA: no logging, no action. correct?
		}
	}
	
	
	private String padString(String input, int length)
	{
		String local = input;
		for (int x = input.length();x<length;x++)
		{
			local = "0" + local;
		}
		return local;
	}
	
	/**
	 * Returns the globalIdentifier.
	 * @return long
	 */
	public long getGlobalIdentifier()
	{
		return globalIdentifier;
	}

	/**
	 * Returns the localIdentifier.
	 * @return long
	 */
	public long getLocalIdentifier()
	{
		return localIdentifier;
	}
	
	public String getScheme()
	{
		return protocol;
	}

	public Object clone()
	{
		return new UniqueIdentifier(globalIdentifier,localIdentifier);
	}
	/**
	 * Sets the globalIdentifier.
	 * @param globalIdentifier The globalIdentifier to set
	 */
	/*
	public void setGlobalIdentifier(long globalIdentifier)
	{
		this.globalIdentifier= globalIdentifier;
	}
	*/

	/**
	 * Sets the localIdentifier.
	 * @param localIdentifier The localIdentifier to set
	 */
	/*
	public void setLocalIdentifier(long localIdentifier)
	{
		this.localIdentifier= localIdentifier;
	}
	*/
	
	public boolean equals(UniqueIdentifier uid)
	{
		if ( (this.globalIdentifier == uid.globalIdentifier) && (this.localIdentifier == uid.localIdentifier)) return true;
		else return false;
	}
}
