/*
 * 	  Created on 16-Sep-2003
 * 
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 */
package alma.archive.database.xmldb;

import java.io.IOException;
import java.io.StringReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.logging.Logger;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;
import org.jdom.xpath.XPath;
import org.xmldb.api.base.XMLDBException;

import alma.archive.database.interfaces.UserManager;
import alma.archive.exceptions.ArchiveException;
import alma.archive.exceptions.general.DatabaseException;
import alma.archive.exceptions.user.RoleAlreadyExistsException;
import alma.archive.exceptions.user.RoleDoesNotExistException;
import alma.archive.exceptions.user.RoleNotAssignedException;
import alma.archive.exceptions.user.UserAlreadyExistsException;
import alma.archive.exceptions.user.UserDoesNotExistException;

/**
 * @author simon
 */
public class XmldbUserManager implements UserManager
{
	private static XmldbUserManager _instance = null;
	private static int access_count = 0; 
	
	public static XmldbUserManager getInstance(Logger logger) throws DatabaseException
	{
		if (_instance == null) _instance = new XmldbUserManager(logger);
		access_count++;
		return _instance;
	}
	
	public void close() throws DatabaseException
	{
		access_count--;
		if (access_count == 0)
		{
			database.close();
			_instance = null;
			
		}
	}
	
	
	private Logger logger;
	//MPA: logger is never read locally
	
	private XmldbDatabase database;
	
	private String location = "archive://users";
	private URI locationUri;
	
	private Document doc = null;
	
	private Element root = null;
	private Element roles = null;
	private Element users = null;
	
	/*
	 * <usermanager>
	 * 	<roles>
	 * 		<role name=""/>
	 * 		...
	 * 	</roles>
	 * 	<users>
	 * 		<user name="" role=""/>
	 * 		...
	 * 	</users>
	 * </usermanager>
	 */
	
	private XmldbUserManager(Logger logger) throws DatabaseException
	{
		try {locationUri = new URI(location);}
		catch (URISyntaxException e1) {e1.printStackTrace();}
        //MPA: no logging, no delaing. correct?
		
		this.logger = logger; 
		try
		{
			database = XmldbDatabase.instance(logger);			// Grab a database
			
			if (database.exists(locationUri))				// Load the namespaces
			{
				SAXBuilder builder = new SAXBuilder();
				builder.setIgnoringElementContentWhitespace(true);
				try
				{
					doc = builder.build(new StringReader(database.getlatest(locationUri)));
					root = doc.getRootElement();
					
					roles = root.getChild("roles");
					users = root.getChild("users");
				}
				catch (JDOMException e) {throw new DatabaseException(e.getMessage());}
				catch (IOException e) {throw new DatabaseException(e.getMessage());}	
				//MPA: in both cases loggin missing
			}
			else
			{
				root = new Element("usermanager");
				roles = new Element("roles");
				users = new Element("users");
				root.addContent(roles);
				root.addContent(users);
				doc = new Document(root);
			}
		}
		catch (XMLDBException e)
		{
			throw new DatabaseException(e.getMessage());
			//MPA: logging missing
		}
	}
	
	public void addUser(String user) 
		throws 
			DatabaseException, 
			ArchiveException, 
			UserAlreadyExistsException
	{
	    //MPA: the esxception UserAlreadyExistsException is never thrown
		Element u = new Element("user");
		u.setAttribute("name",user);
		users.addContent(u);
		
		save();
	}


	public void deleteUser(String user) 
		throws 
			DatabaseException, 
			ArchiveException, 
			UserDoesNotExistException
	{
		try
		{
			String _path = "//user[@name=\"" + user + "\"]";
			XPath path = XPath.newInstance(_path);
			Element result = (Element)path.selectSingleNode(users);
			if (result == null) throw new UserDoesNotExistException();
			else
			{
				users.removeContent(result);
				
				save();	
			}
		}
		catch (JDOMException e)
		{
			throw new DatabaseException(e.getMessage());
			//MPA: logging missing
		}
	}


	public void addRole(String role) 
		throws 
			DatabaseException, 
			ArchiveException, 
			RoleAlreadyExistsException
	{
		Element r = new Element("role");
		r.setAttribute("name",role);
		roles.addContent(r);
		
		save();
	}


	public void deleteRole(String role) 
		throws 
			DatabaseException, 
			ArchiveException, 
			RoleDoesNotExistException
	{
		try
		{
			String _path = "//role[@name=\"" + role + "\"]";
			XPath path = XPath.newInstance(_path);
			Element result = (Element)path.selectSingleNode(roles);
			if (result == null) throw new RoleDoesNotExistException();
			else
			{
				roles.removeContent(result);
				
				save();	
			}
		}
		catch (JDOMException e)
		{
			throw new DatabaseException(e.getMessage());
			//MPA: logging missing
		}
	}


	public void assignRole(String user, String role)
		throws 
			DatabaseException, 
			ArchiveException, 
			RoleDoesNotExistException,
			UserDoesNotExistException
	{
	    //MPA: exception serDoesNotExistException is never thrown
		try
		{
			// Make sure the role exists
			String _path = "//role[@name=\"" + role + "\"]";
			XPath path = XPath.newInstance(_path);
			Element result = (Element)path.selectSingleNode(root);
			if (result == null) throw new RoleDoesNotExistException();
			
			// Go add it to the users
			_path = "//user[@name=\"" + user + "\"]";
			path = XPath.newInstance(_path);
			result = (Element)path.selectSingleNode(root);
			if (result == null) throw new RoleDoesNotExistException();
			else
			{
				Element r = new Element("role");
				r.setAttribute("name",role);
				result.addContent(r);
				
				save();
			}
		}
		catch (JDOMException e)
		{
			throw new DatabaseException(e.getMessage());
			//MPA: logging missing
		}
	}


	public void withdrawRole(String user, String role)
		throws
			DatabaseException,
			ArchiveException,
			RoleDoesNotExistException,
			UserDoesNotExistException,
			RoleNotAssignedException
	{
	    //MPA: exceptions RoleDoesNotExistException, RoleNotAssignedException are never thrown
		try
		{		
			String _path = "//user[@name=\"" + user + "\"]/role[@name=\"" + role + "\"]";
			XPath path = XPath.newInstance(_path);
			Element result = (Element)path.selectSingleNode(root);
			if (result == null) throw new UserDoesNotExistException();
			else
			{
				// Holger: adapted to jdom 1.0 beta 10 cast necessary
				Element parent = (Element) result.getParent();
				parent.removeContent(result);
				
				save();
			}
			
		}
		catch (JDOMException e)
		{
			throw new DatabaseException(e.getMessage());
			//MPA: no logging
		}
	}
	
	public String getRole(String user) throws DatabaseException, UserDoesNotExistException
	{
		try
		{		
			String _path = "//user[@name=\"" + user + "\"]/role";
			XPath path = XPath.newInstance(_path);
			Element result = (Element)path.selectSingleNode(root);
			if (result == null) throw new UserDoesNotExistException();
			else
			{
				return result.getAttributeValue("name");
			}
		}
		catch (JDOMException e)
		{
			throw new DatabaseException(e.getMessage());
			//MPA: no logging
		}
	}
	
	private void save()
		throws
			DatabaseException
	{
		//System.out.println(toXmlString(doc));
		database.putLatest(locationUri,toXmlString(doc));
	}
	
	private String toXmlString(Document doc)
	{

		XMLOutputter out = new XMLOutputter(Format.getPrettyFormat());
		// XMLOutputter out = new XMLOutputter("  ",true,"UTF-8");
		return out.outputString(doc);
	}
	
	
	public static void main(String[] args) throws DatabaseException, RoleAlreadyExistsException, ArchiveException
	{
		Logger logger = Logger.getAnonymousLogger();
		XmldbUserManager man = XmldbUserManager.getInstance(logger);
		
		man.addRole("role");
		man.addUser("user");
		
		man.assignRole("user","role");
		man.withdrawRole("user","role");
		
		man.deleteUser("user");
		man.deleteRole("role");
	}
}
