/*
 * 	  Created on 17-Sep-2003
 * 
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 */
package alma.archive.database.xmldb;

import java.io.IOException;
import java.io.StringReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.logging.Logger;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;
import org.jdom.xpath.XPath;
import org.xmldb.api.base.XMLDBException;

import alma.archive.database.helpers.Cache;
import alma.archive.database.interfaces.SchemaManager;
import alma.archive.exceptions.ArchiveException;
import alma.archive.exceptions.access.PermissionDeniedException;
import alma.archive.exceptions.general.DatabaseException;
import alma.archive.exceptions.general.NamespaceDefinedException;
import alma.archive.exceptions.general.UnknownSchemaException;
import alma.archive.exceptions.syntax.MalformedURIException;
import alma.archive.exceptions.syntax.MalformedXMLException;
import alma.archive.exceptions.user.UserDoesNotExistException;
import alma.archive.wrappers.Permissions;

/**
 * @author simon
 * 
 * This is the XML:DB implementation of the SchemaManager. All of the
 * structure that is created by with the creation of documents is stored
 * as XML, the format is given below:
 * 
 * \<namespaces\>
 * 	\<namespace name="" URI=""/\>
 * 	...
 * \</namespaces\>
 * 
 * \<schemas\>
 * 	\<schema uri="" name="" previousUri=""\>
 * 		\<namespace name="" URI=""/\>
 * 		...
 * 	\</schema\>
 *  ...
 * \</schemas\>
 * 
 * The list of namespaces is stored in a separate document for convenience.
 */
public class XmldbSchemaManager implements SchemaManager
{
	private XmldbDatabase database;
	private Logger logger;
	
	private String _schemas = "archive://schemas";
	private String _namespaces = "archive://namespaces";
	private String _testns = "uid://test";
	private String _schemans = "http://www.w3.org/2001/XMLSchema";
	private URI schemasUri,namespacesUri,testns,schemans;
	
	private final int cacheSize = 100;
	private Cache namespaceCache = null;
	
	private static XmldbSchemaManager _instance = null;
	private static int access_count = 0;
	
	/**
	 * Returns a pointer to the instance of SchemaManager, create the
	 * instance if necessary.
	 * @param logger
	 * @return
	 * @throws DatabaseException
	 */
	protected static XmldbSchemaManager instance(Logger logger)
		throws 
			DatabaseException
	{
		if (_instance == null) 
		{
			_instance = new XmldbSchemaManager(logger);
		}
		access_count++;
		return _instance;
	}
	
	/**
	 * Close the SchemaManager to free resource, it is vital that this
	 * is called.
	 */
	public void close() 
		throws 
			DatabaseException
	{
		access_count--;
		if (access_count == 0)
		{
			
			database.close();
			_instance = null;
			
		}
	}
	 
	private Document nsdoc = null;
	private Document scdoc = null;
	
	private Element namespaces = null; 
	private Element schemas = null;
	
	/**
	 * Create an instance ofthe schema manager
	 * @param logger
	 * @throws DatabaseException
	 */
	private XmldbSchemaManager(Logger logger) 
		throws 
			DatabaseException
	{	
		
		this.logger = logger;
		try 
		{
			schemasUri = new URI(_schemas);
			namespacesUri = new URI(_namespaces);
			testns = new URI(_testns);
			schemans = new URI(_schemans);
		}
		catch (URISyntaxException e) 
		{
			logger.severe(e.getLocalizedMessage());
			throw new DatabaseException(e);
		}
		
		try
		{
			//Get access to the database
			database = XmldbDatabase.instance(logger);
			build();
		}
		catch (XMLDBException e)
		{
			logger.severe(e.getLocalizedMessage());
			throw new DatabaseException(e);
		}
	}
	
	/**
	 * Rebuilds all of the chached schema information from the database.
	 */
	public void refresh() 
		throws 
			DatabaseException
	{
		database.refresh();
		nsdoc=null;
		scdoc=null;
		namespaces=null;
		schemas=null;
		build();
	}
	
	/**
	 * Build the schema information from the database.
	 * @throws DatabaseException
	 */
	private void build() 
		throws 
			DatabaseException
	{	
		logger.info("Getting schema information from Xmldb.");
		//Load the namespaces
		if (database.exists(namespacesUri))
		{
			SAXBuilder builder = new SAXBuilder();
			builder.setIgnoringElementContentWhitespace(true);
			try
			{
				database.flushLatest(namespacesUri);
				nsdoc = builder.build(
					new StringReader(database.getlatest(namespacesUri)));
				namespaces = nsdoc.getRootElement();
			}
			catch (JDOMException e) {
				logger.severe(e.getLocalizedMessage());
				throw new DatabaseException(e);
			}
			catch (IOException e) {
				logger.severe(e.getLocalizedMessage());
				throw new DatabaseException(e);
			}	
		}
		else
		{	//if they aren't there create the structure
			namespaces = new Element("namespaces");
			nsdoc = new Document(namespaces);
		}
		
		//Load the schema information
		if (database.exists(schemasUri))
		{
			SAXBuilder builder = new SAXBuilder();
			builder.setIgnoringElementContentWhitespace(true);
			try
			{
				database.flushLatest(schemasUri);
				scdoc = builder.build(
					new StringReader(database.getlatest(schemasUri)));
				schemas = scdoc.getRootElement();
			}
			catch (JDOMException e) {
				logger.severe(e.getLocalizedMessage());
				throw new DatabaseException(e);
			}
			catch (IOException e) {
				logger.severe(e.getLocalizedMessage());
				throw new DatabaseException(e);
			}
		}
		else
		{	//if they aren't there create the structure
			schemas = new Element("schemas");
			scdoc = new Document(schemas);
		}
		
		//create a cache for namespaces
		namespaceCache = new Cache(cacheSize);
	}

	/**
	 * @see alma.archive.database.interfaces.SchemaManager
	 */
	public void registerNamespace(String tag, URI namespace)
		throws
			DatabaseException, 
			MalformedURIException, 
			NamespaceDefinedException
	{
		//throw an exception if the tag (prefix) already exists
		try
		{
			XPath path = XPath.newInstance(
				"/namespaces/namespace[@name=\"" + 
				tag + 
				"\"]");
			
			Element result = (Element)path.selectSingleNode(namespaces);
			if (result != null)
			{
				String existingUri = result.getAttributeValue("URI");
				if (!existingUri.equalsIgnoreCase(namespace.toASCIIString()))
				{
					throw new NamespaceDefinedException(
						"Namespace: "
						+ tag + 
						" is already defined as: " 
						+ existingUri);
				}
			}
			
			Element ns = new Element("namespace");
			ns.setAttribute("name",tag);
			ns.setAttribute("URI",namespace.toASCIIString());
			namespaces.addContent(ns);
			
			save();
		}
		catch (JDOMException e){
			logger.severe(e.getLocalizedMessage());
			throw new DatabaseException(e);
		}	
	}
	
	/**
	 * @see alma.archive.database.interfaces.SchemaManager
	 */
	public boolean namespaceExists(URI namespace) 
		throws 
			DatabaseException
	{
		if (namespace.equals(testns)) return true;
		try
		{
			XPath path = XPath.newInstance(
				"/namespaces/namespace[@URI=\"" + 
				namespace.toASCIIString() + 
				"\"]");
			
			Element result = (Element)path.selectSingleNode(namespaces);
			if (result != null) return true;
			else return false;
		}
		catch (JDOMException e){
			logger.severe(e.getLocalizedMessage());
			throw new DatabaseException(e);
		}
	}
	
	/**
	 * @see alma.archive.database.interfaces.SchemaManager
	 */
	public void removeNamespace(String name)
		throws
			DatabaseException
	{
		try
		{
			//First remove from the namespaces element
			XPath path = XPath.newInstance(
				"/namespaces/namespace[@name=\"" + 
				name + 
				"\"]");
			
			Element result = (Element)path.selectSingleNode(namespaces);
			namespaces.removeContent(result);
			
			//Then remove from all of the schema elements
			if (schemas.getChildren().size()>0)
			{
				path = XPath.newInstance(
					"/schemas/schema/namespace[@name=\"" + 
					name + 
					"\"]");
				
				List ns = path.selectNodes(schemas);
				ListIterator iter = ns.listIterator();
				while (iter.hasNext())
				{
					Element n = (Element)iter.next();
					Element parent = (Element) n.getParent();
					URI schemaUri = new URI(parent.getAttributeValue("URI"));
					parent.removeContent(n);
					if (namespaceCache.containsKey(schemaUri))
					{
						namespaceCache.remove(schemaUri);
					}
				}
			}
			save();
		}
		catch (JDOMException e){
			logger.severe(e.getLocalizedMessage());
			throw new DatabaseException(e);
		} 
		catch (URISyntaxException e){
			logger.severe(e.getLocalizedMessage());
			throw new DatabaseException(e);
		}
	}
	
	/**
	 * @see alma.archive.database.interfaces.SchemaManager
	 */
	public void removeAll()
		throws
			DatabaseException
	{
		/*//Find all of the URI's and remove them
		try
		{
			String _path = "/schemas/schema";
			XPath schemaPath = XPath.newInstance(_path);
			List results = schemaPath.selectNodes(schemas);
			if (results.size() != 0)
			{
				Iterator iter = results.iterator();
				while (iter.hasNext())
				{
					//Extract the URI for each schema and remove from the db
					Element res = (Element)iter.next();
					URI uri = new URI (res.getAttributeValue("URI"));
					
					if (database.exists(uri))
					{
						database.delDescriptor(uri);
						database.delLatest(uri);
					}				
				}
			}
		}
		catch (JDOMException e){
			logger.severe(e.getLocalizedMessage());
			throw new DatabaseException(e);
		} 
		catch (URISyntaxException e){
			logger.severe(e.getLocalizedMessage());
			throw new DatabaseException(e);
		}
		
		namespaces = new Element("namespaces");
		nsdoc = new Document(namespaces);
		
		schemas = new Element("schemas");
		scdoc = new Document(schemas);
		
		save();*/
		
			// remove all collection:
			XmldbConnector conn = null;
			try {
				conn = XmldbConnector.instance(logger, database.dbConfig.testMode);
			} catch (DatabaseException e1) {
				// TODO Auto-generated catch block
				e1.printStackTrace();
			} catch (XMLDBException e1) {
				// TODO Auto-generated catch block
				e1.printStackTrace();
			}
			conn.cleanCollections();
			refresh();
			
		try {
			((XmldbInternalIF) XmldbInternalIF.instance(logger)).loadIdentifierRangeSchema("xmldbSchemaManager");
		} catch (ArchiveException e) {
			throw new DatabaseException(e);
		}
	}


	/**
	 * @see alma.archive.database.interfaces.SchemaManager
	 */
	public void assignNamespace(String name, URI schemaUri)
		throws 
			DatabaseException, 
			UnknownSchemaException
	{
		try
		{
			String _path = 
				"/schemas/schema[@URI=\"" + 
				schemaUri.toASCIIString() + 
				"\"]";
			
			XPath schemaPath = XPath.newInstance(_path);
			Element schemaResult = 
				(Element)schemaPath.selectSingleNode(schemas);
			
			if (schemaResult == null) throw new UnknownSchemaException();
			else
			{
				XPath nspath = XPath.newInstance(
					"/namespaces/namespace[@name=\"" + 
					name + 
					"\"]");
				
				Element nsresult = (Element)nspath.selectSingleNode(namespaces);
				schemaResult.addContent((Element)nsresult.clone());
				/*
				 * The namespace for this schema could have changed to remove
				 * from the cache
				 */
				if (namespaceCache.containsKey(schemaUri))
				{
					namespaceCache.remove(schemaUri);
				}
				save();
			}
		}
		catch (JDOMException e){
			logger.severe(e.getLocalizedMessage());
			throw new DatabaseException(e);
		}
	}
	
	/**
	 * @see alma.archive.database.interfaces.SchemaManager
	 */
	public void withdrawNamespace(String name, URI schemaUri)
		throws
			DatabaseException,
			UnknownSchemaException
	{
		try
		{
			String _path = "/schemas/schema[@URI=\"" 
			             + schemaUri.toASCIIString() 
			             + "\"]/namespace[@name=\"" 
			             + name 
			             + "\"]";
			XPath path = XPath.newInstance(_path);
			Element result = (Element)path.selectSingleNode(schemas);
			if (result == null) throw new UnknownSchemaException(schemaUri.toString()+" does not contain namespace prefix "+name);
			else
			{
				Element parent = (Element) result.getParent();
				parent.removeContent(result);
				/*
				 * The namespace for this schema could have changed to remove
				 * from the cache
				 */
				if (namespaceCache.containsKey(schemaUri))
				{
					namespaceCache.remove(schemaUri);
				}
				save();
			}
		}
		catch (JDOMException e)
		{
			logger.severe(e.getLocalizedMessage());
			throw new DatabaseException(e);
		}
	}
	
	/**
	 * @see alma.archive.database.interfaces.SchemaManager
	 */
	public HashMap namespaces()
	{
		HashMap results = new HashMap();
		List list = namespaces.getChildren();
		Iterator iter = list.iterator();
		while (iter.hasNext())
		{
			Element ns = (Element)iter.next();
			results.put(
				ns.getAttributeValue("name"),
				ns.getAttributeValue("URI"));
		}
		return results;
	}
	
	/**
	 * @see alma.archive.database.interfaces.SchemaManager
	 */
	public List listSchema()
	{
		List schemaList = schemas.getChildren();
		ListIterator iter = schemaList.listIterator();
		ArrayList results = new ArrayList();
		while (iter.hasNext())
		{
			Element schema = (Element)iter.next();
			String name = schema.getAttributeValue("name");
			results.add(name);
		}
		return results;
	}
	
	/**
	 * @see alma.archive.database.interfaces.SchemaManager
	 */
	private void checkValidXml(String xml)
		throws 
			MalformedXMLException
	{
		SAXBuilder builder = new SAXBuilder();
		builder.setIgnoringElementContentWhitespace(true);
		try
		{
			Document doc = builder.build(
				new StringReader(xml));
		}
		catch (JDOMException e) {
			logger.severe(e.getLocalizedMessage());
			throw new MalformedXMLException(e.getMessage());
		}
		catch (IOException e) {
			logger.severe(e.getLocalizedMessage());
			throw new MalformedXMLException(e.getMessage());
		}
	}


	/**
	 * @see alma.archive.database.interfaces.SchemaManager
	 */
	public void addSchema(
			String schemaName, 
			String xml, 
			String indexConfig, 
			URI schemaURInew, 
			String owner,
			Permissions permissions)
				throws
					DatabaseException,
					UserDoesNotExistException,
					MalformedURIException,
					UnknownSchemaException,
					PermissionDeniedException,
					MalformedXMLException
	{
		checkValidXml(xml);
		// Create the element in the metadata
		Element schema = new Element("schema");
		schema.setAttribute("URI",schemaURInew.toASCIIString());
		schema.setAttribute("name",schemaName);
		
		// More comlplex metadata
		schema.setAttribute("version","1");
		schema.setAttribute("previous","");
		
		// Create an entry in the db
		if (database.exists(schemaURInew)) 
			throw new DatabaseException(
				"URI: " +
				schemaURInew.toASCIIString() +
				" already occupied");
		else
		{
			DocumentDataXml dd = new DocumentDataXml(
				schemaURInew,schemans,"Schema",owner,permissions);
			database.putDescriptor(schemaURInew,dd.toXmlString());
			database.putLatest(schemaURInew,xml);
			schemas.addContent(schema);
			save();
		}
	}
	
	/**
	 * @see alma.archive.database.interfaces.SchemaManager
	 */
	private int version(URI uri)
		throws
			DatabaseException
	{
		String _path = 
			"/schemas/schema[@URI=\"" + 
			uri.toASCIIString() + 
			"\"]";
		try
		{
			XPath path = XPath.newInstance(_path);
			Element result = (Element)path.selectSingleNode(schemas);
			if (result == null) return -1;
			else
			{
				return Integer.parseInt(result.getAttributeValue("version"));
			}
		}
		catch (JDOMException e)
		{
			logger.severe(e.getLocalizedMessage());
			throw new DatabaseException(e);
		}
	}

	/**
	 * @see alma.archive.database.interfaces.SchemaManager
	 */
	public void updateSchema(
			String schemaName, 
			String xml, 
			String indexConfig, 
			URI schemaURIold, 
			URI schemaURInew, 
			String owner, 
			Permissions permissions)
				throws
					DatabaseException,
					UserDoesNotExistException,
					MalformedURIException,
					UnknownSchemaException,
					PermissionDeniedException,
					MalformedXMLException
	{
		checkValidXml(xml);
		// Create the element in the metadata
		Element schema = new Element("schema");
		schema.setAttribute("URI",schemaURInew.toASCIIString());
		schema.setAttribute("name",schemaName);
		
		if (!database.exists(schemaURIold)) 
			throw new DatabaseException(
				"Previous URI: " + 
				schemaURIold.toASCIIString() +
				" does not exist");
		else if (database.exists(schemaURInew)) 
			throw new DatabaseException(
				"URI: " +
				schemaURInew.toASCIIString() +
				" already occupied");
		else
		{
			int pversion = version(schemaURIold);
			schema.setAttribute("version",Integer.toString(++pversion));
			schema.setAttribute("previous",schemaURIold.toASCIIString());
			
			//the schema name for schema is "Schema"
			DocumentDataXml dd = new DocumentDataXml(
				schemaURInew,schemans,"Schema",owner,permissions);
			database.putDescriptor(schemaURInew,dd.toXmlString());
			database.putLatest(schemaURInew,xml);
			
			schemas.addContent(schema);
			save();
		}
	}
	
	/**
	 * @see alma.archive.database.interfaces.SchemaManager
	 */
	public void removeSchema(String schemaName)
		throws
			DatabaseException,
			UserDoesNotExistException,
			UnknownSchemaException,
			PermissionDeniedException
	{
		try
		{
			String _path = 
				"/schemas/schema[@name=\"" + 
				schemaName + 
				"\"]";
			
			XPath schemaPath = XPath.newInstance(_path);
			List results = schemaPath.selectNodes(schemas);
			if (results.size() == 0) 
				throw new UnknownSchemaException(
					"Cannot find schema: " +
					schemaName +
					" to remeove");
			else
			{
				Iterator iter = results.iterator();
				while (iter.hasNext())
				{
					Element res = (Element)iter.next();
					URI uri = new URI (res.getAttributeValue("URI"));
					//remove from the cache and database
					if (namespaceCache.containsKey(uri))
					{
						namespaceCache.remove(uri);
					}
					database.delDescriptor(uri);
					database.delLatest(uri);
					//remove from the xml
					schemas.removeContent(res);				
				}
				save();
			}
		}
		catch (JDOMException e){
			logger.severe(e.getLocalizedMessage());
			throw new DatabaseException(e);
		}
		catch (URISyntaxException e){
			logger.severe(e.getLocalizedMessage());
			throw new DatabaseException(e);
		}
	}
	
	/**
	 * @see alma.archive.database.interfaces.SchemaManager
	 */
	public URI getSchemaURI(String schema) 
		throws 
			DatabaseException, 
			UnknownSchemaException
	{
		if (schema.equalsIgnoreCase("test")) return testns;
		
		try
		{
			String _path = 
				"/schemas/schema[@name=\"" + 
				schema + 
				"\"]";
			
			XPath path = XPath.newInstance(_path);
			List results = path.selectNodes(schemas);
			
			if (results.size() == 0) 
				throw new UnknownSchemaException(
					"Could not find scehma: " + 
					schema);
			else
			{
				// run through and find the most recent version
				Iterator iter = results.iterator();
				int version = -1;
				String uri = "";
				while (iter.hasNext())
				{
					Element result = (Element)iter.next();
					int tmpVersion = Integer.parseInt(
						result.getAttributeValue("version"));
					if (tmpVersion > version)
					{
						version = tmpVersion;
						uri = result.getAttributeValue("URI");
					}
				}
				return new URI(uri);
			}	
		}
		catch (JDOMException e){
			logger.severe(e.getLocalizedMessage());
			throw new DatabaseException(e);
		}
		catch (URISyntaxException e)
		{
			logger.severe(e.getLocalizedMessage());
			throw new DatabaseException(e);
		}
	}
	
	/**
	 * @see alma.archive.database.interfaces.SchemaManager
	 */
	public URI getSchemaURI(String schema, int version)
		throws 
			DatabaseException, 
			UnknownSchemaException
	{
		if (schema.equalsIgnoreCase("test")) return testns;
		try
		{
			String _path = "/schemas/schema[@name=\"" 
			               + schema 
			               + "\" and @version=\'"
			               + Integer.toString(version)
			               + "\']";
			XPath path = XPath.newInstance(_path);
			Element result = (Element)path.selectSingleNode(schemas);
			if (result == null) 
				throw new UnknownSchemaException(
					"Could not find scehma: " + 
					schema);
			else
			{
				return new URI(result.getAttributeValue("URI"));
			}
		}
		catch (JDOMException e){
			logger.severe(e.getLocalizedMessage());
			throw new DatabaseException(e);
		}
		catch (URISyntaxException e){
			logger.severe(e.getLocalizedMessage());
			throw new DatabaseException(e);
		}
	}

	/**
	 * @see alma.archive.database.interfaces.SchemaManager
	 */
	public HashMap getSchemaNamespaces(URI schemaUri)
		throws 
			DatabaseException 
	{	
		if (schemaUri.equals(testns)) return null;
		if (namespaceCache.containsKey(schemaUri))
		{
			return (HashMap)namespaceCache.get(schemaUri);
		}
		
		HashMap results = new HashMap();
		String _path = 
			"/schemas/schema[@URI=\"" + 
			schemaUri.toASCIIString() + 
			"\"]/namespace";
		// find the schema with a particular URI
		try
		{
			XPath path = XPath.newInstance(_path);
			List list = path.selectNodes(schemas);
			
			Iterator iter = list.iterator();
			while (iter.hasNext())
			{
				Element ns = (Element)iter.next();
				results.put(
					ns.getAttributeValue("name"),
					ns.getAttributeValue("URI"));
			}
			//Put the results in the cache for next time
			namespaceCache.put(schemaUri,results);
			return results;
		}
		catch (JDOMException e)
		{
			logger.severe(e.getLocalizedMessage());
			throw new DatabaseException(e);
		}
	}

	/**
	 * @see alma.archive.database.interfaces.SchemaManager
	 */
	public String getSchemaName(URI schemaUri)
		throws 
			DatabaseException, 
			UnknownSchemaException 
	{
		if (schemaUri.equals(testns)) return "test";
		if (schemaUri.equals(schemans)) return "Schema";
		
		String _path = 
			"/schemas/schema[@URI=\"" + 
			schemaUri.toASCIIString() + 
			"\"]";
		try
		{
			XPath path = XPath.newInstance(_path);
			Element result = (Element)path.selectSingleNode(schemas);
			if (result == null) 
				throw new UnknownSchemaException(
					"No schema is registered with the URI: " +
					schemaUri.toASCIIString() + " ");
			else
			{
				return result.getAttributeValue("name");
			}
		}
		catch (JDOMException e)
		{
			logger.severe(e.getLocalizedMessage());
			throw new DatabaseException(e);
		}	
	}
	
	/**
	 * @see alma.archive.database.interfaces.SchemaManager
	 */
	protected boolean verifySchema(URI schemaUri, String schemaName) 
		throws 
			DatabaseException, 
			UnknownSchemaException, 
			MalformedURIException
	{
		if ((schemaUri.equals(testns)) && schemaName.equals("test")) return true;
		if ((schemaUri.equals(schemans)) && schemaName.equals("Schema")) return true;
		
		String name = getSchemaName(schemaUri);
		if (!name.equals(schemaName)) 
			throw new UnknownSchemaException(
				"As scehma is registered at the URL: " +
				schemaUri.toASCIIString() + 
				" But it does not have the name: " +
				schemaName);
		return true;
	}
	
	/**
	 * @see alma.archive.database.interfaces.SchemaManager
	 */
	public int getSchemaVersion(URI schemaUri)
		throws
			DatabaseException,
			UnknownSchemaException,
			MalformedURIException
	{
		if (schemaUri.equals(testns)) return 0;
		return version(schemaUri);
	}
	
	/**
	 * Store the schema information in the database
	 * @throws DatabaseException
	 */
	private void save()
		throws
			DatabaseException
	{
		database.putLatest(namespacesUri,toXmlString(nsdoc));
		database.putLatest(schemasUri,toXmlString(scdoc));
		//System.out.println("namespaces --------------");
		//System.out.println(toPrettyXmlString(nsdoc));
		//System.out.println("schemas -----------------");
		//System.out.println(toPrettyXmlString(scdoc));
	}
	
	/**
	 * Print out the schema information, strictly for debug
	 *
	 */
	public void print()
	{
		System.out.println("namespaces --------------");
		System.out.println(toPrettyXmlString(nsdoc));
		System.out.println("schemas -----------------");
		System.out.println(toPrettyXmlString(scdoc));
	}
	
	/**
	 * Output the schema information as an XML string
	 * @param doc
	 * @return
	 */
	private String toXmlString(Document doc)
	{
		XMLOutputter out = new XMLOutputter(Format.getPrettyFormat());
		// XMLOutputter out = new XMLOutputter("",false,"UTF-8");
		return out.outputString(doc);
	}
	
	/**
	 * Output the schema information as an XML string
	 * @param doc
	 * @return
	 */
	private String toPrettyXmlString(Document doc)
	{
		XMLOutputter out = new XMLOutputter(Format.getRawFormat());
		// XMLOutputter out = new XMLOutputter("  ",true,"UTF-8");
		return out.outputString(doc);
	}
}