/*
 * 	  Created on 16-Sep-2003
 * 
 *    ALMA - Atacama Large Millimiter Array
 *    (c) European Southern Observatory, 2002
 *    Copyright by ESO (in the framework of the ALMA collaboration),
 *    All rights reserved
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2.1 of the License, or (at your option) any later version.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 *    MA 02111-1307  USA
 */
package alma.archive.database.xmldb;

import java.io.IOException;
import java.io.StringReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.logging.Logger;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.output.XMLOutputter;
import org.xmldb.api.base.XMLDBException;

import alma.archive.database.interfaces.IdentifierManager;
import alma.archive.database.oracle.DBConfig;
import alma.archive.exceptions.ModuleCriticalException;
import alma.archive.exceptions.general.DatabaseException;

/**
 * @author simon
 * 
 * This class controls access to the Unique Identifiers (UIDs), the last
 * UID used is stored in a location in the database. Whenever a UID is
 * returned the global namespace is incremented.
 */
public class XmldbIdentifierManager extends IdentifierManager
{	
	private String location = "archive://identifier";
	private URI locationUri;
	
	private XmldbDatabase database;
	private static XmldbIdentifierManager _instance;
	private static int access_count = 0;
	
	private Logger logger = null;
	
	/**
	 * Returns an instance of the the identifier manager, creates the
	 * the class if necessary.
	 * @param logger
	 * @return
	 * @throws DatabaseException
	 */
	public static XmldbIdentifierManager instance(Logger logger) 
		throws 
			DatabaseException
	{
		if (_instance == null)
		{
			try{
				XmldbDatabase database = XmldbDatabase.instance(logger);
				_instance = new XmldbIdentifierManager(database,logger);
			}
			catch (XMLDBException e){
				throw new DatabaseException(e.getMessage());
				//MPA: no proper logging. e is not passed as parameter
			}
		}
		access_count++;
		return _instance;
	}
	
	/**
	 * This class must be closed when it is finished with.
	 */
	public void close()
		throws 
			DatabaseException
	{
		access_count--;
		if (access_count == 0)
		{
			_instance = null;
			database.close();
		}
	}
	
	/**
	 * Create the class, fetches the configuration and initialises
	 * the UID if necessary.
	 * @param logger
	 * @throws DatabaseException
	 */
	protected XmldbIdentifierManager(XmldbDatabase db, Logger logger) 
		throws 
			DatabaseException
	{
		this.logger = logger;
		try {locationUri = new URI(location);}
		catch (URISyntaxException e) {e.printStackTrace();} 
		
		//fetch the configuration and get access to the db
		database = db;
		this.logger = logger;
		//only call at start of day
	}
	
	/**
	 * Creates empty storeage if there is nothing there already, ensures the
	 * return of 0 as the default value.
	 * @throws DatabaseException
	 */
	private void configure()
		throws
			DatabaseException
	{
		this.setArchiveId("00");
		this.setRangeId(0);
		try {
			//Must make a call to initialize to set up the archive id if necessary
			this.initialize();
		} catch (ModuleCriticalException e) {
			throw new DatabaseException(e);
		}
	}
	
	private String fetchAttribute(String name) 
		throws 
			DatabaseException
	{
		if (!database.exists(locationUri))
		{
			configure();
		}
		
		String xml = database.getlatest(locationUri);
		SAXBuilder builder = new SAXBuilder();
		try
		{
			Document doc = builder.build(
				new StringReader(xml));
			Element ids = doc.detachRootElement();
			String value = ids.getAttributeValue(name); 
			return value;
		}
		catch (JDOMException e) {throw new DatabaseException(e);}
		catch (IOException e) {throw new DatabaseException(e);}
	}
	
	private void setAttribute(String name, String value) 
		throws 
			DatabaseException
	{
		String xml = "";
		Element ids = null;
		if (database.exists(locationUri))
		{
			xml = database.getlatest(locationUri);
			SAXBuilder builder = new SAXBuilder();
			try
			{
				Document doc = builder.build(
					new StringReader(xml));
				ids = doc.getRootElement();
			}
			catch (JDOMException e) {throw new DatabaseException(e);}
			catch (IOException e) {throw new DatabaseException(e);}
		}
		else
		{
			ids = new Element("ids");
		}
		ids.setAttribute(name,value);
		XMLOutputter out = new XMLOutputter();
		xml = out.outputString(ids);
		database.putLatest(locationUri,xml);
	}
	
	/**
	 * Return the Archive ID value stored in the Database, if no value is set
	 * return an empty string "" not null.
	 * @return
	 * @throws DatabaseException
	 */
	public String getArchiveId() throws DatabaseException
	{String archId=fetchAttribute("archiveid");
		try {
		return "X"+String.valueOf(Long.parseLong(archId));
		} catch (NumberFormatException e) {
			// new style archiveId starting with a character:
			return archId;
		}
	}
	
	/**
	 * Store the Archive ID in the Database, calling set will overwrite any
	 * previous value. 
	 * @param archiveId
	 * @throws DatabaseException
	 */
	protected void setArchiveId(String archiveId) throws DatabaseException
	{
		setAttribute("archiveid",archiveId);
	}
	
	/**
	 * Return the Range ID value stored in the Database, if no value is set
	 * return and empty string "" not null;
	 * @return
	 * @throws DatabaseException
	 */
	protected long getRangeId() throws DatabaseException
	{
		return Long.parseLong(fetchAttribute("rangeid"));
	}
	
	/**
	 * Store the Range ID in the database, calling set will overwrite any 
	 * previous value.
	 * @param rangeId
	 * @throws DatabaseException
	 */
	protected void setRangeId(long rangeId) throws DatabaseException
	{
		setAttribute("rangeid",String.valueOf(rangeId));
	}
}